try {
    /**
     * CA Wily Introscope(R) Version 10.3.0.15 Build 990015
     * Copyright (c) 2016 CA. All Rights Reserved.
     * Introscope(R) is a registered trademark of CA.
     */
    /**
     * ECMAScript 5 and below do not have Sets or Heaps
     * So, let's do it ourselves
     */
    function BrowserAgentSet() {
        // Use the JS Object to hold the values in the Set
        this._items = {};
        this._size = 0;
    }

    BrowserAgentSet.prototype = {
        /**
         * Adds the given item to the set, if not found already
         * @param val
         */
        add : function ( val ) {
            if ( !this.contains(val) ) {
                this._items[JSON.stringify(val)] = val;
                this._size++;
            }
        },
        /**
         * Determines if the set contains the given item
         * @param val
         * @returns {boolean}
         */
        contains : function ( val ) {
            return typeof this._items[JSON.stringify(val)] !== "undefined";
        },
        /**
         * Tests if the set is empty
         * @returns {*|number}
         */
        isEmpty : function () {
            return this._size === 0;
        },
        /**
         * Removes the given item from the set
         * @param val
         */
        remove : function remove( val ) {
            if ( this.contains(val) ) {
                delete this._items[JSON.stringify(val)];
                this._size--;
            }
        },
        /**
         * Returns the current size of the set
         * @returns {*}
         */
        length : function () {
            return this._size;
        }
    };

    function BrowserAgentMinHeap( comparator ) {
        // Use a JS array to hold the items in the heap
        this._items = [];
        this._size = 0;
        this._getLeftChildIndex = function ( currentIndex ) {
            return (2 * currentIndex) + 1;
        };
        this._getRightChildIndex = function ( currentIndex ) {
            return (2 * currentIndex) + 2;
        };
        this._getParentIndex = function ( childIndex ) {
            return Math.ceil(childIndex / 2) - 1;
        };
        this._compareItems = comparator || function ( val1, val2 ) {
                return val1 === val2 ? 0 : val1 < val2 ? -1 : 1;
            };
        this._heapify = function ( currentIndex ) {
            var leftChildIndex = this._getLeftChildIndex(currentIndex);
            var rightChildIndex = this._getRightChildIndex(currentIndex);
            var smallest = currentIndex;

            if ( this._size > leftChildIndex &&
                 this._compareItems(this._items[leftChildIndex], this._items[smallest]) < 0 ) {
                smallest = leftChildIndex;
            }
            if ( this._size > rightChildIndex &&
                 this._compareItems(this._items[rightChildIndex], this._items[smallest]) < 0 ) {
                smallest = rightChildIndex;
            }
            if ( smallest != currentIndex ) {
                //swap
                var temp = this._items[smallest];
                this._items[smallest] = this._items[currentIndex];
                this._items[currentIndex] = temp;

                //recurse
                this._heapify(smallest);
            }

        };
        this._sift = function ( currentIndex ) {
            var parentIndex = this._getParentIndex(currentIndex);
            if ( parentIndex >= 0 &&
                 this._compareItems(this._items[parentIndex], this._items[currentIndex]) > 0 ) {
                // Swap
                var temp = this._items[parentIndex];
                this._items[parentIndex] = this._items[currentIndex];
                this._items[currentIndex] = temp;
                // Recurse
                this._sift(parentIndex);
            }
        };
    }

    BrowserAgentMinHeap.prototype = {
        isEmpty : function () {
            return this._size === 0;
        },
        getMin : function () {
            if ( this.isEmpty() ) {
                return null;
            }
            return this._items[0];
        },
        length : function () {
            return this._size;
        },
        pop : function () {
            if ( this.isEmpty() ) {
                return null;
            }
            if ( this._size === 1 ) {
                this._size--;
                return this._items.pop();
            }
            var value = this._items[0];
            // Put the bottom element at the top and let it drift down.
            this._items[0] = this._items.pop();
            this._heapify(0);

            this._size--;
            return value;
        },
        push : function ( item ) {
            this._items.push(item);
            this._sift(this._items.length - 1);
            this._size++;
        }
    };

    /**
     * This object is responsible for Browser Agent Browser Logging
     */
    var BrowserAgentLogger = {
        // All BrowserAgent Browser logs precede with [APM BrowserAgent]:
        logPrefix : " [APM BrowserAgent]: ",
        // Currently, the log levels cannot be configured from the Agent. It is merely a placeholder
        // to think about in the next release. So, the browser code is currently logging them under
        // different levels, which is not very useful at this point.
        logLevelPrefix : {
            DEBUG : " [DEBUG] ",
            ERROR : " [ERROR] ",
            INFO : " [INFO] ",
            WARN : " [WARN] "
        },
        /**
         * This function logs the given msg to the browser console as a DEBUG msg
         * @param msg - message that is to be logged
         */
        debug : function ( msg ) {
            if ( BrowserAgentLogger.isOk() ) {
                window.console.log(BrowserAgentUtils.timeUtils.getCurrTime() + BrowserAgentLogger.logPrefix +
                                   BrowserAgentLogger.logLevelPrefix.DEBUG + msg);
            }
        },
        /**
         * This function logs the given msg to the browser console as a ERROR msg
         * @param msg - message that is to be logged
         */
        error : function ( msg ) {
            if ( BrowserAgentLogger.isOk() ) {
                window.console.log(BrowserAgentUtils.timeUtils.getCurrTime() + BrowserAgentLogger.logPrefix +
                                   BrowserAgentLogger.logLevelPrefix.ERROR + msg);
            }
        },
        /**
         * This function logs the given msg to the browser console as a INFO msg
         * @param msg - message that is to be logged
         */
        info : function ( msg ) {
            if ( BrowserAgentLogger.isOk() ) {
                window.console.log(BrowserAgentUtils.timeUtils.getCurrTime() + BrowserAgentLogger.logPrefix +
                                   BrowserAgentLogger.logLevelPrefix.INFO + msg);
            }
        },
        /**
         * This functions makes sure that there is a console to log and the logging is enabled
         * @returns {boolean}
         */
        isOk : function () {
            if ( !window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_BROWSERLOGGINGENABLED]
                 || !window.console || typeof window.console != "object" ) {
                return false
            }
            return true;
        },
        /**
         * This function logs the given msg to the browser console as a WARN msg
         * @param msg
         */
        warn : function ( msg ) {
            if ( BrowserAgentLogger.isOk() ) {
                window.console.log(BrowserAgentUtils.timeUtils.getCurrTime() + BrowserAgentLogger.logPrefix +
                                   BrowserAgentLogger.logLevelPrefix.WARN + msg);
            }
        }
    };
    /**
     * This object provides utility functions
     */
    var BrowserAgentUtils = {
        BrowserAgentTimer : null,
        /**
         * Initializes each Utility
         */
        init : function () {
            BrowserAgentUtils.BrowserAgentTimer = BrowserAgentUtils.timeUtils.getTimerObj();
            BrowserAgentUtils.configUtils.init();
            if ( BrowserAgentUtils.configUtils.isPageExcluded ) {
                BrowserAgentLogger.info("Skipping all instrumentation because page is configured to be EXCLUDED.");
                return;
            }
            BrowserAgentUtils.cookieUtils.init();
            BrowserAgentUtils.browserUtils.init();
            BrowserAgentUtils.storageUtils.init();
            BrowserAgentUtils.funcUtils.init();
            BrowserAgentUtils.metricUtils.init();
        },
        /**
         * Browser Utility
         * Responsible for browser related tasks: Geo-Location
         */
        browserUtils : {
            CorBrowsGUID : null,
            init : function () {
                // Obtain Geo-Location upon page load and store the location inside sessionStorage
                // For the remainder of the session, use the same geo-location co-ordinates
                if ( window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_GEOENABLED] ) {
                    this.getGeoLocation();
                }
            },
            /**
             * Obtain Latitude and Longitude with HTML5 Geo-Location API
             * Note: The Latitude and Longitude will be returned by the Callbacks
             */
            getGeoLocation : function () {
                if ( !navigator || !navigator.geolocation ) {
                    BrowserAgentLogger.warn("getGeoLocation : Geolocation is not supported in this browser.");

                    BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLAT,
                                                                       "-255", true);
                    BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLONG,
                                                                       "-255", true);
                    return;

                }
                var lat = BrowserAgentUtils.storageUtils.getFromSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLAT);
                var lon = BrowserAgentUtils.storageUtils.getFromSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLONG)
                // Compute Geo-location only if BrowserAgentLat or BrowserAgentLong are not present in sessionStorage
                if ( lat === null || lon === null ) {
                    var timeout = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_GEOTIMEOUT];
                    var maxAge = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_GEOMAXIMUMAGE];
                    var accuracy = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_GEOHIGHACCURACYENABLED];
                    var options = {
                        timeout : timeout,
                        maximumAge : maxAge,
                        enableHighAccuracy : accuracy
                    };
                    BrowserAgentLogger.info("getGeoLocation : Attempting to calculate geo location");

                    navigator.geolocation.getCurrentPosition(this.geoLocationFound,
                                                             this.geoLocationNotFound, options);
                    // In some browsers, the error callback is never called upon cancellation of the
                    // location popup (e.g. clicking the X button) and in some browsers, the timeout
                    // option is non-functional. So, do it ourselves.
                    // After timeout and some grace time period, set the sessionStorage geo location to
                    // the same values as those of an error condition
                    setTimeout(function () {
                        if ( BrowserAgentUtils.storageUtils.getFromSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLAT) ===
                             null ) {
                            BrowserAgentLogger.warn("getGeoLocation: never received a response for Geolocation. Setting co-ordinates to -255,-255.");
                            BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLAT,
                                                                               "-255", true);
                            BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLONG,
                                                                               "-255", true);
                        }
                    }, parseInt(timeout) + 5000);
                }
                else {
                    BrowserAgentGlobals.geo.lat = Number(lat);
                    BrowserAgentGlobals.geo.long = Number(lon);
                }
            },
            /**
             * Success Callback for getGeoLocation
             * @param position - HTML5 position object
             */
            geoLocationFound : function ( position ) {
                BrowserAgentGlobals.geo.lat = position.coords.latitude;
                BrowserAgentGlobals.geo.long = position.coords.longitude;

                BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLAT,
                                                                   BrowserAgentGlobals.geo.lat, true);
                BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLONG,
                                                                   BrowserAgentGlobals.geo.long, true);
            },
            /**
             * Error Callback for getGeoLocation
             * @param error - error code from the HTML5 geolocation API
             */
            geoLocationNotFound : function ( error ) {
                switch ( error.code ) {
                    case error.PERMISSION_DENIED:
                        BrowserAgentLogger.warn(
                            "geoLocationNotFound : Browser indicates that user denied the request for Geolocation.");
                        break;
                    case error.POSITION_UNAVAILABLE:
                        BrowserAgentLogger.warn(
                            "geoLocationNotFound : Browser's Geolocation information is unavailable.");
                        break;
                    case error.TIMEOUT:
                        BrowserAgentLogger.warn(
                            "geoLocationNotFound : Browser's request to obtain Geolocation timed out.");
                        break;
                    default:
                        BrowserAgentLogger.warn(
                            "geoLocationNotFound : An unknown error occurred while browser attempted Geolocation.");
                        break;
                }
                BrowserAgentGlobals.geo.lat = -255;
                BrowserAgentGlobals.geo.long = -255;

                BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLAT,
                                                                   BrowserAgentGlobals.geo.lat, true);
                BrowserAgentUtils.storageUtils.putInSessionStorage(BrowserAgentUtils.storageUtils.storageKeys.GEOLONG,
                                                                   BrowserAgentGlobals.geo.long, true);
            }
        },
        /**
         * Config Utility
         * Responsible for Agent <---> Browser Configuration
         */
        configUtils : {
            /**
             * Known configuration parameter MACROS
             */
            configNames : {
                // Toggles AJAX metrics
                BROWSERAGENT_AJAXMETRICSENABLED : "BROWSERAGENT_AJAXMETRICSENABLED",
                // Threshold to control AJAX metrics
                // Note: AJAX calls for which the Total Resource Load Time < threshold,
                //       AJAX metrics are ignored
                BROWSERAGENT_AJAXMETRICSTHRESHOLD : "BROWSERAGENT_AJAXMETRICSTHRESHOLD",
                // Toggles Browser Logging
                BROWSERAGENT_BROWSERLOGGINGENABLED : "BROWSERAGENT_BROWSERLOGGINGENABLED",
                // Toggles BROWSERAGENT
                BROWSERAGENT_ENABLED : "BROWSERAGENT_ENABLED",
                // List of URL paths for which the BROWSERAGENT monitoring is to be ignored
                // Exact Match only, No regex matching
                BROWSERAGENT_EXCLUDELIST : "BROWSERAGENT_EXCLUDELIST",
                // Toggles Geo-Location
                BROWSERAGENT_GEOENABLED : "BROWSERAGENT_GEOENABLED",
                // Toggles Geo-Location High Accuracy Mode
                BROWSERAGENT_GEOHIGHACCURACYENABLED : "BROWSERAGENT_GEOHIGHACCURACYENABLED",
                // Specifies the interval for which the previous Geo-Location values are to be used
                BROWSERAGENT_GEOMAXIMUMAGE : "BROWSERAGENT_GEOMAXIMUMAGE",
                // Specifies the interval to wait for the current Geo-Location calculation
                BROWSERAGENT_GEOTIMEOUT : "BROWSERAGENT_GEOTIMEOUT",
                // List of URL paths for which the BROWSERAGENT monitoring is to be used
                // Exact Match only, No regex matching
                BROWSERAGENT_INCLUDELIST : "BROWSERAGENT_INCLUDELIST",
                // Toggles JS function metrics
                BROWSERAGENT_JSFUNCTIONMETRICSENABLED : "BROWSERAGENT_JSFUNCTIONMETRICSENABLED",
                // Note: JS function calls for which the Average Execution Time < threshold,
                //       JS function metrics are ignored
                BROWSERAGENT_JSFUNCTIONMETRICSTHRESHOLD : "BROWSERAGENT_JSFUNCTIONMETRICSTHRESHOLD",
                // Specifies the frequency of BROWSERAGENT metric dispatch from the browser
                BROWSERAGENT_METRICFREQUENCY : "BROWSERAGENT_METRICFREQUENCY",
                // Toggles page load metrics
                BROWSERAGENT_PAGELOADMETRICSENABLED : "BROWSERAGENT_PAGELOADMETRICSENABLED",
                // Threshold to control page load metrics
                // Note: Pages for which the Average Page Load Complete Time < threshold,
                //       page load metrics are ignored
                BROWSERAGENT_PAGELOADMETRICSTHRESHOLD : "BROWSERAGENT_PAGELOADMETRICSTHRESHOLD",
                // Toggles metrics based on URL context
                BROWSERAGENT_URLMETRICOFF : "BROWSERAGENT_URLMETRICOFF",
                // URL to which the metrics are to be dispatched
                BROWSERAGENT_WILYURL : "BROWSERAGENT_WILYURL"
            },
            /**
             * Default values for the known configuration parameters
             */
            defaults : {
                BROWSERAGENT_AJAXMETRICSENABLED : true,
                BROWSERAGENT_AJAXMETRICSTHRESHOLD : 10,
                BROWSERAGENT_BROWSERLOGGINGENABLED : false,

                BROWSERAGENT_ENABLED : true,
                BROWSERAGENT_EXCLUDELIST : [],
                BROWSERAGENT_GEOENABLED : false,
                BROWSERAGENT_GEOHIGHACCURACYENABLED : false,
                BROWSERAGENT_GEOMAXIMUMAGE : 10000,
                BROWSERAGENT_GEOTIMEOUT : 5000,
                BROWSERAGENT_INCLUDELIST : [],
                BROWSERAGENT_JSFUNCTIONMETRICSENABLED : false,
                BROWSERAGENT_JSFUNCTIONMETRICSTHRESHOLD : 10,
                BROWSERAGENT_METRICFREQUENCY : 0,
                BROWSERAGENT_PAGELOADMETRICSENABLED : true,
                BROWSERAGENT_PAGELOADMETRICSTHRESHOLD : 10,
                BROWSERAGENT_URLMETRICOFF : false,
                BROWSERAGENT_WILYURL : window.location.protocol + "//" + window.location.host +
                                       window.location.pathname + "?WilyCmd=cmdMetrics"
            },
            init : function () {
                // Initialize window configs
                for ( var configName in BrowserAgentUtils.configUtils.configNames ) {
                    window[configName] = BrowserAgentUtils.configUtils.getConfig(configName);
                }
                BrowserAgentUtils.configUtils.isPageExcluded =
                    BrowserAgentUtils.configUtils.isUrlExcluded(BrowserAgentGlobals.currentFullURL);
            },
            isPageExcluded : false,
            /**
             * Obtains the current configuration given a valid configuration parameter
             * @param configParam - String from the configUtils.configNames
             * @returns {*} - configuration value if found; else, null
             */
            getConfig : function ( configParam ) {
                if ( !configParam || typeof configParam != 'string' || !this.configNames[configParam] ) {
                    BrowserAgentLogger.warn("getConfig : Could not obtain configuration for invalid parameter [ " +
                                            configParam + " ]");
                    return null;
                }
                if ( window[configParam] != null && window[configParam] != undefined ) {
                    return window[configParam];
                }
                return BrowserAgentUtils.configUtils.defaults[configParam];
            },
            /**
             * Get the full url of a given url including protocol, host, port, pathname and query parameters
             * @param url - url string that can be a relative path or a full url
             * @returns full url including protocol, host, port, pathname and query parameters
             */
            getFullURL : function ( url ) {
                if ( !url || typeof url !== 'string' ) {
                    BrowserAgentLogger.warn("getFullURL : Not a valid URL. Skipping parse...");
                    return;
                }
                // Note: there is no need to remove the element as the element is removed
                // outside the scope of this method as it has no parent
                var parser = document.createElement('a');
                // Let the browser do the work
                parser.href = url;
                return parser.href;
            },
            /**
             * Determines if the given full URL is to be ignored for BrowserAgent by comparing to the
             * ExcludeURLList and IncludeURLList
             * @param url - url path with no trailing slashes
             * @returns boolean : true if the given URL is to be excluded; false otherwise
             */
            isUrlExcluded : function ( url ) {
                if ( !url || typeof url !== 'string' ) {
                    BrowserAgentLogger.warn("isUrlExcluded: Invalid URL. Skipping URL exclusion check...");
                    return false;
                }
                var includeUrlList = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_INCLUDELIST];
                var excludeUrlList = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_EXCLUDELIST];
                if ( !includeUrlList || !excludeUrlList ) {
                    BrowserAgentLogger.warn("isUrlExcluded: Invalid Include/Exclude URL List. Skipping url exclusion check...");
                    return false;
                }
                if ( (excludeUrlList.length > 0 &&
                      BrowserAgentUtils.configUtils.isUrlInRegexList(url, excludeUrlList)) ||
                     (includeUrlList.length > 0 &&
                      !BrowserAgentUtils.configUtils.isUrlInRegexList(url, includeUrlList)) ) {
                    return true;
                }
                return false;
            },
            /**
             * Determines if an url matches a pattern in the regex list
             * @param url - to be tested
             * @param list - regex list
             * @returns {boolean}
             */
            isUrlInRegexList : function ( url, list ) {
                for ( var i = 0; i < list.length; i++ ) {
                    if ( (new RegExp(list[i])).test(url) ) {
                        return true;
                    }
                }
                return false;
            }
        },
        /**
         * Storage Utility
         * Responsible for managing browser localStorage and sessionStorage
         */
        storageUtils : {
            storageKeys : {
                GEOLAT : "BrowserAgentLat",
                GEOLONG : "BrowserAgentLong"
            },
            init : function () {
                //empty
            },
            /**
             * Stores an item in Session Storage
             * @param key
             * @param itemToStore
             * @param isIdempotent [when 'false', item will be stored only if it is not already present]
             */
            putInSessionStorage : function ( key, itemToStore, isIdempotent ) {
                if ( typeof key !== 'string' || itemToStore == null || itemToStore == undefined ||
                     typeof isIdempotent !== 'boolean' || !window.Storage ) {
                    BrowserAgentLogger.warn("putInSessionStorage: cannot access sessionStorage");
                    return;
                }
                else {
                    if ( isIdempotent || (!isIdempotent && sessionStorage.getItem(key) == null) ) {
                        sessionStorage.setItem(key, itemToStore);
                    }
                }
            },
            /**
             * Obtains an item from Session Storage
             * @param key
             * @returns {null}
             */
            getFromSessionStorage : function ( key ) {
                if ( typeof key !== 'string' || !window.Storage ) {
                    BrowserAgentLogger.warn("getFromSessionStorage: cannot access sessionStorage")
                    return null;
                }
                return sessionStorage.getItem(key);
            },
            /**
             * Removes an item from Session Storage
             * @param key
             */
            removeFromSessionStorage : function ( key ) {
                if ( typeof key !== 'string' || !window.Storage ) {
                    BrowserAgentLogger.warn("removeFromSessionStorage: cannot access sessionStorage")
                    return;
                }
                // Note: removeItem is idempotent
                sessionStorage.removeItem(key);
            }
        },
        /**
         * Cookie Utility
         * Responsible for managing BrowserAgent cookies
         */
        cookieUtils : {
            /**
             * Cookies used between browser and the Agent
             */
            cookies : {
                // Response Cookie to Agent
                // Stores business txn information as sent by the agent
                BTRESP : "x-apm-brtm-response-bt",
                // Request Cookie to Agent
                // Stores an unique identifier for an instrumented AJAX call
                BTRESPID : "x-apm-brtm-response-bt-id",
                // Response Cookie from Agent
                // Stores business txn information for the current page as sent by the agent
                BTPAGERESP : "x-apm-brtm-response-bt-page",
                // Response Cookie from Agent
                // Stores server time (ms) when the response was sent. Used to calculate client server gap time
                SERVERTIME : "x-apm-brtm-servertime",
                // Request Cookie to Agent
                // Stores client server gap time (s)
                GAPTIME : "x-apm-brtm-gaptime",
                // Request Cookie to Agent
                // Stores page load times
                // Note: Inherited from old BrowserAgent
                GLOBAL : "WMRUMC",
                // Request Cookie to Agent
                // Stores Browser Name
                PLATFORM : "x-apm-brtm-bt-p",
                // Request cookie to agent
                // Stores browser major version
                PLATFORMVER : "x-apm-brtm-bt-pv"
            },
            /**
             * Keys inside BTRESP, BTPAGERESP as defined above
             */
            cookieKeys : {
                // Introscope txn trace start time
                apmStartTimeChar : "startTime",
                // Business segment
                bsChar : "bs",
                // Business txn
                btChar : "bt",
                // Business txn component
                btcChar : "btc",
                // Txn Trace Correlation ID
                CorBrowsGUIDChar : "CorBrowsGUID",
                // Geo-location
                geoChar : "g",
                // Browser name
                platformChar : "p",
                // Browser major version
                platformVerChar : "pv",
                // Txn trace flag
                ttChar : "c"
            },
            document : document,
            init : function () {
                // Update BS, BT context for the current page from the response cookies
                var pageBTCookieName = BrowserAgentUtils.cookieUtils.cookies.BTPAGERESP + "-" +
                                       encodeURIComponent(window.location.pathname);
                var responseCookies = BrowserAgentUtils.cookieUtils.getRawCookie(pageBTCookieName);
                BrowserAgentUtils.cookieUtils.deleteCookie(pageBTCookieName, "/", null);
                if ( responseCookies ) {
                    var map = BrowserAgentUtils.cookieUtils.tokenizeCookieIntoMap(responseCookies, ',');
                    BrowserAgentUtils.cookieUtils.updateObjWithCookieDataPage(map);
                } else {
                    BrowserAgentLogger.warn("Cannot get page bt cookie for url = " + window.location.pathname);
                }
                // Pull gap times from snippet and delete server page time cookie
                if ( BrowserAgentSnippet && BrowserAgentSnippet.gapTimeInMillis ) {
                    BrowserAgentGlobals.gapTimeInMillis = BrowserAgentSnippet.gapTimeInMillis;
                } else {
                    BrowserAgentLogger.warn("Snippet not available. Setting gap time to 0");
                    BrowserAgentGlobals.gapTimeInMillis = 0;
                }
                BrowserAgentUtils.cookieUtils.deleteCookie(BrowserAgentUtils.cookieUtils.cookies.SERVERTIME, "/", null);

                // Get browser platform and version info
                BrowserAgentGlobals.p =
                    BrowserAgentUtils.cookieUtils.getRawCookie(BrowserAgentUtils.cookieUtils.cookies.PLATFORM);
                BrowserAgentGlobals.pv =
                    BrowserAgentUtils.cookieUtils.getRawCookie(BrowserAgentUtils.cookieUtils.cookies.PLATFORMVER);
            },
            /**
             * Delete a cookie given its name
             * Note: The path and domain must match that of the cookie at the time it was created
             * @param name - name of the cookie to be deleted
             * @param path - path of the cookie at the time it was created
             * @param domain - domain of the cookie at the time it was created
             */
            deleteCookie : function ( name, path, domain ) {
                if ( !name ) {
                    BrowserAgentLogger.warn("deleteCookie : Cannot delete cookie by name " + name);
                    return;
                }
                this.document.cookie = name + "=" + "; expires=Thu, 01-Jan-1970 00:00:01 GMT" +
                                       ((domain ) ? "; domain=" + domain : "" ) +
                                       ((path ) ? "; path=" + path : "");
            },
            /**
             * Obtain cookie given a name
             * @param name - name of cookie to get
             * @returns {string}
             */
            getCookie : function ( name ) {
                var cName = "";
                if ( !name ) {
                    BrowserAgentLogger.warn("getCookie : Cannot obtain cookie - " + name);
                    return cName;
                }
                var pCookies = document.cookie.split('; ');
                for ( var bb = 0; bb < pCookies.length; bb++ ) {
                    var NmeVal = pCookies[bb].split('=');
                    if ( NmeVal[0] == name ) {
                        cName = NmeVal[1];
                    }
                }
                return decodeURIComponent(cName);
            },
            /**
             * Obtain the cookie object, given a name
             * @param cookie  - name of the cookie object
             * @returns {*}
             */
            getCookieObject : function ( cookie ) {
                if ( cookie ) {
                    return JSON.parse(this.getRawCookie(cookie));
                }
                return JSON.parse(this.getRawCookie(this.cookies.GLOBAL));
            },
            /**
             * Given a name, obtain the corresponding cookie value
             * @param name - name of the cookie
             * @returns {*}
             */
            getRawCookie : function ( name ) {
                if ( !name ) {
                    BrowserAgentLogger.warn("getRawCookie : Cannot obtain cookie " + name);
                    return null;
                }
                if ( this.document.cookie.length > 0 ) {
                    var cs = this.document.cookie.indexOf(name + "=");
                    if ( cs !== -1 ) {
                        cs = cs + name.length + 1;
                        var ce = this.document.cookie.indexOf(";", cs);
                        if ( ce === -1 ) {
                            ce = this.document.cookie.length;
                        }
                        // Java Agent URLEncoder encodes space to "+" which is different from JavaScript
                        // encodeURIComponent. We replace "+" with "%20" here so JavaScript can decode correctly.
                        return decodeURIComponent(
                            BrowserAgentUtils.metricUtils.replaceAll(this.document.cookie.substring(cs, ce), "\\+",
                                                                     "%20"));
                    } else {
                        return null;
                    }
                } else {
                    return null;
                }
            },
            /**
             * Tokenize a given cookie value into a JS map
             * @param str - value of a cookie as a String
             * @param delimiter - token to split the cookie value by
             * @returns {{}}
             */
            tokenizeCookieIntoMap : function ( str, delimiter ) {
                var map = {};
                if ( !str || !delimiter ) {
                    BrowserAgentLogger.warn("tokenizeCookieIntoMap : Cannot parse " + str + " by " +
                                            delimiter);
                    return map;
                }
                str = decodeURIComponent(str).replace(/["]/g, "");
                var lines = str.split(delimiter);
                for ( var i = 0; i < lines.length; i++ ) {
                    var pieces = lines[i].split("=");
                    if ( pieces.length === 2 ) {
                        map[pieces[0]] = pieces[1];
                    }
                }
                return map;
            },
            /**
             * Set a cookie into document.cookie
             * @param name - name of the cookie to be created
             * @param value - value of the cookie to be stored
             * @param expiry - the number of seconds this cookie is to
             *                 be active from the time of creation
             * @param path - path for the cookie (e.g. /)
             * @param domain - domain for the cookie
             */
            setRawCookie : function ( name, value, expiry, path, domain ) {
                if ( !name ) {
                    BrowserAgentLogger.warn("setRawCookie : Cannot set cookie with name " + name);
                    return;
                }
                var et = new Date(BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch() +
                                  (expiry * 1000));
                this.document.cookie = name + "=" + encodeURIComponent(value) +
                                       ((expiry) ? "; expires=" + et.toUTCString() : "" ) +
                                       ((domain ) ? "; domain=" + domain : "" ) +
                                       ((path ) ? "; path=" + path : "");
            },
            /**
             * Update an existing cookie in document.cookie
             * Note: the cookie value must be a JS object
             * @param cookieName - name of the cookie that is to be updated
             * @param value - value to update with
             */
            updateCookie : function ( cookieName, value ) {
                if ( !cookieName ) {
                    BrowserAgentLogger.warn("updateCookie : cannot update cookie with name " + cookieName);
                    return;
                }
                var cookieObject = JSON.parse(this.getRawCookie(cookieName));
                var newCookieObject = {};
                for ( var i in cookieObject ) {
                    newCookieObject[i] = cookieObject[i];
                }
                if ( typeof value == 'object' ) {
                    for ( var j in value ) {
                        if ( value[j] != null ) {
                            newCookieObject[j] = value[j];
                        } else {
                            delete newCookieObject[j];
                        }
                    }
                } else {
                    newCookieObject = value;
                }
                this.setRawCookie(cookieName,
                                  JSON.stringify(newCookieObject), null,
                                  "/", null);
            },
            /**
             * Update a given object holding AJAX data with AJAX data from the cookie
             * @param cookieData - cookie data containing tokens from cookieUtils.cookieKeys
             * @param objToUpdate - a JS object containing AJAX related data
             */
            updateObjWithCookieDataAjax : function ( cookieData, objToUpdate ) {
                if ( !cookieData || !objToUpdate || typeof cookieData !== 'object'
                     || typeof objToUpdate !== 'object' ) {
                    BrowserAgentLogger.warn("updateObjWithCookieDataAjax : cannot update object with data from cookie");
                    return;
                }
                var key;
                var decodedCookieVal;
                for ( var item in BrowserAgentUtils.cookieUtils.cookieKeys ) {
                    key = BrowserAgentUtils.cookieUtils.cookieKeys[item];
                    if ( cookieData[key] ) {
                        decodedCookieVal = decodeURIComponent(cookieData[key]);
                        switch ( key ) {
                            case BrowserAgentUtils.cookieUtils.cookieKeys.CorBrowsGUIDChar :
                                objToUpdate[key] = decodedCookieVal;
                                break;
                            case BrowserAgentUtils.cookieUtils.cookieKeys.bsChar :
                                objToUpdate[key] =
                                    (decodedCookieVal === BrowserAgentGlobals.UNDEFINED ) ?
                                    BrowserAgentGlobals[key] : decodedCookieVal;
                                break;
                            case BrowserAgentUtils.cookieUtils.cookieKeys.btChar :
                                objToUpdate[key] =
                                    (decodedCookieVal === BrowserAgentGlobals.UNDEFINED ) ?
                                    BrowserAgentGlobals[key] : decodedCookieVal;
                                break;
                            case BrowserAgentUtils.cookieUtils.cookieKeys.btcChar :
                                objToUpdate[key] =
                                    (decodedCookieVal === BrowserAgentGlobals.UNDEFINED ) ?
                                    BrowserAgentGlobals[key] : decodedCookieVal;
                                break;
                            default :
                                objToUpdate[key] = decodedCookieVal;
                                break;
                        }
                    }
                }
            },
            /**
             * Update a given object holding page data with page data from the cookie
             * @param cookieData - cookie data containing tokens from cookieUtils.cookiekeys
             */
            updateObjWithCookieDataPage : function ( cookieData ) {
                if ( !cookieData || typeof cookieData !== 'object' ) {
                    BrowserAgentLogger.warn("updateObjWithCookieDataPage : cannot update object with data from cookie");
                    return;
                }
                var key;
                for ( var item in BrowserAgentUtils.cookieUtils.cookieKeys ) {
                    key = BrowserAgentUtils.cookieUtils.cookieKeys[item];
                    if ( cookieData[key] ) {
                        BrowserAgentGlobals[key] = decodeURIComponent(cookieData[key]);
                    }
                }
            }
        },
        /**
         * Function Utility
         * Responsible for JS Function instrumentation
         */
        funcUtils : {
            init : function () {
                try {
                    // Add the XHR functions to be instrumented
                    this.addFuncToGlobalsInstrumentMap("XHR_Open",
                        { name : "XMLHttpRequest.prototype.open" });
                    this.addFuncToGlobalsInstrumentMap("XHR_Send",
                        { name : "XMLHttpRequest.prototype.send" });
                    // EXTENSION POINT to add JS functions in the window scope to be instrumented
                    if ( BrowserAgentExtension ) {
                        BrowserAgentExtension.extAddJSFuncToInstrument();
                    }
                    // Instrument the JS functions that were added above
                    this.instrumentAllFunc();
                } catch ( e ) {
                    BrowserAgentLogger.error("funcUtils.init : " + e.message);
                }
            },
            /**
             * Adds a JS Function with the given key into BrowserAgentGlobals.functionsToInstrument
             * @param key - an unique string
             * @param obj (e.g. {name : "XMLHttpRequest.prototype.open"}
             * Note: Add 'prototype' keyword for all JS member functions
             */
            addFuncToGlobalsInstrumentMap : function ( key, obj ) {
                if ( !key || typeof key !== 'string' || !obj || typeof obj !== 'object' ) {
                    BrowserAgentLogger.warn(
                        "addFuncToGlobalsInstrumentMap: could not add JS Function because key or value is not of the correct type");
                    return;
                }
                // Check for mandatory fields inside the given function object
                if ( !obj.name || typeof obj.name !== 'string' ) {
                    BrowserAgentLogger.warn(
                        "addFuncToGlobalsInstrumentMap: value does not contain the mandatory field 'name'");
                    return;
                }
                // Add Function
                // NOTE: this will not work in IE8 and below
                Object.defineProperty(BrowserAgentGlobals.functionsToInstrument, key, {
                    value : obj,
                    writable : true,
                    enumerable : true,
                    configurable : true
                });
            },
            /**
             * Trace function that succeeds the monitored application source
             * @returns {*}
             */
            endTrace : function () {
                return BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
            },
            /**
             * Instruments a given JS Function by redefining it with BrowserAgent logic.
             * @param funcInString - name of the function to instrument as a string
             * @param key - the unique string that was used to add this function with
             *              addFuncToGlobalsInstrumentMap
             * @param maxRetryCount - Try to leave this alone as the retry logic depends on it
             * @returns function
             */
            instrumentFunc : function ( funcInString, key, maxRetryCount ) {
                if ( !BrowserAgentGlobals.functionsToInstrument ) {
                    BrowserAgentLogger.info("Nothing to instrument.");
                    return null;
                }
                if ( !funcInString || typeof funcInString !== 'string' || !key ) {
                    BrowserAgentLogger.warn(
                        "instrumentFunc: Function name was not specified. Skipping instrumentation...");
                    return null;
                }
                try {
                    eval("var origObjFromWindow = (window." + funcInString + ") ? window." +
                         funcInString + " : null;");
                } catch ( e ) {
                    BrowserAgentLogger.warn(
                        "instrumentFunc: " + funcInString + " could not be found in the browser window scope.");
                    this.retryInstrumentFunc(funcInString, key, maxRetryCount);
                    return null;
                }
                // Needless to say, if the original function is not in scope, there is nothing to do
                // So, Skip the instrumentation
                if ( !origObjFromWindow ) {
                    BrowserAgentLogger.warn(
                        "instrumentFunc: " + funcInString + " is null or undefined in the browser window scope.");
                    this.retryInstrumentFunc(funcInString, key, maxRetryCount);
                    return null;
                }
                // If the function is already instrumented, don't bother to instrument it
                if ( origObjFromWindow._isBrowserAgentInstrumented ) {
                    BrowserAgentLogger.info(funcInString +
                                            " already instrumented. Skipping instrumentation...");
                    return null;
                }
                this.saveOrigObj(key, origObjFromWindow);
                // If the function was not saved in the original function map, then don't
                // proceed with instrumentation
                if ( !BrowserAgentGlobals.origFuncMap || !BrowserAgentGlobals.origFuncMap[key] ) {
                    BrowserAgentLogger.warn("instrumentFunc: JS Function [" + funcInString +
                                            "] could not be saved. Skipping instrumentation...");
                    return null;
                }
                var origFunc = origObjFromWindow;
                // Track any errors during instrumentation
                var isError = false;
                BrowserAgentLogger.info("Instrumenting " + funcInString);
                var redefinedFunc = null;
                switch ( funcInString ) {
                    case "XMLHttpRequest.prototype.send":
                        // Generate a new XMLHttpRequest.prototype.send function
                        redefinedFunc = function () {
                            var ajaxFullURL = this._fullURL;
                            var isAsync = this._async;
                            var isFuncEnabled = false;
                            var isAjaxInstrumented = this._isAjaxInstrumented;
                            var funcMetricData = {};
                            // Wrap the BrowserAgent instrumentation in a try, catch...
                            try {
                                isFuncEnabled =
                                    window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_JSFUNCTIONMETRICSENABLED];
                            } catch ( e ) {
                                isError = true;
                                BrowserAgentLogger.error("instrumentFunc (Checkpoint # 1): " + e.message);
                            }
                            // If the XHR is used for synchronous purposes, then don't bother
                            // to trace the callbacks.
                            if ( !isError && isAsync && isAjaxInstrumented ) {
                                var ajaxMetricData = {};
                                // This uniqueID serves as the identifier between instances of
                                // the same resource being loaded many times within an interval
                                // via Ajax.
                                var uniqueID = '_' + Math.random().toString(36).substr(2, 9);
                                BrowserAgentUtils.cookieUtils.setRawCookie(
                                    BrowserAgentUtils.cookieUtils.cookies.BTRESPID,
                                    uniqueID, 2, "/", null);
                                // Redefine XHR onload
                                var origOnload = this.onload;
                                if ( origOnload ) {
                                    // Assign a new function to onload
                                    this.onload = function () {
                                        // Wrap the BrowserAgent instrumentation in a try, catch...
                                        try {
                                            if ( !isError ) {
                                                ajaxMetricData[BrowserAgentGlobals.timestampNames.CALLBACK_START_TIME] =
                                                    BrowserAgentUtils.funcUtils.startTrace();
                                            }
                                        } catch ( e ) {
                                            isError = true;
                                            BrowserAgentLogger.error("instrumentFunc (Checkpoint # 2): " +
                                                                     e.message);
                                        }

                                        ///////////// Start of ORIGINAL ONLOAD ////////////
                                        origOnload.apply(this, arguments);
                                        ///////////// End of ORIGINAL ONLOAD /////////////

                                        // Wrap the BrowserAgent instrumentation in a try, catch...
                                        try {
                                            if ( !isError ) {
                                                ajaxMetricData[BrowserAgentGlobals.timestampNames.CALLBACK_END_TIME] =
                                                    BrowserAgentUtils.funcUtils.endTrace();
                                                ajaxMetricData["URL"] = ajaxFullURL;
                                                // Add BT information from the response Cookie
                                                var responseCookie = BrowserAgentUtils.cookieUtils.getRawCookie(
                                                    BrowserAgentUtils.cookieUtils.cookies.BTRESP +
                                                    "-" +
                                                    uniqueID);
                                                if ( responseCookie ) {
                                                    var cookieData = BrowserAgentUtils.cookieUtils.tokenizeCookieIntoMap(
                                                        responseCookie,
                                                        ",");
                                                    BrowserAgentUtils.cookieUtils.updateObjWithCookieDataAjax(
                                                        cookieData,
                                                        ajaxMetricData);

                                                }
                                                // No longer need the x-apm-brtm-response-bt-_uniqueID
                                                // cookie. So, delete it.
                                                BrowserAgentUtils.cookieUtils.deleteCookie(
                                                    BrowserAgentUtils.cookieUtils.cookies.BTRESP +
                                                    "-" +
                                                    uniqueID, "/",
                                                    null);

                                                BrowserAgentUtils.metricUtils.addDataPoint(
                                                    BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX],
                                                    uniqueID,
                                                    ajaxMetricData);
                                                BrowserAgentUtils.metricUtils.harvestMetrics(
                                                    BrowserAgentGlobals.metricType.AJAX,
                                                    uniqueID);
                                            }
                                        } catch ( e ) {
                                            isError = true;
                                            BrowserAgentLogger.error("instrumentFunc (Checkpoint # 3): " +
                                                                     e.message);
                                        }
                                    };
                                }
                                var origOsrc = this.onreadystatechange;
                                // Assign a new function to onreadystatechange
                                this.onreadystatechange = function () {
                                    var startTime;
                                    // Wrap the BrowserAgent instrumentation in a try, catch...
                                    try {
                                        if ( !isError ) {
                                            startTime = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
                                            if ( this.readyState === this.LOADING ) {
                                                // time it on first invocation, not all
                                                if ( !ajaxMetricData[BrowserAgentGlobals.timestampNames.FIRST_BYTE] ) {
                                                    ajaxMetricData[BrowserAgentGlobals.timestampNames.FIRST_BYTE] =
                                                        startTime;
                                                }
                                            }
                                            if ( this.readyState === this.DONE ) {
                                                ajaxMetricData[BrowserAgentGlobals.timestampNames.LAST_BYTE] =
                                                    startTime;
                                            }
                                        }
                                    } catch ( e ) {
                                        isError = true;
                                        BrowserAgentLogger.error("instrumentFunc (Checkpoint # 4): " +
                                                                 e.message);
                                    }
                                    if ( origOsrc ) {
                                        // Wrap the BrowserAgent instrumentation in a try, catch...
                                        try {
                                            if ( !isError && this.readyState === this.DONE ) {
                                                ajaxMetricData[BrowserAgentGlobals.timestampNames.CALLBACK_START_TIME] =
                                                    BrowserAgentUtils.funcUtils.startTrace();
                                            }
                                        } catch ( e ) {
                                            isError = true;
                                            BrowserAgentLogger.error("instrumentFunc (Checkpoint # 5): " +
                                                                     e.message);
                                        }

                                        ///////////// Start of ORIGINAL ONREADYSTATECHANGE ////////////
                                        origOsrc.apply(this, arguments);
                                        ///////////// End of ORIGINAL ONREADYSTATECHANGE /////////////

                                        // Wrap the BrowserAgent instrumentation in a try, catch...
                                        try {
                                            if ( !isError && this.readyState === this.DONE ) {
                                                ajaxMetricData[BrowserAgentGlobals.timestampNames.CALLBACK_END_TIME] =
                                                    BrowserAgentUtils.funcUtils.endTrace();
                                                // Add BT information from the response Cookie
                                                var responseCookie = BrowserAgentUtils.cookieUtils.getRawCookie(
                                                    BrowserAgentUtils.cookieUtils.cookies.BTRESP +
                                                    "-" +
                                                    uniqueID);
                                                if ( responseCookie ) {
                                                    var cookieData = BrowserAgentUtils.cookieUtils.tokenizeCookieIntoMap(
                                                        responseCookie,
                                                        ",");
                                                    BrowserAgentUtils.cookieUtils.updateObjWithCookieDataAjax(
                                                        cookieData,
                                                        ajaxMetricData);
                                                }
                                                // No longer need the x-apm-brtm-response-bt-_uniqueID
                                                // cookie. So, delete it.
                                                BrowserAgentUtils.cookieUtils.deleteCookie(
                                                    BrowserAgentUtils.cookieUtils.cookies.BTRESP +
                                                    "-" +
                                                    uniqueID, "/",
                                                    null);
                                                ajaxMetricData["URL"] = ajaxFullURL;
                                                BrowserAgentUtils.metricUtils.addDataPoint(
                                                    BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX],
                                                    uniqueID,
                                                    ajaxMetricData);
                                                BrowserAgentUtils.metricUtils.harvestMetrics(
                                                    BrowserAgentGlobals.metricType.AJAX,
                                                    uniqueID);
                                            }
                                        } catch ( e ) {
                                            isError = true;
                                            BrowserAgentLogger.error("instrumentFunc (Checkpoint # 6): " +
                                                                     e.message);
                                        }
                                    }
                                };
                            }
                            // Wrap the BrowserAgent instrumentation in a try, catch...
                            try {
                                if ( !isError && isFuncEnabled ) {
                                    funcMetricData[BrowserAgentGlobals.timestampNames.START_TIME] =
                                        BrowserAgentUtils.funcUtils.startTrace();
                                }
                            } catch ( e ) {
                                isError = true;
                                BrowserAgentLogger.error("instrumentFunc (Checkpoint # 7) : " + e.message);
                            }

                            //////////////// Start of ORIGINAL JS Function ////////////////
                            var funcRet = origFunc.apply(this, arguments);
                            //////////////// End of ORIGINAL JS Function ////////////////

                            // Wrap the rest of the BrowserAgent instrumentation in a try, catch...
                            try {
                                if ( !isError && isFuncEnabled ) {
                                    funcMetricData[BrowserAgentGlobals.timestampNames.END_TIME] =
                                        BrowserAgentUtils.funcUtils.endTrace();
                                    BrowserAgentUtils.metricUtils.addDataPoint(
                                        BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION],
                                        key, funcMetricData);
                                    BrowserAgentUtils.metricUtils.harvestMetrics(
                                        BrowserAgentGlobals.metricType.FUNCTION);
                                }
                                // If XHR is used for synchronous purposes, don't bother to record
                                // Ajax data point
                                if ( !isError && isAsync && isAjaxInstrumented ) {
                                    var ajaxData = {};
                                    ajaxData[BrowserAgentGlobals.timestampNames.END_TIME] =
                                        BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
                                    BrowserAgentUtils.metricUtils.addDataPoint(BrowserAgentGlobals.ajaxSendDataMap,
                                                                               uniqueID, ajaxData);
                                }
                            } catch ( e ) {
                                isError = true;
                                BrowserAgentLogger.error("instrumentFunc (Checkpoint # 8): " + e.message);
                            }
                            return funcRet;
                        };
                        break;
                    default:
                        // Generate a new function
                        redefinedFunc = function () {
                            var isFuncEnabled = false;
                            var funcMetricData = {};
                            // Wrap the BrowserAgent instrumentation in a try, catch...
                            try {
                                if ( !isError ) {
                                    isFuncEnabled =
                                        window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_JSFUNCTIONMETRICSENABLED];
                                    if ( !isFuncEnabled ) {
                                        BrowserAgentLogger.info("JS Function Metrics are DISABLED.");
                                    }
                                    // If the function is XMLHttpRequest.prototype.open, then save
                                    // its arguments to be used later
                                    if ( funcInString === "XMLHttpRequest.prototype.open" ) {
                                        // In the XHR open call, the resource URL is a required argument
                                        // Store the URL in the XHR object from which the open originates
                                        // so that it can be used later to correlate metrics
                                        this._url = arguments[1];
                                        this._fullURL = BrowserAgentUtils.configUtils.getFullURL(this._url);
                                        this._async = true;
                                        this._isAjaxInstrumented = true;
                                        if ( arguments.length >= 3 ) {
                                            this._async = arguments[2];
                                        }
                                        var isAjaxEnabled = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_AJAXMETRICSENABLED];
                                        var isAjaxExcluded = BrowserAgentUtils.configUtils.isUrlExcluded(this._fullURL);
                                        if ( !isAjaxEnabled ) {
                                            BrowserAgentLogger.info("AJAX Metrics are DISABLED.");
                                        }
                                        if ( isAjaxExcluded ) {
                                            BrowserAgentLogger.info("AJAX URL " + this._fullURL +
                                                                    " is configured to be EXCLUDED.");
                                        }
                                        if ( !isAjaxEnabled || isAjaxExcluded ) {
                                            this._isAjaxInstrumented = false;
                                        }
                                    }
                                    if ( isFuncEnabled ) {
                                        funcMetricData[BrowserAgentGlobals.timestampNames.START_TIME] =
                                            BrowserAgentUtils.funcUtils.startTrace();
                                    }
                                }
                            } catch ( e ) {
                                isError = true;
                                BrowserAgentLogger.error("instrumentFunc (Checkpoint # 9) : " + e.message);
                            }

                            //////////////// Start of ORIGINAL JS Function ////////////////
                            var funcRet = origFunc.apply(this, arguments);
                            //////////////// End of ORIGINAL JS Function /////////////////

                            // Wrap the rest of the BrowserAgent instrumentation in a try, catch...
                            try {
                                if ( !isError && isFuncEnabled ) {
                                    funcMetricData[BrowserAgentGlobals.timestampNames.END_TIME] =
                                        BrowserAgentUtils.funcUtils.endTrace();
                                    BrowserAgentUtils.metricUtils.addDataPoint(
                                        BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION],
                                        key, funcMetricData);
                                    BrowserAgentUtils.metricUtils.harvestMetrics(
                                        BrowserAgentGlobals.metricType.FUNCTION);
                                }
                            } catch ( e ) {
                                isError = true;
                                BrowserAgentLogger.error("instrumentFunc (Checkpoint # 10) : " + e.message);
                            }
                            return funcRet;
                        };
                        break;
                }
                BrowserAgentLogger.info("Finished Instrumentation of " + funcInString);
                // This is a flag (e.g. hack) to prevent re-instrumentation of the same function.
                // Imagine frames embedding HTML pages within a top level HTML page. When the top
                // level HTML page begins to load, BrowserAgent JS function instrumentation occurs. Then, when
                // each of the frames load their own HTML pages, the BrowserAgent JS function instrumentation
                // occurs again. However, the functions have been already instrumented with custom
                // BrowserAgent logic. So, upon re-instrumentation, the custom logic is recursively invoked,
                // leading to a stack overflow.
                if ( redefinedFunc ) {
                    redefinedFunc._isBrowserAgentInstrumented = true;
                }
                return redefinedFunc;
            },
            /**
             * Retry the instrumentation of a js function for a maximum of 5 times with at least 5 seconds interval
             * @param funcInString - name of the js function as declared in apmbrowseragentextensibility.js
             * @param key - key of the js function as declared in apmbrowseragentextensibility.js
             * @param maxRetryCount -
             */
            retryInstrumentFunc : function ( funcInString, key, maxRetryCount ) {
                if ( typeof funcInString != 'string' || typeof key != 'string' ||
                     !BrowserAgentGlobals.functionsToInstrument || !BrowserAgentGlobals.functionsToInstrument[key] ) {
                    BrowserAgentLogger.warn("retryInstrumentFunc: JS Function [" + funcInString +
                                            "] could not be instrumented.");
                }
                // Instrumentation could fail if the JS Function to instrument
                // has not come into the window's scope yet. As an example, consider function
                // A() in another file that would be downloaded from a CDN at a later time than
                // our BrowserAgent JS code. So, we retry the instrumentation for N number of times.
                if ( maxRetryCount == null || maxRetryCount == undefined ) {
                    maxRetryCount = 5;
                }
                if ( maxRetryCount < 0 ) {
                    BrowserAgentLogger.warn("retryInstrumentFunc: JS Function [" + funcInString +
                                            "] could not be instrumented.");
                } else {
                    BrowserAgentLogger.info("Retrying instrumentation for JS Function [" +
                                            funcInString + "]...");
                    setTimeout(function () {
                        eval("var res = BrowserAgentUtils.funcUtils.instrumentFunc('" + funcInString + "','" + key +
                             "'," + (maxRetryCount - 1) + "); if (res) { window." +
                             BrowserAgentGlobals.functionsToInstrument[key].name + " = res; }");
                    }, 5000);
                }
            },
            /**
             * Instrument all the functions placed in BrowserAgentGlobals.functionsToInstrument
             */
            instrumentAllFunc : function () {
                if ( !BrowserAgentGlobals.functionsToInstrument ) {
                    BrowserAgentLogger.info("No JS functions to instrument.");
                    return;
                }
                // Save XHR object itself
                if ( window && window.XMLHttpRequest ) {
                    this.saveOrigObj("XHR_ctor", window.XMLHttpRequest);
                }
                // NOTE: this will not work in IE8 and below
                for ( var item in BrowserAgentGlobals.functionsToInstrument ) {
                    if ( BrowserAgentGlobals.functionsToInstrument[item].name ) {
                        try {
                            // The JS function is in the scope of the window and needs to be
                            // assigned the newly generated function by BrowserAgent. The assignment at run
                            // time can be achieved by eval
                            eval(
                                "var res = BrowserAgentUtils.funcUtils.instrumentFunc(BrowserAgentGlobals.functionsToInstrument['" +
                                item + "'].name, '" + item + "'); if (res) { window." +
                                BrowserAgentGlobals.functionsToInstrument[item].name + " = res; }");
                        } catch ( e ) {
                            // Do nothing. Retry is implemented in instrumentFunc
                        }
                    }
                }
            },
            /**
             * Given a object, this function saves them in BrowserAgentGlobals.origFuncMap
             * @param key - an unique string
             * @param obj - JS function object or JS function to store
             */
            saveOrigObj : function ( key, obj ) {
                if ( !obj || !key ) {
                    BrowserAgentLogger.warn(
                        "saveOrigObj : Cannot save original object without key or the object itself.");
                    return;
                }
                // NOTE: this will not work in IE8 and below
                Object.defineProperty(BrowserAgentGlobals.origFuncMap, key, {
                    value : obj,
                    writable : false,
                    enumerable : false,
                    configurable : false
                });
            },
            /**
             * Trace function that precedes the monitored application source
             * @returns {*}
             */
            startTrace : function () {
                return BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
            }
        },
        /**
         * Metric Utility
         * Responsible for BrowserAgent Metrics
         */
        metricUtils : {
            batchID : null,
            XHRToSendMetrics : null,
            // NOTE: There is no need for locking in any of these methods
            // because Javascript, by design, is single threaded
            init : function () {
                try {
                    if ( window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_JSFUNCTIONMETRICSENABLED] ) {
                        this.addMetricType("function", this.harvestFuncMetrics);
                    }
                    if ( window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_AJAXMETRICSENABLED] ) {
                        this.addMetricType("ajax", this.harvestAjaxMetrics);
                    }
                } catch ( e ) {
                    BrowserAgentLogger.error("metricUtils.init : " + e.message);
                }
            },
            /**
             * Add a data point into the given accumulator for a given key
             * Assumption: data points reside in an array for a given key
             * If the key is present, then append the data point into the array for that key
             * If key is not present, then create a new array with that data point
             * @param acc - Accumulator in which the data point is to be added
             * @param key - an unique string
             * @param data - preferably, a JS Object containing data
             */
            addDataPoint : function ( acc, key, data ) {
                if ( this.isValid(acc, key) && data != null && data != undefined ) {
                    // 1st Data point
                    if ( !acc[key] ) {
                        acc[key] = [data];
                    } else {
                        acc[key].push(data);
                    }
                }
            },
            /**
             * Adds a metric type into BrowserAgentGlobals.metricType map.
             * BrowserAgent has 4 flavors of metrics and each corresponds to a metric Type
             * 1 : Page
             * 2 : Function
             * 3 : Ajax
             * Furthermore, each metric type has an accumulator and a registered harvest function
             * that reaps the data points of that metric type every so often as governed by the
             * metric frequency config parameter.
             * @param metricType - type of metric (e.g. AngularJS)
             * @param harvestFunction - JS Function used for metric reap
             */
            addMetricType : function ( metricType, harvestFunction ) {
                if ( !metricType || typeof metricType !== 'string' || !BrowserAgentGlobals ||
                     !BrowserAgentGlobals.metricType || !harvestFunction ||
                     typeof harvestFunction !== 'function' || !BrowserAgentGlobals.metricTypeToAccumulatorMap ||
                     !BrowserAgentGlobals.metricTypeToHarvestMap ) {
                    BrowserAgentLogger.error("addMetricType : cannot add metric type for " + metricType);
                    return;
                }
                var lowerCaseMetricType = metricType.toLowerCase();
                // NOTE: this will not work in IE8 and below
                Object.defineProperty(BrowserAgentGlobals.metricType, metricType.toUpperCase(), {
                    value : lowerCaseMetricType,
                    writable : true,
                    enumerable : true,
                    configurable : true
                });
                // Add Accumulator
                // NOTE: this will not work in IE8 and below
                Object.defineProperty(BrowserAgentGlobals.metricTypeToAccumulatorMap,
                                      lowerCaseMetricType, {
                        value : {},
                        writable : true,
                        enumerable : true,
                        configurable : true
                    });
                // Register Harvest Function
                // NOTE: this will not work in IE8 and below
                Object.defineProperty(BrowserAgentGlobals.metricTypeToHarvestMap, lowerCaseMetricType, {
                    value : harvestFunction,
                    writable : true,
                    enumerable : true,
                    configurable : true
                });
            },
            /**
             * Construct the metric path for a given metric in the metric browser tree
             * @param metricPfx - metric path that would succeed Business Segment | BS | BT | BTC or
             * Business Segment | <domain/port> | <path>
             * @param metricName - metric name displayed in the metric browser
             * @param metricVal - metric value (Must be an number)
             * @param metricAggregatorType - refer to BrowserAgentGlobals.metricAggregatorType
             * @returns {string}
             */
            constructMetricPath : function ( metricPfx, metricName, metricVal,
                                             metricAggregatorType ) {
                if ( typeof metricPfx !== 'string' || typeof metricName !== 'string' ||
                     typeof metricVal !== 'number' || typeof metricAggregatorType !== 'number' ) {
                    BrowserAgentLogger.warn("constructMetricPath : cannot construct metric path for {" +
                                            metricPfx + ", " + metricName + ", " + metricVal + ", " +
                                            metricAggregatorType + "}");
                    return "";
                }
                // If Metric Prefix is non-empty, then include colon char in path
                if ( metricPfx.length > 0 ) {
                    metricPfx = BrowserAgentUtils.metricUtils.metricPrefixSanitize(metricPfx) +
                                BrowserAgentGlobals.colonChar;
                }
                var unformattedMetricPath = metricPfx + metricName + BrowserAgentGlobals.equalChar +
                                            BrowserAgentGlobals.openParenChar +
                                            metricVal + BrowserAgentGlobals.commaChar +
                                            metricAggregatorType +
                                            BrowserAgentGlobals.closedParenChar + BrowserAgentGlobals.semiColonChar;
                var formattedMetricPath = unformattedMetricPath;
                if ( BrowserAgentExtension ) {
                    formattedMetricPath = BrowserAgentExtension.extNameFormatter(unformattedMetricPath);
                }
                return formattedMetricPath;
            },
            /**
             * Validation function used before adding data points into an accumulator
             * @param acc - Accumulator (JS Object)
             * @param key - a unique identifier
             * @returns {boolean}
             */
            isValid : function ( acc, key ) {
                if ( !key ) {
                    BrowserAgentLogger.error("isValid : Metric key cannot be null or undefined");
                    return false;
                }
                if ( !acc ) {
                    BrowserAgentLogger.error("isValid : Metric location must be defined");
                    return false;
                }
                if ( typeof acc !== 'object' ) {
                    BrowserAgentLogger.error("isValid : Metric location must be an object");
                    return false;
                }
                return true;
            },
            /**
             * Validation function used to verify whether a given AJAX data point has all
             * the mandatory attributes
             * @param ajaxObj - JS Object containing AJAX data
             * @returns {boolean}
             */
            isValidAjaxDataPoint : function ( ajaxObj ) {
                if ( ajaxObj && ajaxObj["URL"] &&
                     ajaxObj[BrowserAgentGlobals.timestampNames.FIRST_BYTE] &&
                     ajaxObj[BrowserAgentGlobals.timestampNames.LAST_BYTE] &&
                     ajaxObj[BrowserAgentGlobals.timestampNames.CALLBACK_START_TIME] &&
                     ajaxObj[BrowserAgentGlobals.timestampNames.CALLBACK_END_TIME] ) {
                    return true;
                }
                return false;
            },
            /**
             * Obtains Txn Trace Geo-location property
             * @returns {string}
             */
            getGeoLocationOptionalParams : function () {
                return BrowserAgentUtils.cookieUtils.cookieKeys.geoChar +
                       BrowserAgentGlobals.equalChar +
                       BrowserAgentGlobals.geo.lat +
                       BrowserAgentGlobals.commaChar +
                       BrowserAgentGlobals.geo.long +
                       BrowserAgentGlobals.semiColonChar;
            },
            /**
             * Obtains Txn Trace properties for page and ajax metrics
             * @param ttFlag
             * @param corId - Agent Correlation ID
             * @param apmStartTime - Introscope Start Time

             * @param browserAgentStartTime
             * @param duration
             * @param ttfb - time to first byte
             * @returns {string}
             */
            getOptionalParameters : function ( ttFlag, corId, apmStartTime, browserAgentStartTime, duration, ttfb ) {
                var optParam = "";
                apmStartTime = parseInt(apmStartTime);
                if ( ttFlag !== "1" || typeof corId !== 'string' || !apmStartTime ||
                     typeof browserAgentStartTime !== 'number' ||
                     typeof duration !== 'number' || duration < 1 ) {
                    return optParam;
                }
                //add geo location data
                optParam += this.getGeoLocationOptionalParams();

                optParam += "Duration=" + duration + ";p=" + BrowserAgentGlobals.p + ";pv=" + BrowserAgentGlobals.pv +
                            ";CorBrowsGUID=" + corId + ";StartTime=";
                // if browser start time is off, use server start time and ttfb to adjust it
                if ( (browserAgentStartTime > apmStartTime) || ((browserAgentStartTime + duration) < apmStartTime) ) {
                    optParam += apmStartTime - parseInt(ttfb / 2);
                } else {
                    optParam += browserAgentStartTime;
                }
                // EXTENSION POINT for adding Txn Trace properties
                if ( BrowserAgentExtension ) {
                    optParam += BrowserAgentExtension.extAddCustomOptionalProperty();
                }
                return optParam;
            },
            /**
             * Given a URL, strip the pathname from it
             * Assumes that the URL has been sanitized with sanitizeURL
             * @param url - sanitized URL
             * @returns {*}
             */
            getPathnameFromURL : function ( url ) {
                if ( !url || typeof url !== 'string' ) {
                    return url;
                }
                var slashIdx = url.indexOf(BrowserAgentGlobals.forwardSlashChar);
                var pathName = BrowserAgentGlobals.forwardSlashChar;
                if ( slashIdx !== -1 ) {
                    pathName = url.substring(slashIdx);
                }
                return pathName;
            },
            /**
             * Redefine the XHR object's open and send call to original
             * Dispatch of BrowserAgent metrics requires XHR object. However, BrowserAgent instrumentation
             * add custom logic into the methods of this object. Hence, redefine them to
             * the original for the sake of metrics dispatch
             */
            getRedefinedXHRForMetrics : function () {
                if ( BrowserAgentGlobals.origFuncMap && BrowserAgentGlobals.origFuncMap["XHR_ctor"] &&
                     BrowserAgentGlobals.origFuncMap["XHR_Open"] &&
                     BrowserAgentGlobals.origFuncMap["XHR_Send"] ) {
                    BrowserAgentUtils.metricUtils.XHRToSendMetrics =
                        new (BrowserAgentGlobals.origFuncMap["XHR_ctor"])();
                    BrowserAgentUtils.metricUtils.XHRToSendMetrics.open = BrowserAgentGlobals.origFuncMap["XHR_Open"];
                    BrowserAgentUtils.metricUtils.XHRToSendMetrics.send = BrowserAgentGlobals.origFuncMap["XHR_Send"];
                } else {
                    // If the original functions are not found, do not use the redefined ones
                    // as this will generate an infinite loop
                    // Use null instead
                    BrowserAgentUtils.metricUtils.XHRToSendMetrics = null;
                }
            },
            /**
             * Reaps the AJAX data points stored into the AJAX metric accumulator
             * When dataId is specified, then reaps that particular AJAX data point
             * @param dataId - an unique identifier that corresponds to a valid AJAX data point,
             *                 residing in the  AJAX accumulator,
             *                 BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX]
             * @returns {string}
             */
            harvestAjaxMetrics : function ( dataId ) {
                var ajaxPostData = "";
                var ajaxMetricAccumulator = BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX];
                var ajaxSendAccumulator = BrowserAgentGlobals.ajaxSendDataMap;
                if ( !ajaxMetricAccumulator || !ajaxSendAccumulator ) {
                    return ajaxPostData;
                }
                var ajaxDataPoint;
                var sendDataPoint;
                var ttfb;
                var trlt;
                var cbet;
                var ttt;
                var resourceURL;
                var ajaxMetrics = {};
                var TTParams;
                var bs;
                var bt;
                var btc;
                var urlInitialMetricPfx = BrowserAgentUtils.metricUtils.convertURL(
                    BrowserAgentUtils.metricUtils.sanitizeURL(BrowserAgentGlobals.currentURL));
                var threshold = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_AJAXMETRICSTHRESHOLD];
                var isURLMetricOff = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_URLMETRICOFF];
                var aggregatorType = BrowserAgentGlobals.metricAggregatorType.INT_LONG_DURATION;
                for ( var id in ajaxMetricAccumulator ) {
                    var currentAjaxPostData = "";
                    // If dataId is specified, then a single Ajax Data Point needs to be harvested
                    // Do so by setting the id = dataId;
                    if ( dataId ) {
                        id = dataId;
                    }
                    ajaxDataPoint =
                        (ajaxMetricAccumulator[id]) ? ajaxMetricAccumulator[id][0] : null;
                    sendDataPoint =
                        (ajaxSendAccumulator[id]) ? ajaxSendAccumulator[id][0] : null;
                    if ( !ajaxDataPoint || !sendDataPoint ) {
                        if ( dataId ) {
                            break;
                        } else {
                            continue;
                        }
                    }
                    // If the Ajax Data Points are not valid, then why bother with calculations?
                    // Just delete them and move on
                    // Validity is based on the presence of required keys in the data point object:
                    // URL, f, l, cs, ce, e
                    if ( !BrowserAgentUtils.metricUtils.isValidAjaxDataPoint(ajaxDataPoint) ||
                         !sendDataPoint[BrowserAgentGlobals.timestampNames.END_TIME] ) {
                        BrowserAgentLogger.warn("Obtained invalid AJAX data point. Deleting it...");
                        delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX][id];
                        delete BrowserAgentGlobals.ajaxSendDataMap[id];
                        if ( dataId ) {
                            break;
                        } else {
                            continue;
                        }
                    }
                    resourceURL =
                        BrowserAgentUtils.metricUtils.convertURL(
                            BrowserAgentUtils.metricUtils.sanitizeURL(ajaxDataPoint['URL']));
                    // Obtain BS info
                    bs = (ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.bsChar] &&
                          ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.bsChar] !==
                          BrowserAgentGlobals.UNDEFINED) ?
                         ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.bsChar] :
                         BrowserAgentGlobals.bs;
                    // If bs is not defined, then the data point is in URL context. Check if the
                    // URL metrics are on. If not, skip the rest of the logic
                    if ( bs === BrowserAgentGlobals.UNDEFINED ) {
                        if ( isURLMetricOff ) {
                            BrowserAgentLogger.info("Skipping harvest of AJAX metrics as NON-BT based metrics are OFF");
                            delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX][id];
                            delete BrowserAgentGlobals.ajaxSendDataMap[id];
                            if ( dataId ) {
                                break;
                            } else {
                                continue;
                            }
                        }
                    }
                    trlt = ajaxDataPoint[BrowserAgentGlobals.timestampNames.CALLBACK_END_TIME] -
                           sendDataPoint[BrowserAgentGlobals.timestampNames.END_TIME];
                    // Skip the rest of logic if Total Resource Load Time < Threshold
                    // with continue
                    if ( trlt < threshold ) {
                        BrowserAgentLogger.info("Skipping harvest of AJAX metrics for " +
                                                ajaxDataPoint['URL'] +
                                                " as it is below the configured AJAX Metric Threshold");
                        delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX][id];
                        delete BrowserAgentGlobals.ajaxSendDataMap[id];
                        if ( dataId ) {
                            break;
                        } else {
                            continue;
                        }
                    }
                    // Obtain bt and btc info
                    bt =
                        (bs === BrowserAgentGlobals.UNDEFINED ||
                         !ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.btChar]) ?
                        BrowserAgentGlobals.bt : ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.btChar];
                    btc =
                        (bs === BrowserAgentGlobals.UNDEFINED ||
                         !ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.btcChar]) ?
                        BrowserAgentGlobals.btc :
                        ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.btcChar];

                    // If in URL context, then set metric path appropriately
                    var urlPath = BrowserAgentGlobals.UNDEFINED;
                    var finalMetricPfx = "";
                    if ( bs === BrowserAgentGlobals.UNDEFINED ) {
                        urlPath = urlInitialMetricPfx;
                        finalMetricPfx = urlInitialMetricPfx + BrowserAgentGlobals.pipeChar;
                    }
                    finalMetricPfx += BrowserAgentGlobals.metricPath.AJAX + BrowserAgentGlobals.pipeChar +
                                      resourceURL;
                    // Perform AJAX metric calculations
                    ttfb = ajaxDataPoint[BrowserAgentGlobals.timestampNames.FIRST_BYTE] -
                           sendDataPoint[BrowserAgentGlobals.timestampNames.END_TIME];
                    ttt = ajaxDataPoint[BrowserAgentGlobals.timestampNames.LAST_BYTE] -
                          ajaxDataPoint[BrowserAgentGlobals.timestampNames.FIRST_BYTE];
                    cbet = ajaxDataPoint[BrowserAgentGlobals.timestampNames.CALLBACK_END_TIME] -
                           ajaxDataPoint[BrowserAgentGlobals.timestampNames.CALLBACK_START_TIME];
                    ajaxMetrics[BrowserAgentGlobals.defaultMetricNames.AJAX_TRLT] =
                        [trlt, aggregatorType];
                    ajaxMetrics[BrowserAgentGlobals.defaultMetricNames.AJAX_TTFB] =
                        [ttfb, aggregatorType];
                    ajaxMetrics[BrowserAgentGlobals.defaultMetricNames.AJAX_TTT] =
                        [ttt, aggregatorType];
                    ajaxMetrics[BrowserAgentGlobals.defaultMetricNames.AJAX_CBET] =
                        [cbet, aggregatorType];
                    ajaxMetrics[BrowserAgentGlobals.defaultMetricNames.AJAX_ICPI] =
                        [1, BrowserAgentGlobals.metricAggregatorType.LONG_INTERVAL_COUNTER];
                    // EXTENSION POINT for adding custom AJAX metrics
                    if ( BrowserAgentExtension ) {
                        BrowserAgentExtension.extAddCustomAjaxMetric();
                        for ( var item in BrowserAgentExtension.extCustomAjaxMetricMap ) {
                            ajaxMetrics[item] = BrowserAgentExtension.extCustomAjaxMetricMap[item];
                        }
                        BrowserAgentExtension.extCustomAjaxMetricMap = {};
                    }
                    // Validate the AJAX metrics, including the custom ones from the EXTENSION POINT
                    BrowserAgentUtils.metricUtils.validateMetrics(ajaxMetrics);
                    for ( var metric in ajaxMetrics ) {
                        currentAjaxPostData += BrowserAgentUtils.metricUtils.constructMetricPath(finalMetricPfx,
                                                                                                 metric,
                                                                                                 ajaxMetrics[metric][0],
                                                                                                 ajaxMetrics[metric][1]);
                    }
                    // Obtain Txn Trace properties
                    TTParams = BrowserAgentUtils.metricUtils.getOptionalParameters(
                        ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.ttChar],
                        ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.CorBrowsGUIDChar],
                        ajaxDataPoint[BrowserAgentUtils.cookieUtils.cookieKeys.apmStartTimeChar],
                        sendDataPoint[BrowserAgentGlobals.timestampNames.END_TIME], trlt, ttfb);
                    if ( currentAjaxPostData.length > 0 ) {
                        // Form the data in Post Parameter Data Format
                        // b0: <optional Params>$bs=-1,bt=-1,btc=-1;Responses Per Interval=(2,1);
                        currentAjaxPostData = BrowserAgentUtils.metricUtils.placeInPostParameterFormat(bs,
                                                                                                       bt,
                                                                                                       btc,
                                                                                                       urlPath,
                                                                                                       TTParams,
                                                                                                       currentAjaxPostData);
                        ajaxPostData += currentAjaxPostData;
                        BrowserAgentGlobals.bCount++;
                    }
                    delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.AJAX][id];
                    delete BrowserAgentGlobals.ajaxSendDataMap[id];
                    if ( dataId ) {
                        break;
                    }
                }
                return ajaxPostData;
            },
            /**
             * Reaps data points of all metric types
             * Used in BATCH MODE
             */
            harvestAllMetrics : function () {
                try {
                    BrowserAgentGlobals.postData = "";
                    for ( var metricType in BrowserAgentGlobals.metricTypeToHarvestMap ) {
                        BrowserAgentGlobals.postData += (BrowserAgentGlobals.metricTypeToHarvestMap[metricType])();
                    }
                    if ( BrowserAgentGlobals.postData.length !== 0 ) {
                        BrowserAgentGlobals.postData = "bCount=" + BrowserAgentGlobals.bCount +
                                                       BrowserAgentGlobals.postData;
                        BrowserAgentUtils.metricUtils.sendMetrics(
                            window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_WILYURL],
                            BrowserAgentGlobals.postData);
                    }
                } catch ( e ) {
                    // Reset
                    if ( BrowserAgentGlobals ) {
                        BrowserAgentGlobals.postData = ""
                        BrowserAgentGlobals.bCount = 0;
                        // Don't keep any data points around if there is an error
                        for ( var acc in BrowserAgentGlobals.metricTypeToAccumulatorMap ) {
                            BrowserAgentGlobals.metricTypeToAccumulatorMap[acc] = {};
                        }
                    }
                    BrowserAgentLogger.error("harvestAllMetrics : " + e.message);
                }
            },
            /**
             * Reaps the JS Func data points stored into the JS Func metric accumulator
             * @returns {string}
             */
            harvestFuncMetrics : function () {
                var funcPostData = "";
                var accumulator = BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION];
                if ( !accumulator ) {
                    return funcPostData;
                }
                var aet = 0;
                var funcMetrics = {};
                var metricPfx = BrowserAgentGlobals.metricPath.FUNC;
                var urlPath = BrowserAgentGlobals.UNDEFINED;
                // If bs is not defined, then the data point is in URL context. Check if the
                // URL metrics are on. If not, skip the rest of the logic
                if ( BrowserAgentGlobals.bs === BrowserAgentGlobals.UNDEFINED ) {
                    if ( window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_URLMETRICOFF] ) {
                        BrowserAgentLogger.info(
                            "Skipping harvest of JS Function metrics as NON-BT based metrics are OFF");
                        BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION] = {};
                        return funcPostData;
                    }
                    // Set metric path appropriately for URL context
                    metricPfx =
                        BrowserAgentUtils.metricUtils.convertURL(
                            BrowserAgentUtils.metricUtils.sanitizeURL(BrowserAgentGlobals.currentURL));
                    urlPath = metricPfx;
                    metricPfx += BrowserAgentGlobals.pipeChar + BrowserAgentGlobals.metricPath.FUNC;
                }
                var threshold = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_JSFUNCTIONMETRICSTHRESHOLD];
                for ( var funcName in accumulator ) {
                    for ( var i = 0; i < accumulator[funcName].length; i++ ) {
                        if ( !accumulator[funcName][i][BrowserAgentGlobals.timestampNames.END_TIME] ||
                             !accumulator[funcName][i][BrowserAgentGlobals.timestampNames.START_TIME] ) {
                            BrowserAgentLogger.warn("Obtained invalid JavaScript Function data point. Deleting it...");
                            delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION][funcName][i];
                            continue;
                        }
                        aet = accumulator[funcName][i][BrowserAgentGlobals.timestampNames.END_TIME] -
                              accumulator[funcName][i][BrowserAgentGlobals.timestampNames.START_TIME];
                        // Skip the rest of logic if Average Execution Time < Threshold
                        // with continue
                        if ( aet < threshold ) {
                            BrowserAgentLogger.info("Skipping harvest of JS function metrics for " +
                                                    funcName +
                                                    " as it is below the configured JS Function Metric Threshold");
                            delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION][funcName][i];
                            continue;
                        }
                        funcMetrics[BrowserAgentGlobals.defaultMetricNames.FUNC_AET] =
                            [aet, BrowserAgentGlobals.metricAggregatorType.INT_LONG_DURATION];
                        funcMetrics[BrowserAgentGlobals.defaultMetricNames.FUNC_ICPI] =
                            [1, BrowserAgentGlobals.metricAggregatorType.LONG_INTERVAL_COUNTER];
                        // EXTENSION POINT for adding custom JS Function metrics
                        if ( BrowserAgentExtension ) {
                            BrowserAgentExtension.extAddCustomJSFuncMetric();
                            for ( var item in BrowserAgentExtension.extCustomJSFuncMetricMap ) {
                                funcMetrics[item] = BrowserAgentExtension.extCustomJSFuncMetricMap[item];
                            }
                            BrowserAgentExtension.extCustomJSFuncMetricMap = {};
                        }
                        // Validate the AJAX metrics, including the custom ones from the EXTENSION POINT
                        BrowserAgentUtils.metricUtils.validateMetrics(funcMetrics);
                        for ( var metric in funcMetrics ) {
                            funcPostData += BrowserAgentUtils.metricUtils.constructMetricPath(metricPfx +
                                                                                              BrowserAgentGlobals.pipeChar +
                                                                                              funcName,
                                                                                              metric,
                                                                                              funcMetrics[metric][0],
                                                                                              funcMetrics[metric][1]);
                        }
                    }
                    delete BrowserAgentGlobals.metricTypeToAccumulatorMap[BrowserAgentGlobals.metricType.FUNCTION][funcName];
                }
                // Form the data in Post Parameter Data Format
                // b0: <optional Params>$bs=-1,bt=-1,btc=-1;Responses Per Interval=(2,1);
                if ( funcPostData.length > 0 ) {
                    funcPostData = BrowserAgentUtils.metricUtils.placeInPostParameterFormat(BrowserAgentGlobals.bs,
                                                                                            BrowserAgentGlobals.bt,
                                                                                            BrowserAgentGlobals.btc,
                                                                                            urlPath,
                                                                                            "",
                                                                                            funcPostData);
                    BrowserAgentGlobals.bCount++;
                }
                return funcPostData;
            },
            /**
             * Given a metric type, invoke its appropriate harvest function
             * Note: this works in two modes governed by the configuration parameter metricFrequency
             *       1. BATCH MODE - Here, the metric frequency is a positive number in ms. The idea
             *       here is to reap the metrics every so often.
             *       2. EVENT MODE - Here, the metric frequency is zero. The idea here is to reap
             *       the metrics as soon the interested event occurs.
             * @param metricType - metric type as per BrowserAgentGlobals.metricType
             * @param dataId - an unique identifier correponding to a particular data point
             */
            harvestMetrics : function ( metricType, dataId ) {
                try {
                    var metricFreq = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_METRICFREQUENCY];
                    if ( metricFreq > 0 ) {
                        // BATCH MODE. Trigger the harvest of all metric types periodically
                        if ( !BrowserAgentUtils.metricUtils.batchID ) {
                            BrowserAgentUtils.metricUtils.batchID =
                                setInterval(BrowserAgentUtils.metricUtils.harvestAllMetrics,
                                            metricFreq);
                        }
                        return;
                    }
                    // Here, metric frequency must be 0. So, clear the periodic harvest
                    if ( BrowserAgentUtils.metricUtils.batchID ) {
                        clearInterval(BrowserAgentUtils.metricUtils.batchID);
                        BrowserAgentUtils.metricUtils.batchID = null;
                    }
                    if ( !metricType || typeof metricType !== 'string' ) {
                        BrowserAgentLogger.warn("harvestMetrics : cannot harvest metrics for metric type " +
                                                metricType);
                        return;
                    }
                    BrowserAgentGlobals.postData = "";
                    // Invoke the harvest function associated with the given metric type
                    BrowserAgentGlobals.postData += (BrowserAgentGlobals.metricTypeToHarvestMap[metricType])(dataId);
                    if ( BrowserAgentGlobals.postData.length > 0 ) {
                        BrowserAgentGlobals.postData = "bCount=" + BrowserAgentGlobals.bCount +
                                                       BrowserAgentGlobals.postData;
                        // Send the metrics
                        BrowserAgentUtils.metricUtils.sendMetrics(
                            window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_WILYURL],
                            BrowserAgentGlobals.postData);
                    }
                } catch ( e ) {
                    // Reset
                    if ( BrowserAgentGlobals ) {
                        BrowserAgentGlobals.postData = ""
                        BrowserAgentGlobals.bCount = 0;
                        // Don't keep any data points around if there is an error
                        for ( var acc in BrowserAgentGlobals.metricTypeToAccumulatorMap ) {
                            BrowserAgentGlobals.metricTypeToAccumulatorMap[acc] = {};
                        }
                    }
                    BrowserAgentLogger.error("harvestMetrics : " + e.message);
                }
            },
            /**
             * Reaps the page data points stored into a global cookie and Browser Agent variables
             * Note: Page metrics are not periodic. They are only reaped as soon as the page is
             * loaded.
             * @returns {string}
             */
            harvestPageMetrics : function () {
                var pageLoadData = "";
                // Skip the rest of logic if Average Page Load Complete Time is < Threshold
                // and return empty pageLoadData
                var threshold = window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_PAGELOADMETRICSTHRESHOLD];
                if ( BROWSERAGENT.tOnLoad < threshold ) {
                    BrowserAgentLogger.info(
                        "Skipping harvest of Page metrics for as it is below the configured Page Metric Threshold");
                    return pageLoadData;
                }
                var metricPrefix = "";
                var pageURLPath = BrowserAgentGlobals.UNDEFINED;
                // BT is not defined
                if ( BrowserAgentGlobals.bs === BrowserAgentGlobals.UNDEFINED ) {
                    if ( window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_URLMETRICOFF] ) {
                        BrowserAgentLogger.info("Skipping harvest of Page metrics as NON-BT based metrics are OFF");
                        return pageLoadData;
                    }
                    metricPrefix =
                        BrowserAgentUtils.metricUtils.convertURL(
                            BrowserAgentUtils.metricUtils.sanitizeURL(BrowserAgentGlobals.currentURL));
                    pageURLPath = metricPrefix;
                }
                // Ideally, we are supposed to calculate the page load metrics here. But, the
                // old BrowserAgent code does the calculations in the page onload() method.
                // So, steal it from there.
                var pageLoadMetrics = {};
                var aggregatorType = BrowserAgentGlobals.metricAggregatorType.INT_LONG_DURATION;
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ABRT] =
                    [BROWSERAGENT.tBRT, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ACET] =
                    [BROWSERAGENT.tConn, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ADNSLT] =
                    [BROWSERAGENT.tDNS, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ADCT] =
                    [BROWSERAGENT.tDomReady, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_APLCT] =
                    [BROWSERAGENT.tOnLoad, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_APPUT] =
                    [BROWSERAGENT.tUnload, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ARTT] =
                    [BROWSERAGENT.tRTT, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ATTFB] =
                    [BROWSERAGENT.tFirstByte, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ATTLB] =
                    [BROWSERAGENT.tLastByte, aggregatorType];
                pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.OTHER_RPI] =
                    [1, BrowserAgentGlobals.metricAggregatorType.LONG_INTERVAL_COUNTER];
                // EXTENSION POINT for adding custom page metrics
                if ( BrowserAgentExtension ) {
                    BrowserAgentExtension.extAddCustomPageMetric();
                    for ( var item in BrowserAgentExtension.extCustomPageMetricMap ) {
                        pageLoadMetrics[item] = BrowserAgentExtension.extCustomPageMetricMap[item];
                    }
                    BrowserAgentExtension.extCustomPageMetricMap = {};
                }
                // Validate the page metrics, including the custom ones from the EXTENSION POINT
                BrowserAgentUtils.metricUtils.validateMetrics(pageLoadMetrics);

                BrowserAgentLogger.info(" ####### PAGE TIMINGS #######");
                for ( var metric in pageLoadMetrics ) {
                    BrowserAgentLogger.info(metric + " = " + pageLoadMetrics[metric][0]);
                }
                // Obtain Txn Trace properties
                var TTParams = BrowserAgentUtils.metricUtils.getOptionalParameters(BrowserAgentGlobals.c,
                                                                                   BrowserAgentGlobals.CorBrowsGUID,
                                                                                   BrowserAgentGlobals.startTime,
                                                                                   BROWSERAGENT.cookie.bts,
                                                                                   BROWSERAGENT.tOnLoad,
                                                                                   pageLoadMetrics[BrowserAgentGlobals.defaultMetricNames.NTAPI_ATTFB][0]);
                for ( var metric in pageLoadMetrics ) {
                    pageLoadData += BrowserAgentUtils.metricUtils.constructMetricPath(metricPrefix,
                                                                                      metric,
                                                                                      pageLoadMetrics[metric][0],
                                                                                      pageLoadMetrics[metric][1]);
                }
                if ( pageLoadData.length > 0 ) {
                    // Form the data in Post Parameter Data Format
                    // b0: <optional Params>$bs=-1,bt=-1,btc=-1;Responses Per Interval=(2,1);
                    pageLoadData = BrowserAgentUtils.metricUtils.placeInPostParameterFormat(BrowserAgentGlobals.bs,
                                                                                            BrowserAgentGlobals.bt,
                                                                                            BrowserAgentGlobals.btc,
                                                                                            pageURLPath,
                                                                                            TTParams,
                                                                                            pageLoadData);
                    BrowserAgentGlobals.bCount++;
                }
                return pageLoadData;
            },
            /**
             * Sanity checking of the metric prefix
             * Replace colons with underscore. More Sanity checking can be added here.
             * @param str
             * @returns {*}
             */
            metricPrefixSanitize : function ( str ) {
                if ( !str || typeof str !== 'string' ) {
                    BrowserAgentLogger.warn("metricPrefixSanitize : cannot sanitize metrics for " + str);
                    return str;
                }
                var saneStr = str;
                // Replace colons with underscore
                if ( str ) {
                    saneStr = saneStr.replace(/:/g, "_");
                }
                return saneStr;
            },
            /**
             * Converts a sanitized URL into hostname/port|pathname
             * Assumption: URL has been sanitized with sanitizeURL function
             * @param urlStr - sanitized URL string
             * @returns {*}
             */
            convertURL : function ( urlStr ) {
                if ( !urlStr || typeof urlStr !== 'string' || urlStr.length < 1 ) {
                    BrowserAgentLogger.warn("convertURL : cannot convert URL - " + urlStr);
                    return "";
                }
                var slashIdx = urlStr.indexOf(BrowserAgentGlobals.forwardSlashChar);
                var noColonStr = urlStr.replace(/:/g, '/');
                // No pathname? Just have an empty path starting with /
                var convertedStr = noColonStr + BrowserAgentGlobals.pipeChar +
                                   BrowserAgentGlobals.forwardSlashChar;
                // If pathname, then hostname/port|pathname|...
                if ( slashIdx !== -1 ) {
                    convertedStr = noColonStr.substring(0, slashIdx) + BrowserAgentGlobals.pipeChar +
                                   noColonStr.substring(slashIdx);
                }
                return convertedStr;
            },
            /**
             * Given the data, place it in the post parameter data format for dispatch to Agent
             * @param bs - CEM business segment
             * @param bt - CEM BT
             * @param btc - CEM BTC
             * @param urlPath - path of the current webpage
             * @param optionalParams - Txn Trace properties
             * @param data - string as given by metricUtils.constructMetricPath
             * @returns {string}
             */
            placeInPostParameterFormat : function ( bs, bt, btc, urlPath, optionalParams, data ) {
                if ( typeof data !== 'string' || data.length < 1 ) {
                    return "";
                }
                if ( typeof bs !== 'string' || typeof bt !== 'string' || typeof btc !== 'string' ) {
                    bs = BrowserAgentGlobals.UNDEFINED;
                    bt = BrowserAgentGlobals.UNDEFINED;
                    btc = BrowserAgentGlobals.UNDEFINED;
                }
                if ( typeof urlPath !== 'string' ) {
                    urlPath = "";
                }
                if ( typeof optionalParams !== 'string' ) {
                    optionalParams = "";
                }
                return BrowserAgentGlobals.ampersandChar + "b" + BrowserAgentGlobals.bCount +
                       BrowserAgentGlobals.equalChar + optionalParams +
                       BrowserAgentGlobals.dollarChar + BrowserAgentUtils.cookieUtils.cookieKeys.bsChar +
                       BrowserAgentGlobals.equalChar + encodeURIComponent(bs) +
                       BrowserAgentGlobals.commaChar + BrowserAgentUtils.cookieUtils.cookieKeys.btChar +
                       BrowserAgentGlobals.equalChar + encodeURIComponent(bt) +
                       BrowserAgentGlobals.commaChar + BrowserAgentUtils.cookieUtils.cookieKeys.btcChar +
                       BrowserAgentGlobals.equalChar + encodeURIComponent(btc) +
                       BrowserAgentGlobals.commaChar + BrowserAgentGlobals.urlChar + BrowserAgentGlobals.equalChar +
                       urlPath + BrowserAgentGlobals.semiColonChar + data.substr(0, data.length - 1);
            },
            /**
             * Replaces all occurrences of a character or a string inside a string with another.
             * Note: Need to escape special characters in "find" before passing in!
             * @param str - original string
             * @param find - string to be replaced
             * @param replace - string to replace
             * @returns {*}
             */
            replaceAll : function ( str, find, replace ) {
                if ( typeof str !== 'string' || typeof find !== 'string' || typeof replace !== 'string' ) {
                    BrowserAgentLogger.warn("replaceAll : invalid inputs");
                    return null;
                }
                return str.replace(new RegExp(find, 'g'), replace);
            },
            /**
             * Utility function to clear the give accumulator
             * @param accName - name of the accumulator (JS Object)
             */
            resetAccumulator : function ( accName ) {
                if ( !accName || typeof accName !== 'string' || !BrowserAgentGlobals ||
                     !BrowserAgentGlobals[accName] ) {
                    BrowserAgentLogger.warn("resetAccumulator : could not rest accumulator " + accName);
                    return;
                }
                BrowserAgentGlobals[accName] = {};
            },
            /**
             * Sanity checks the given URL as a string
             * 1. Strip Query Params
             * 2. Remove Anchors
             * 3. Remove chars after semi-colons
             * 4. Decode
             * 5. Remove leading and trailing slashes
             * 6. Remove protocol (http, https)
             * 7. Append Port # if not present
             * @param urlStr - HTTP URL
             * @returns {*}
             */
            sanitizeURL : function ( urlStr ) {
                // does steps 1 to 5
                var saneStr = BrowserAgentUtils.metricUtils.trimURL(urlStr);
                if ( !saneStr ) {
                    return null;
                }
                // Which protocol
                var defaultPortNum = 80;
                if ( saneStr.search(/^(http|https):/g) !== -1 ) {
                    if ( saneStr.indexOf("https") !== -1 ) {
                        defaultPortNum = 443;
                    }
                    saneStr = saneStr.replace(/(http|https):\/\//g, '');
                } else {
                    if ( saneStr.indexOf("/") === 0 ) {
                        saneStr = window.location.host + saneStr;
                    } else {
                        saneStr = window.location.host + "/" + saneStr;
                    }
                }
                // If Port # is not there, then add it
                var slashIdx = saneStr.indexOf(BrowserAgentGlobals.forwardSlashChar);
                var stringToSearch = (slashIdx !== -1) ? saneStr.substring(0, slashIdx) :
                                     saneStr;
                if ( stringToSearch.indexOf(":") === -1 ) {
                    stringToSearch = stringToSearch + ":" + defaultPortNum;
                }
                if ( slashIdx !== -1 ) {
                    saneStr = stringToSearch + saneStr.substring(slashIdx);
                } else {
                    saneStr = stringToSearch;
                }
                // Remove leading and trailing slashes, if any
                if ( saneStr ) {
                    saneStr = saneStr.replace(/^\/|\/$/g, '');
                }
                return saneStr;
            },
            /**
             * Dispatches metrics to the Agent with AJAX POST payload
             * @param URL - preferably, BROWSERAGENT_WILYURL
             * @param data - string as per http://www.w3schools.com/ajax/ajax_xmlhttprequest_send.asp
             */
            sendMetrics : function ( URL, data ) {
                if ( !URL || !data || typeof URL !== 'string' ) {
                    BrowserAgentLogger.warn("sendMetrics : Cannot send BrowserAgent Metrics to URL: " + URL +
                                            " with data as {{ " + data + " }}.");
                    BrowserAgentGlobals.bCount = 0;
                    BrowserAgentGlobals.postData = "";
                    return;
                }
                // Metrics will be sent via XHR Send. However, we do not want to
                // collect metrics for such an invocation of XHR Send. Redefine them to original.
                BrowserAgentUtils.metricUtils.getRedefinedXHRForMetrics();
                var XHR = BrowserAgentUtils.metricUtils.XHRToSendMetrics;
                if ( XHR ) {
                    XHR.open("POST", URL, true);
                    XHR.onreadystatechange = function () {
                        if ( this.readyState === this.DONE && this.status === 0 ) {
                            BrowserAgentLogger.error(
                                "sendMetrics : BrowserAgent Metrics Send Error. Browser is most likely discarding them.");
                            BrowserAgentGlobals.bCount = 0;
                            BrowserAgentGlobals.postData = "";
                        }
                    };
                    XHR.setRequestHeader("Content-type", "application/x-www-form-urlencoded; charset=utf-8");
                    XHR.send(data);
                    BrowserAgentLogger.info("Sending POST with {{ " + data + " }}");
                }
                // Reset
                BrowserAgentGlobals.bCount = 0;
                BrowserAgentGlobals.postData = "";
            },
            /**
             * Decodes and trims the full URL:
             * 1. Strip Query Params
             * 2. Remove Anchors
             * 3. Remove chars after semi-colons
             * 4. Decode
             * @param urlStr - HTTP URL
             * @returns {*}
             */
            trimURL : function ( urlStr ) {
                if ( !urlStr || typeof urlStr !== 'string' ) {
                    BrowserAgentLogger.warn("trimURL : invalid URL - " + urlStr);
                    return null;
                }
                // Decode URL Encoding, if any
                var saneStr = decodeURIComponent(urlStr);
                var queryIdx = saneStr.indexOf('?');
                // Remove Query Parameters, if any
                if ( queryIdx !== -1 ) {
                    saneStr = saneStr.substring(0, queryIdx);
                }
                // Remove Anchors, if any
                var anchorIdx = saneStr.indexOf('#');
                if ( saneStr && anchorIdx !== -1 ) {
                    saneStr = saneStr.substring(0, anchorIdx);
                }
                // Remove things after Semi-colons
                // e.g. URL/path;jsessionid=39y459hnfannfla
                var semiColonIdx = saneStr.indexOf(BrowserAgentGlobals.semiColonChar);
                if ( saneStr && semiColonIdx !== -1 ) {
                    saneStr = saneStr.substring(0, semiColonIdx);
                }
                return saneStr;
            },
            /**
             * Given a metric Obj in a certain format, validates the metrics within it.
             * In the event that the metric value is negative, NaN or undefined, zero it out.
             * Format : "key" --> [metric value, metric aggregator type]
             * @param metricObj - JS Object where keys are metric names and values are
             *                    [metric value, metric aggregator type]
             */
            validateMetrics : function ( metricObj ) {
                if ( !metricObj || typeof metricObj !== 'object' ) {
                    BrowserAgentLogger.warn("validateMetrics : cannot validate metrics for undefined metric object");
                    return;
                }
                var metricVal;
                for ( var metric in metricObj ) {
                    if ( !metricObj[metric] || metricObj[metric].length !== 2 ) {
                        metricObj[metric] = [0, BrowserAgentGlobals.metricAggregatorType.INT_LONG_DURATION];
                        BrowserAgentLogger.warn(
                            "validateMetrics : invalid metric format. Zeroing it out and defaulting metric aggregator to INT_LONG_DURATION...");
                        continue;
                    }
                    metricVal = metricObj[metric][0];
                    if ( metricVal == null || metricVal == undefined ||
                         typeof metricVal !== 'number' || metricVal < 0 || isNaN(metricVal) ) {
                        BrowserAgentLogger.warn("validateMetrics : Obtained metric out of range : " +
                                                metric +
                                                ". Zeroing it out...");
                        metricObj[metric][0] = 0;
                    } else {
                        metricObj[metric][0] = parseInt(metricObj[metric][0]);
                    }
                }
            }
        },
        /**
         * Timing Utility
         * Responsible for time related tasks
         */
        timeUtils : {
            /**
             * Obtain current Date and Time with native JS Date()
             * @returns {Date}
             */
            getCurrTime : function () {
                return new Date();
            },
            /**
             * Obtain Time since the current page load in ms
             * @returns {number}
             */
            getCurrTimeInMillis : function () {
                if ( !BrowserAgentUtils.BrowserAgentTimer ) {
                    BrowserAgentLogger.warn("getCurrTimeInMillis: BrowserAgent Timer is not present.");
                    return 0;
                }
                return BrowserAgentUtils.BrowserAgentTimer.now();
            },
            /**
             * Obtain Time since Epoch in ms
             * @returns {number}
             */
            getCurrTimeInMillisSinceEpoch : function () {
                return new Date().getTime();
            },
            /**
             * Obtains Timer function
             * Obtain the browser "performance.now" function for browsers that support it
             * Otherwise, use a function that makes use of native JS Date()
             * @returns {*|{}}
             */
            getTimerObj : function () {
                var performance = window.performance || {};
                performance.now = (function () {
                    return performance.now || performance.webkitNow ||
                           performance.msNow || performance.oNow ||
                           performance.mozNow ||
                           function () {
                               return new Date().getTime();
                           };
                })();
                return performance;
            }
        }
    };
    /**
     * This object comprises of BrowserAgent global data structures and variables
     * that are accessed in other BrowserAgent objects.
     */
    var BrowserAgentGlobals = {
        // This map is needed to store the AJAX Send timestamps
        // Since AJAX is asynchronous, it is not know when the callback will occur
        // or it will ever occur. But, the AJAX metrics need send timestamps for its
        // calculations. So, store them in a separate map for easy access.
        ajaxSendDataMap : {},
        ampersandChar : "&",
        bCount : 0,
        // The current page Business Segment
        bs : "-1",
        // The current page Business Txn
        bt : "-1",
        // The current page Business Txn Component
        btc : "-1",
        BussSeg : "Business Segment",
        // Txn Trace Flag
        c : "0",
        closedParenChar : ")",
        colonChar : ":",
        commaChar : ",",
        CookiePath : window.location.pathname,
        // Txn Correlation ID
        CorBrowsGUID : null,
        currentURL : window.location.protocol + "//" + window.location.host +
                     window.location.pathname,
        currentFullURL : window.location.href,
        // Metric Name MACROS
        defaultMetricNames : {
            NTAPI_ABRT : "Average Browser Render Time (ms)",
            NTAPI_ADCT : "Average DOM Construction Time (ms)",
            NTAPI_APLCT : "Average Page Load Complete Time (ms)",
            NTAPI_APPUT : "Average Previous Page Unload Time (ms)",
            NTAPI_ARTT : "Average Round Trip Time (ms)",
            NTAPI_ADNSLT : "Average DNS Lookup Time (ms)",
            NTAPI_ATTFB : "Average Time to First Byte (ms)",
            NTAPI_ATTLB : "Average Time to Last Byte (ms)",
            NTAPI_ACET : "Average Connection Establishment Time (ms)",
            OTHER_RPI : "Responses Per Interval",
            FUNC_AET : "Average Execution Time (ms)",
            FUNC_ICPI : "Invocation Count Per Interval",
            AJAX_TRLT : "Total Resource Load Time (ms)",
            AJAX_TTFB : "Time To First Byte (ms)",
            AJAX_TTT : "Response Download Time (ms)",
            AJAX_CBET : "Callback Execution Time (ms)",
            AJAX_ICPI : "Invocation Count Per Interval",
            AJAX_NSHPI : "Non-Server Hits Per Interval"
        },
        dollarChar : "$",
        equalChar : "=",
        forwardSlashChar : "/",
        // BrowserAgent provides capability to instrument any JS functions as long as it is in the
        // scope of the current window. The JS functions that need to be instrumented are
        // added here automatically with BrowserAgentUtils.funcUtils.addFuncToGlobalsInstrumentMap.
        // The format is "key" : {name : "Name of the function"}
        // Note: If the JS Function to be instrumented is a member function inside a JS object,
        //       you may need to add the keyword prototype
        // Example 1 - "Math_Random" : { name : "Math.random" }
        // Example 2 - "XHR_SRH" : { name : "XMLHttpRequest.prototype.setRequestHeader" }
        // See BrowserAgentUtils.funcUtils.addFuncToGlobalsInstrumentMap for more details.
        functionsToInstrument : {},
        // Client server gap time in milliseconds
        gapTimeInMillis : 0,
        geo : {
            lat : -255,
            long : -255
        },
        openParenChar : "(",
        // Metric aggregator types are synonymous with Agent accumulator
        // types such as LONG INTERVAL COUNTER, AVERAGE PER INTERVAL and so on
        metricAggregatorType : {
            INT_LONG_DURATION : 0,
            LONG_INTERVAL_COUNTER : 1
        },
        // Metric Path MACROS
        metricPath : {
            BROWSER : "Browser",
            AJAX : "AJAX Call",
            FUNC : "JavaScript Function",
            URL : "URL"
        },
        // Since BrowserAgent instruments JS Functions within the current window's scope,
        // the original JS Functions are maintained in this map
        // See BrowserAgentUtils.funcUtils.saveOrigObj for more details
        // Metric types are stored here
        // See BrowserAgentUtils.metricUtils.addMetricType for more details
        metricType : {},
        // This map provides mapping between metric types and their respective accumulators
        // For example, metric type of "AJAX" has its own accumulator to store its data points
        //              metric type of "Page" has its own accumulator to store its data points
        // Note: Each accumulator is a JS object
        metricTypeToAccumulatorMap : {},
        // Just as each metric type has its own accumulator,
        // Each metric type has its own harvest function to reap and send metrics using
        // the stored data points
        metricTypeToHarvestMap : {},
        origFuncMap : {},
        p : "-1",
        pv : "-1",
        // Timestamp MACROS
        timestampNames : {
            START_TIME : "s",
            END_TIME : "e",
            CALLBACK_START_TIME : "cs",
            CALLBACK_END_TIME : "ce",
            FIRST_BYTE : "f",
            LAST_BYTE : "l",
            EXTERNAL : "ex"
        },
        pipeChar : "|",
        postData : "",
        semiColonChar : ";",
        // The current page Txn Trace start time
        startTime : null,
        UNDEFINED : "-1",
        urlChar : "url",
        userAgents : { UNSUPPORTED : { name : "Unsupported", ver : -1 } }
    };
    /**
     * This object is responsible for BrowserAgent page initialization and page tasks
     */
    var BROWSERAGENT = {
        bts : 0,
        cookie : null,
        document : document,
        dts : 0,
        ets : 0,
        method : "",
        navTiming : null,
        sts : 0,
        tBRT : -1,
        tConn : -1,
        tDNS : -1,
        tDomLoading : -1,
        tDomReady : -1,
        tFirstByte : -1,
        tLastByte : -1,
        tOnLoad : -1,
        tRTT : -1,
        tUnload : -1,
        window : window,
        init : function () {
            // If page is excluded, no page, ajax or js function instrumentation is needed
            if ( BrowserAgentUtils.configUtils.isPageExcluded ) {
                return;
            }
            // If page load metrics are disabled, no need to instrument page load metrics
            if ( !window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_PAGELOADMETRICSENABLED] ) {
                BrowserAgentLogger.info("Skipping page instrumentation because Page Load Metrics are DISABLED");
                return;
            }
            // Check for Navigation Timing API
            BROWSERAGENT.navTiming = this.window.performance || this.window.msPerformance ||
                                     this.window.webkitPerformance || this.window.mozPerformance;
            // If Navigation Timing API is present, then obtain
            // page load metrics from the performance.timing object
            if ( BROWSERAGENT.navTiming && BROWSERAGENT.navTiming.timing ) {
                BrowserAgentLogger.info("Navigation Timing API is present.");
                BrowserAgentLogger.info("Attaching to onload event...");
                this.attachEvent(this.window, "load", function () {
                    BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL, {
                        "bts" : BROWSERAGENT.navTiming.timing.navigationStart,
                        "sts" : BROWSERAGENT.navTiming.timing.unloadEventEnd
                    });
                    BROWSERAGENT.dts = BROWSERAGENT.navTiming.timing.domContentLoadedEventStart;
                    BROWSERAGENT.ets = BROWSERAGENT.navTiming.timing.loadEventStart;
                    BROWSERAGENT.onload();
                }, false);
            } else {
                // If Navigation Timing API is not present, then obtain
                // page load metrics by attaching to window events such as onbeforeunload,
                // onunload, onload and document.onreadystatechange
                BrowserAgentLogger.info("Navigation Timing API is NOT present. Resorting to alternatives...");
                document.onreadystatechange = function () {
                    if ( document.readyState == "complete" ) {
                        BROWSERAGENT.dts = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
                    }
                };
                // Event triggered when user decides and has acted to leave
                // the page
                BrowserAgentLogger.info("Attaching to onbeforeunload event...");
                this.attachEvent(this.window, "beforeunload", BROWSERAGENT.beforeunload, true);
                // Set the earlier sent cookie to Agent to false.
                // The value in the cookie should be set to false as soon as
                // possible because the cookie might prevent other requests in
                // which the snippet should indeed be inserted.
                //
                // This is specifically for browsers (e.g. Mobile Safari)
                // that do not support onbeforeunload event.
                // Use Pagehide event instead.
                BrowserAgentLogger.info("Attaching to pagehide event...");
                window.addEventListener("pagehide", BROWSERAGENT.pagehide, false);

                // Event triggered when DOM is unloaded from the window
                // frame
                BrowserAgentLogger.info("Attaching to onunload event...");
                this.attachEvent(this.window, "unload", BROWSERAGENT.unload, false);

                // Event triggered when onload
                BrowserAgentLogger.info("Attaching to onload event...");
                this.attachEvent(this.window, "load", BROWSERAGENT.onload, true);
            }
            // EXTENSION POINT for custom init
            if ( BrowserAgentExtension ) {
                BrowserAgentExtension.init();
            }
        },
        /**
         * Attaches an event to the given element in the browser
         * @param element - DOM element on which the event is to be attached (e.g. document)
         * @param type : event - name of the event to attach to (e.g. "load")
         * @param expression - callback function to be invoked upon the event
         * @param bubbling
         * @returns {boolean}
         */
        attachEvent : function ( element, type, expression, bubbling ) {
            bubbling = bubbling || false;
            if ( this.window.addEventListener ) {
                element.addEventListener(type, expression, bubbling);
                return true;
            } else if ( this.window.attachEvent ) {
                if ( type == "DOMContentLoaded" ) {
                    this.ieContentLoaded(this.window, expression);
                } else {
                    if ( element.attachEvent("on" + type, expression) ) {
                        return true;
                    } else {
                        BrowserAgentLogger.warn("attachEvent failed");
                    }
                }
                return true;
            } else {
                return false;
            }
        },
        /**
         * Callback Function for window.onbeforeunload event
         * Note: Not all browsers support the onbeforeunload event.
         *       Mobile Safari does not.
         */
        beforeunload : function () {
            BrowserAgentLogger.info("Detected 'onbeforeunload' event...");
            BROWSERAGENT.bts = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
            // For browsers that do not support the Navigation Timing API, use the onbeforeunload
            // event time as the Start Time of the current page.
            // For browsers that do not support the onbeforeunlod event, see unload callback above.
            BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL,
                { "bts" : BROWSERAGENT.bts });
        },
        pagehide : function () {
            BrowserAgentLogger.info("Detected pagehide event...");
            // For some browsers that neither support the Navigation Timing API nor the onbeforeunload
            // event, use pagehide event time as the Start Time of the current page
            if ( BROWSERAGENT.bts === 0 ) {
                BROWSERAGENT.bts = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
                BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL,
                    { "bts" : BROWSERAGENT.bts });
            }
            BROWSERAGENT.sts = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
            BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL,
                { "sts" : BROWSERAGENT.sts });
        },
        ieContentLoaded : function ( w, fn ) {
            var d = w.document,
                done = false,
                init = function () {
                    if ( !done ) {
                        done = true;
                        fn();
                    }
                };
            (function () {
                try {
                    d.documentElement.doScroll('left');
                } catch ( e ) {
                    setTimeout(arguments.callee, 50);
                    return;
                }
                init();
            })();
            d.onreadystatechange = function () {
                if ( d.readyState === 'complete' ) {
                    d.onreadystatechange = null;
                    init();
                }
            };
        },
        /**
         *  Callback Function for the window.onload event
         */
        onload : function () {
            BrowserAgentLogger.info("Detected 'onload' event...");
            // Calculate the BrowserAgent Page Load metrics with the use of the performance.timing object
            // Note: This object is available only in browsers that support the navigation
            // timing API
            BROWSERAGENT.ets = BROWSERAGENT.ets || BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
            BROWSERAGENT.cookie = BrowserAgentUtils.cookieUtils.getCookieObject();
            if ( BROWSERAGENT.navTiming && BROWSERAGENT.navTiming.timing ) {
                BROWSERAGENT.tDomLoading = BROWSERAGENT.navTiming.timing.domLoading -
                                           BROWSERAGENT.navTiming.timing.navigationStart;
                BROWSERAGENT.tRTT = BROWSERAGENT.navTiming.timing.responseEnd -
                                    BROWSERAGENT.navTiming.timing.requestStart;
                BROWSERAGENT.tDNS = BROWSERAGENT.navTiming.timing.domainLookupEnd -
                                    BROWSERAGENT.navTiming.timing.domainLookupStart;
                BROWSERAGENT.tConn = BROWSERAGENT.navTiming.timing.connectEnd -
                                     BROWSERAGENT.navTiming.timing.connectStart;
                BROWSERAGENT.tFirstByte = BROWSERAGENT.navTiming.timing.responseStart -
                                          BROWSERAGENT.navTiming.timing.navigationStart;
                BROWSERAGENT.tLastByte = BROWSERAGENT.navTiming.timing.responseEnd -
                                         BROWSERAGENT.navTiming.timing.navigationStart;
            } else {
                if ( (BROWSERAGENT.cookie) && (BROWSERAGENT.dts > 0) && (BROWSERAGENT.cookie.bts > 0) &&
                     ((BROWSERAGENT.dts - BROWSERAGENT.cookie.bts) > 0) ) {
                    BROWSERAGENT.tDomLoading = BROWSERAGENT.dts - BROWSERAGENT.cookie.bts;
                }
            }
            // Determine if this is a page reload or a first time visit
            if ( BROWSERAGENT.document.referrer.indexOf(BROWSERAGENT.document.domain) > -1 &&
                 (BROWSERAGENT.cookie) ) {
                // From same site
                BROWSERAGENT.tUnload = 0;
                if ( BROWSERAGENT.cookie.sts && BROWSERAGENT.cookie.bts ) {
                    BROWSERAGENT.tUnload = BROWSERAGENT.cookie.sts - BROWSERAGENT.cookie.bts;
                }
                BROWSERAGENT.tDomReady = 0;
                if ( BROWSERAGENT.dts != 0 && BROWSERAGENT.cookie.bts ) {
                    BROWSERAGENT.tDomReady = BROWSERAGENT.dts - BROWSERAGENT.cookie.bts;
                }
                BROWSERAGENT.tOnLoad = 0;
                if ( BROWSERAGENT.cookie.bts ) {
                    BROWSERAGENT.tOnLoad = BROWSERAGENT.ets - BROWSERAGENT.cookie.bts;
                }
                BROWSERAGENT.method = "beforeunload";
            } else if ( BROWSERAGENT.cookie && BROWSERAGENT.cookie.bts > 0 &&
                        BROWSERAGENT.document.referrer.indexOf(BROWSERAGENT.document.domain) == -1 ) {
                BROWSERAGENT.tUnload = 0;
                if ( BROWSERAGENT.cookie.bts && BROWSERAGENT.cookie.sts && BROWSERAGENT.cookie.sts > 0 ) {
                    BROWSERAGENT.tUnload = BROWSERAGENT.cookie.sts - BROWSERAGENT.cookie.bts;
                }
                BROWSERAGENT.tDomReady = 0;
                if ( BROWSERAGENT.dts != 0 && BROWSERAGENT.cookie.bts ) {
                    BROWSERAGENT.tDomReady = BROWSERAGENT.dts - BROWSERAGENT.cookie.bts;
                }
                BROWSERAGENT.tOnLoad = 0;
                if ( BROWSERAGENT.cookie.bts ) {
                    BROWSERAGENT.tOnLoad = BROWSERAGENT.ets - BROWSERAGENT.cookie.bts;
                }
                BROWSERAGENT.method = "initial";
            }
            // Dispatch the Page Load metrics
            if ( BROWSERAGENT.method ) {
                BROWSERAGENT.tBRT = BROWSERAGENT.tOnLoad - BROWSERAGENT.tDomLoading;
                var pagePOSTData = BrowserAgentUtils.metricUtils.harvestPageMetrics();
                if ( pagePOSTData.length !== 0 ) {
                    pagePOSTData = "bCount=" + BrowserAgentGlobals.bCount + pagePOSTData;
                    BrowserAgentUtils.metricUtils.sendMetrics(
                        window[BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_WILYURL],
                        pagePOSTData);
                }
            } else {
                BrowserAgentLogger.warn("no method determined");
            }
            // Reset
            BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL,
                { "sts" : null, "bts" : null, "dts" : null });
            BROWSERAGENT.sts = 0;
            BROWSERAGENT.bts = 0;
            BROWSERAGENT.dts = 0;
        },
        /**
         * Callback Function for window.unload event
         */
        unload : function () {
            BrowserAgentLogger.info("Detected 'onunload' event...");
            // For browsers that do neither support the Navigation Timing API not the onbefoeunload
            // event, use onunload event time as the Start Time of the current page
            if ( BROWSERAGENT.bts === 0 ) {
                BROWSERAGENT.bts = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
                BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL,
                    { "bts" : BROWSERAGENT.bts });
            }
            BROWSERAGENT.sts = BrowserAgentUtils.timeUtils.getCurrTimeInMillisSinceEpoch();
            BrowserAgentUtils.cookieUtils.updateCookie(BrowserAgentUtils.cookieUtils.cookies.GLOBAL,
                { "sts" : BROWSERAGENT.sts });
        }
    };
    (function () { // Initialize BrowserAgent only if it is enabled.
        if ( BrowserAgentUtils.configUtils.getConfig(BrowserAgentUtils.configUtils.configNames.BROWSERAGENT_ENABLED) ) {
            var p = BrowserAgentUtils.cookieUtils.getRawCookie(BrowserAgentUtils.cookieUtils.cookies.PLATFORM);
            if ( p && p != BrowserAgentGlobals.userAgents.UNSUPPORTED.name ) {
                BrowserAgentUtils.init();
                BROWSERAGENT.init();
            } else { // If platform cookie doesn't exist, parse user agent and set cookies
                if ( BrowserAgentSnippet ) {
                    var browserInfo = BrowserAgentSnippet.getBrowserInfo(navigator.userAgent);
                    document.cookie =
                        BrowserAgentUtils.cookieUtils.cookies.PLATFORM + "=" + browserInfo.name + "; path=/";
                    document.cookie =
                        BrowserAgentUtils.cookieUtils.cookies.PLATFORMVER + "=" + browserInfo.ver + "; path=/";
                    if ( browserInfo.isSupported ) {
                        BrowserAgentUtils.init();
                        BROWSERAGENT.init();
                    } else {
                        BrowserAgentLogger.warn("Unsupported browser. Skipping all instrumentation");
                    }
                } else {
                    BrowserAgentLogger.warn("Cannot get BrowserAgentSnippet. Skipping all instrumentation");
                }
            }
        }
    })();
} catch ( e ) {
    if ( window.console && typeof window.console == "object" ) {
        window.console.log(e.message);
    }
}
