#!/usr/bin/env bash

BASE_PATH="$( cd "$(dirname "$0")" ; pwd -P )"
cd "${BASE_PATH}"

MAIN_CLASS=com.ca.apm.gaps.GapsMain
PID_PATH=".monitoring-gaps.pid"
LOG_PATH="logs/monitoring-gaps.log"
WRAPPER_LOG_PATH="logs/monitoring-gaps-wrapper.log"
SYSCTL_SVC_NAME="monitoring-gaps.service"
SYSCTL_SVC_PATH="/usr/lib/systemd/system/${SYSCTL_SVC_NAME}"

JCMD="$(which jcmd 2> /dev/null)"
if [ -z "$JCMD" ]; then
  if [ ! -z "$JAVA_HOME" ]; then
      JCMD="$JAVA_HOME/bin/jcmd"
  fi
fi

if [ -f ./default_env.sh ] ; then 
  . ./default_env.sh ; 
fi 

jvmParamsSetup(){
    APM_JAVA_OPTS=""

    if [ -z "${APM_HEAP_XMS}" ]; then
        APM_HEAP_XMS="4096"
    fi

    if [ -z "${APM_HEAP_XMX}" ]; then
        APM_HEAP_XMX="4096"
    fi

    if [ -z "${APM_HEAP_SETTINGS}" ]; then
        APM_HEAP_SETTINGS="-Xms${APM_HEAP_XMS}m -Xmx${APM_HEAP_XMX}m -Xss512k -XX:+UseG1GC -XX:ParallelGCThreads=4 -XX:ConcGCThreads=2 -XX:+PrintGC"
    fi

    if [ -n "${APM_DEBUG_ENABLED}" ]; then
        debug="-Xdebug -Xrunjdwp:server=y,transport=dt_socket,address=9009,suspend=n"
        if [ -n "${APM_DEBUG_CUSTOM}" ]; then
            debug="${APM_DEBUG_CUSTOM}"
        fi
    fi

    APM_FIPS_BOOTCLASSPATH="-Xbootclasspath/a:$(find libs -name bc-fips\*jar)"

    if [ x"${APM_FIPS_ENABLE}" = xtrue ]; then
        APM_FIPS_SETTINGS="-Dcom.ca.apm.common.crypto.fipsEnable=true"
        APM_FIPS_SETTINGS+=" -Dorg.bouncycastle.fips.approved_only=true"
        APM_FIPS_SETTINGS+=" -Dorg.bouncycastle.jca.enable_jks=true"
        APM_FIPS_SETTINGS+=" -Djavax.net.ssl.trustStoreType=FIPS"
        APM_RANDOM_SOURCE="file:/dev/random"
        APM_FIPS_BOOTCLASSPATH+=":$(find libs -iname bctls-fips\*jar)"
    fi

    APM_RANDOM_SOURCE=${APM_RANDOM_SOURCE:-"file:/dev/./urandom"}

    APM_JAVA_OPTS="
     ${APM_FIPS_BOOTCLASSPATH}
     ${APM_HEAP_SETTINGS}
     "-Djava.security.egd=${APM_RANDOM_SOURCE}"
     ${APM_FIPS_SETTINGS}
     ${APM_JAVA_OPTS_ADDITIONAL}
     ${debug}"

     export APM_JAVA_OPTS;
}

prepare_jemalloc() {
    if [ -z "${APM_JEMALLOC}" ]; then
        APM_JEMALLOC="abort_conf:true,narenas:16,lg_tcache_max:13"
    fi 
    
    if [ -z "${APM_JEMALLOC_DISABLE}" ]; then
        export LD_PRELOAD=/usr/lib/libjemalloc.so.2
        export MALLOC_CONF=${APM_JEMALLOC}
    fi 
}

prepare_env() {
    jvmParamsSetup

    mkdir -p logs
    export CLASS_PATH=`( find debuglibs libs ../common/libs -iname "spring-*.jar" ; find debuglibs libs ../common/libs -iname "*.jar" -and -not -iname "spring-*.jar" ) 2>/dev/null | tr '\n' ':'`
    export APP_ARGS="--spring.config.location=./config/"
}

wait_up() {
    for retry in {1..30};do
        check_pid
        if [ "$?" == "1" ];then
            return 1
        fi
        grep 'APM Monitoring Gaps Service started.' ${WRAPPER_LOG_PATH} >/dev/null 2>&1
        if [ "$?" == "0" ];then
            return 0
        fi
        sleep 1
    done
    return 1
}

start_java() {
    prepare_env;
    nohup java ${APM_JAVA_OPTS} -cp ${CLASS_PATH} ${MAIN_CLASS} ${APP_ARGS} \
        0<&- 1>/dev/null 2>${WRAPPER_LOG_PATH} &
    pid=$!
    echo ${pid} > "${PID_PATH}" 
    echo "monitoring-gaps waiting to be up: PID ${pid}"
    wait_up
    is_up=$?
    if [ "${is_up}" == "0" ];then
        echo "monitoring-gaps started: PID ${pid}"
        return 0
    else
        echo "monitoring-gaps start failed. Please check the logs."
        return 1
    fi
}

run_java() {
    prepare_env;    
    exec java ${APM_JAVA_OPTS} -cp ${CLASS_PATH} ${MAIN_CLASS} ${APP_ARGS}
}

check_pid() {
    if [ -f ${PID_PATH} ];then
        PID=`cat ${PID_PATH}`
        if [ -z "$JCMD" ]; then
            PIDCMD=`ps -p ${PID} -o args= | grep ' com.ca.apm.gaps.GapsMain '`
        else
            PIDCMD=`$JCMD | grep 'com.ca.apm.gaps.GapsMain'`
        fi
        if [ "x${PIDCMD}" == "x" ];then
            rm $PID_PATH
            return 1
        fi
        return 0
    fi
    return 1
}

start() {
    if [ -f ${PID_PATH} ];then
        PID=`cat ${PID_PATH}`
        echo "monitoring-gaps already running: PID ${PID}";
        exit 128;
    else
        start_java;
        return $?
    fi
}

run() {
    if [ -f ${PID_PATH} ];then
        PID=`cat ${PID_PATH}`
        echo "monitoring-gaps already running: PID ${PID}";
        exit 128;
    else
        run_java;
    fi
}

stop() {
    if [ -f ${PID_PATH} ];then
        PID=`cat ${PID_PATH}`;
        kill $PID;
        rm $PID_PATH;
        echo "monitoring-gaps stopped: PID ${PID}";
        return 0
    else
        echo "monitoring-gaps not running";
        return 0
    fi
}

status() {
    if [ -f ${PID_PATH} ];then
        PID=`cat ${PID_PATH}`;
        echo "monitoring-gaps running: PID ${PID}";
        return 0
    else
        echo "monitoring-gaps not running";
        return 0
    fi
}

logs() {
    tail -F ${LOG_PATH} 
}

version() {
    echo "2022.11.0.11"
}

install() {
    if [ "${EUID}" -ne 0 ];then
        echo "install requires root user."
        exit 1
    fi
    echo "Installing ${SYSCTL_SVC_NAME} to ${SYSCTL_SVC_PATH}"
    sed 's|<base_path>|'${BASE_PATH}'|g' ${BASE_PATH}/${SYSCTL_SVC_NAME} > ${SYSCTL_SVC_PATH}
    chown root:root ${SYSCTL_SVC_PATH}
    chmod 644 ${SYSCTL_SVC_PATH}
    echo "Reloading systemctl daemon"
    systemctl daemon-reload
    echo "Enabling service ${SYSCTL_SVC_NAME}"
    systemctl enable ${SYSCTL_SVC_NAME}
    echo "Starting service ${SYSCTL_SVC_NAME}"
    systemctl start ${SYSCTL_SVC_NAME}
}

uninstall() {
    if [ "${EUID}" -ne 0 ];then
        echo "install requires root user."
        exit 1
    fi
    if [ -f ${SYSCTL_SVC_PATH} ];then
        echo "Stopping  ${SYSCTL_SVC_NAME} service"
        systemctl stop ${SYSCTL_SVC_NAME}
        echo "Disabling ${SYSCTL_SVC_NAME} service"
        systemctl disable ${SYSCTL_SVC_NAME}
        echo "Removing ${SYSCTL_SVC_NAME} from ${SYSCTL_SVC_PATH}"
        rm ${SYSCTL_SVC_PATH}
        echo "Reloading systemctl daemon"
        systemctl daemon-reload
        return 0
    else
        echo "${SYSCTL_SVC_PATH} not found. Uninstall skipped."
        return 1
    fi
}

check_pid 

case "$1" in
    start)
        start
        ;;
    run)
        run
        ;;
    stop)
        stop
        ;;
    status)
        status
        ;;
    logs)
        logs
        ;;
    restart)
        stop
        start
        ;;
    version)
        version
        ;;
    install)
        install
        ;;
    uninstall)
        uninstall
        ;;
    *)
        echo "monitoring-gaps $(version)";
        echo "Usage:  {start|run|stop|restart|status|logs|version|install|uninstall}"
        exit 1
        ;;
esac
exit $?
