/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j


@Slf4j
class GlassFishInfo {
    def versionFinder(File jarFile) {
        log.debug("Find GlassFish version string")

        if (!jarFile.isFile()) {
            log.error("$jarFile IS MISSING")
        }

        log.debug("jarFile: $jarFile")

        def pb_url = jarFile.toURI().toURL()

        log.debug("Adding: $pb_url")

        def loader = this.class.classLoader.rootLoader

        log.debug("Root loader is: $loader")

        if (loader == null) {
            log.debug("No root loader found, taking this class classloader")
            loader = this.class.getClassLoader()
        }

        loader.addURL(pb_url)

        def clsname = "com.sun.appserv.server.util.Version"

        log.debug("Finding class: $clsname")

        def cls = Class.forName(clsname)

        log.debug("Instantiate class")

        def clsinst = cls.newInstance()

        log.debug("Get properties object")
        def p = clsinst.getProperties()

        log.debug("Extract information from returned properties:")
        log.debug(p.toString())

        // Take the "fullVersion" in preference to attempting to reconstruct the version string from the individual elements,
        // but strip out the product name.
        def vstr = p.fullVersion.minus(p.productName).trim()

        def ret = [p.productName, vstr]

        return ret
    }


    def tryPath(File commonUtilJar) {

        if (!commonUtilJar.exists()) {
            log.debug("$commonUtilJar not found")
        } else {
            log.debug("Found $commonUtilJar")

            def ver = versionFinder(commonUtilJar)
            log.debug("Discovered GlassFish version from $commonUtilJar as: $ver")

            return ver
        }

        return null
    }


    def handler(request) {
        log.debug("GlassFish discovery script invoked")
        // Default response if we do not find any signs of WebSphere.
        def json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: []
        )
        def response = json.toString()

        // Parse JSON Request
        def inputs = new JsonSlurper().parseText(request)

        // Set this property as version glassfish v4 needs it set
        def installRoot = inputs.get("com.sun.aas.installRoot")

        if (installRoot == null || installRoot.isEmpty()) {
            log.debug("com.sun.aas.installRoot is not set: no glassfish")
            return response
        }

        // At this point we know we a glassfish installation of some description

        log.info("GlassFish discovered at {}", installRoot)

        def name = "Sun Java System Application Server / GlassFish"
        def version = "Unknown Version"

        installRoot = new File(installRoot)

        if (!installRoot.exists()) {
            log.error("Glassfish installroot $installRoot does not exist")
        } else {
            log.debug("com.sun.aas.installRoot is $installRoot")
            System.setProperty("com.sun.aas.installRoot", installRoot.toString())

            // Jars to search for under the glassfish installation
            def paths = ["lib/appserv-ext.jar", "modules/common-util.jar"]

            for (path in paths) {

                def ver = tryPath(new File(installRoot, path))

                if (ver != null) {
                    log.info("Discovered GlassFish version is: $ver")

                    // Success
                    name = ver[0]
                    version = ver[1]
                    break
                }
            }
        }

        // Create the json message
        json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: [
                        "acc.agent.appserver.name"   : name,
                        "acc.agent.appserver.version": version
                ]
        )
        response = json.toString()
        return response
    }
}

response = new GlassFishInfo().handler(request)
