/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j

@Slf4j
class WeblogicInfo {
    /**
     *  if s is null then return null, otherwise if s starts with a slash or backslash then return empty string otherwise slash
     */
    static def absolutePath(s) {
        s ? (s ==~ /(\/|\\).*/ ? '' : '/') : null
    }

    /**
     * parse App Server name from tokenized version string.
     */
    static def asName(tokens) {
        if (tokens.size() >= 2) {
            return tokens[0..1].join(' ')
        } else {
            return ""
        }
    }

    /**
     * parse App Server version number from tokenized version string
     */
    static def asVersion(tokens) {
        if (tokens.size() >= 10) {
            return tokens[2..tokens.size() - 8].join(' ')
        } else if (tokens.size() >= 3) {
            return tokens[2]
        } else {
            return ""
        }
    }

    /**
     * parse App Server built date from tokenized version string
     */
    static def asBuilt(tokens) {
        if (tokens.size() >= 10) {
            return tokens[tokens.size() - 7..tokens.size() - 1].join(' ')
        } else {
            return ""
        }
    }

    /**
     * extract path to weblogic JAR from string using given path separator.
     * 1) Prepend path separator so that find succeeds when required path at start of string
     * 2) Try to match from separator to jar
     * 3) Return NULL if not found
     */
    static def extractPathToJar(cp, sep) {
        def pattern = "$sep[^$sep]+weblogic.jar"
        def found = (sep + cp).find(pattern)

        if (found != null) {
            found = found.substring(1, found.length())
        }
        return found
    }
    
    /**
     * make a JSON structure containing the Response message and return as a String
     */
    static def makeResponse(errs = [], n = null, v = null, b = null) {
        def json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: [
                        "acc.agent.appserver.name"   : n,
                        "acc.agent.appserver.version": v,
                        "acc.agent.appserver.built"  : b
                ],
                errors: errs
        )
        return json.toString()
    }

    def handler(request) {
        // Default response if we do not find any signs of WebSphere.
        def json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: []
        )
        def response = json.toString()

        // Parse JSON Request message
        def inputs = new JsonSlurper().parseText(request)
        // Get required properties
        def path = inputs["java.class.path"]
        def sep = inputs["path.separator"]

        if (path != null) {

            def arg = extractPathToJar(path, sep)

            if (arg != null) {
                def abs = absolutePath(arg)
                arg = "file://$abs$arg"
                log.info("Weblogic discovered at {}", arg)

                // Load and instantiate weblogic.version class from weblogic.jar and extract version information
                try {
                    this.class.classLoader.addURL(new URL(arg))
                    def obj = Class.forName("weblogic.version").newInstance()
                    def methodName1 = 'getVersions'

                    // Split version string into tokens separated by whitespace:
                    //    Assume first 2 tokens contain name of App Server
                    //    Assume last 7 tokens are the build date
                    //    Assume remaining tokens inbetween are the version number (in some cases, it spans multiple tokens and contains
                    //        letters as well as digits
                    // Functions concatenate tokens back together to form a string using "join"
                    def weblogicVersionString = obj."$methodName1"()
                    def tokens = weblogicVersionString.split()
                    log.debug("Weblogic Version = {}", weblogicVersionString)
                    response = makeResponse([], asName(tokens), asVersion(tokens), asBuilt(tokens))
                } catch (Exception e) {
                    log.error("Error in loading or accessing weblogic.version class", e)
                    def errors = []
                    def anErr = ["errno": "EAPPSERVER", "arguments": ["WebLogic", e.toString()]]
                    errors.add(anErr)
                    response = makeResponse(errors)
                }
            }
        }

        return response
    }
}

response = new WeblogicInfo().handler(request)
