/*
 * Copyright (c) 2025 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.json.JsonBuilder
import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j

import javax.net.ssl.HostnameVerifier
import javax.net.ssl.HttpsURLConnection
import javax.net.ssl.SSLContext
import javax.net.ssl.X509TrustManager
import java.nio.file.*
import java.nio.file.attribute.BasicFileAttributes
import java.nio.file.attribute.PosixFilePermission
import java.nio.file.attribute.UserPrincipal
import java.util.function.Function
import java.util.function.Predicate
import java.util.regex.Matcher
import java.util.stream.Collectors
import java.util.zip.ZipEntry
import java.util.zip.ZipFile

@Slf4j
@CompileStatic
class ACCException extends Exception {
    String errno
    String[] arguments

    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }

    ACCException(String msg, String errno, Throwable t) {
        super((msg == null) ? "<null msg>" : msg, t)
        this.errno = errno
        log.info("ACCException: $msg")
    }

    void setParams(String ... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }

    def getAccError() {
        return ["errno":this.errno, "arguments":arguments]
    }
}

@CompileStatic
class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

@CompileStatic
class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom.toAbsolutePath().normalize().toString())
    }
}

@CompileStatic
class ACCWriteAccessException extends ACCException {

    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

@CompileStatic
class ACCNoEntityException extends ACCException {

    ACCNoEntityException(String msg, enitity) {
        super(msg, "ENOENT")
        setParams(enitity.toString())
    }
}

@CompileStatic
class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

@CompileStatic
class ACCFrameworkException extends ACCException {
    ACCFrameworkException(String msg, String info) {
        super(msg, "EFRAMEWORKERROR")
        setParams(info)
    }
}

@Slf4j
@CompileStatic
class FilePuller  {

    void httpGet(URL url, Path to, String token, boolean trustSelfSigned, boolean useHttp) {
        // Write the file to a temporary file first
        Path temp = Paths.get(to.toString() + ".tmp")

        InputStream is
        try {
            URLConnection urlConnection = url.openConnection()
            if (urlConnection instanceof HttpURLConnection) {
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "text/plain")
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "application/octet-stream")
            }

            if (useHttp) {
                is = urlConnection.getInputStream()
            } else if (urlConnection instanceof HttpsURLConnection) {
                HttpsURLConnection connection = (HttpsURLConnection)urlConnection
                connection.setRequestProperty("Authorization", "Bearer " + token)
                if(trustSelfSigned)
                {
                    def nullTrustManager = [
                        checkClientTrusted: { chain, authType ->  },
                        checkServerTrusted: { chain, authType ->  },
                        getAcceptedIssuers: { null }
                    ]

                    SSLContext sc = SSLContext.getInstance("SSL")
                    sc.init(null, [nullTrustManager as  X509TrustManager] as  X509TrustManager[], null)
                    connection.setSSLSocketFactory(sc.getSocketFactory())

                    def nullHostnameVerifier = [
                        verify: { hostname, session -> true }
                    ]
                    connection.setHostnameVerifier(nullHostnameVerifier as HostnameVerifier)
                }
                is = connection.getInputStream()
            } else {
                ((HttpURLConnection) urlConnection).setRequestProperty("Authorization", "Bearer " + token)
                is = urlConnection.getInputStream()
            }
        }
        catch (IOException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }

        try {
            Files.deleteIfExists(temp)
            Files.copy(is, temp)
            Files.move(temp, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
        }
        catch (NoSuchFileException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
        finally {
            try {
                Files.deleteIfExists(temp)
            }
            catch (Exception ignore) {
                log.info("Failed to delete $temp")
            }

            is.close()
        }
    }

    void copyBrokerFile(URL broker, String from, Path to, String token, boolean trustSelfSigned, boolean useHttp) {

        URL url = new URL(broker.toString() + from)

        log.debug("copyBrokerFile: $url -> $to")
        httpGet(url, to, token, trustSelfSigned, useHttp)
    }
}

@CompileStatic
enum CommandType{
    RESTART,
    RESTART_WITH_ACC
}

@CompileStatic
class EventLog{
    String message;
    long timeStamp;
    EventLog(String message, long timeStamp) {
        this.message = message
        this.timeStamp = timeStamp
    }
}

@CompileStatic
class CommandResponse{
    public String response
    public int exitCode

    public CommandResponse(int exitCode, String response){
        this.exitCode = exitCode
        this.response = response
    }
}


@Slf4j
@CompileStatic
class ApplyApmiaBSPackage {
    public static final String TMP_DIRNAME = "tmp"
    public static final String PACKAGETMP_FILENAME = "package.tmp"
    public static final String APM_RELEASES = "releases"
    public static final String ACC_CONTROLLER = "acc-controller"
    public static final String PROFILE_PATH = "core"+File.separator+"config"
    public static final String RESTART_MSG = "Manual restart of the APM Infrastructure Agent is required. " +
        "The Agent Controller bundle is present within the same Agent instance "
    public static final String ENV_ACC_AGENT_MERGE_ENABLED = "ACC_AGENT_MERGE_ENABLED"
    List<EventLog> eventLog = []
    public static final String EXTENSIONS_PATH = "extensions"
    public static final String DEPLOY_PATH = EXTENSIONS_PATH + File.separator + "deploy"
    public static final String AGENT_RELEASE_CONF = "agent-release.conf"
    def requestJson
    URL broker
    String partialUrl, authToken, userName
    boolean trustSelfSigned, useHttp
    Path accDir, agentDir, currentAgentDir, userDir, agentParentDir, agentTmpDir, archive
    String releaseVersion, currentReleaseVersion, releaseServerId, releasePackageId, os, agentName
    String format = "zip"
    List<String> releasesLocked = new ArrayList<>()
    Map<String, String> responseProp = new HashMap<>()

    void computeAgentReleasePath() {
        //resolve symlinks, if any.
        Path agentPath = this.agentDir.toRealPath()
        //default to parent directory of running agent.
        agentParentDir = agentPath.getParent()
    }

    void initEnv() {

        log.debug("Running on: {}", System.getProperty("os.name"))

        accDir = Paths.get(extractOptionalParam("acc.controller.dir", "."))
        log.debug("accDir: {}", accDir.toAbsolutePath().normalize())

        agentName = extractOptionalParam("introscope.agent.agentName", "{agentName}")
        log.debug("agentName: {}", agentName)

        currentAgentDir = extractDirParam("acc.agent.installpath")
        agentDir = extractDirParam("acc.agent.installpath")
        if(agentDir.toString().contains(APM_RELEASES)){
            agentDir=agentDir.getParent().getParent()
        }
        agentParentDir = agentDir.getParent()

        log.info("agentDir: {}", agentDir.toAbsolutePath().normalize())
        log.info("agentParentDir: {}", agentParentDir.toAbsolutePath().normalize())

        // Make sure we can resolve this path before we start.
        log.debug("Profile path={}", makeUserPath(extractParam("com.wily.introscope.agentProfile")));

        computeAgentReleasePath()

        releaseVersion = extractParam("acc.package.release.agent.version");
        log.info("releaseVersion: {}", releaseVersion)

        currentReleaseVersion = extractCurrentVersion()
        log.info("currentReleaseVersion: {}", currentReleaseVersion)

        releaseServerId = extractParam("acc.package.release.server.id");
        log.info("releaseServerId: {}", releaseServerId)

        releasePackageId = extractParam("acc.package.release.package.id");
        log.info("releasePackageId: {}", releasePackageId)

        for (int i = 1; ; i++) {
            final String releaseId = extractOptionalParam("acc.package.release.locked.release." + i, null);
            if (releaseId == null) {
                break;
            }
            this.releasesLocked.add(releaseId)
        }
        if (this.releasesLocked.isEmpty()) {
            log.info("No locked releases")
        } else {
            log.info("Locked releases: {}", this.releasesLocked)
        }

        partialUrl = extractParam("acc.package.url")

        if(!partialUrl.contains(format)){
            format="tar"
        }
        String b = requestJson["acc.broker.url"]
        setBroker((b != null) ? b : readBrokerUrl())

        userName = extractParam("user.name")

        authToken = extractParam("com.ca.apm.acc.controller.configurationServer.token")

        if(this.broker.getProtocol() == "https"){
            trustSelfSigned = true
            useHttp = false
        }else{
            useHttp = true
        }

        log.debug("broker: {}", this.broker)
        log.debug("partialUrl: {}", this.partialUrl)

        os = requestJson["os.name"]
        log.info("OS: {}", os)

        agentTmpDir = agentParentDir.resolve(TMP_DIRNAME)
        Files.createDirectories(agentTmpDir)
        agentTmpDir = agentTmpDir.toRealPath()

        archive = agentTmpDir.resolve(PACKAGETMP_FILENAME)
        log.debug("archive: {}", this.archive)
    }

    private String extractCurrentVersion() {
        String releaseVersion = extractParam("acc.agent.version")
        String versionPattern = /(\d+\.\d+(?:\.\d+){0,2})/

        Matcher matcher = (releaseVersion =~ versionPattern)
        if (matcher.find()) {
            ArrayList versionList = (ArrayList)matcher[0]
            releaseVersion = versionList.get(0)
        } else {
            releaseVersion = releaseVersion.split(" ")[0]
        }

        return releaseVersion
    }

    void setBroker(String b) {
        this.broker = new URL((b.endsWith("/")) ? b : b + "/" )
    }

    Path makeAccPath(String other) {
        return (other != null) ? accDir.resolve(other) : accDir
    }

    Path makeUserPath(String other) {

        if (other != null) {
            Path pathOther = Paths.get(other)
            if (pathOther.isAbsolute()) {
                return pathOther;
            }
        }

        /*
         * Need to resolve the paths against userDir.
         * Note that we only try this when the other path is relative
         * because we might not have permissions to actually read user.dir
         */
        if (userDir == null) {
            userDir = extractDirParam("user.dir")
            log.debug("userDir: {}", userDir.toAbsolutePath().normalize())
        }

        return (other != null) ? userDir.resolve(other) : userDir
    }

    String readBrokerUrl() {

        String serverUrl = "com.ca.apm.acc.controller.configurationServer.url";
        String serverUrlShort = "configurationServer.url";

        String brokerString = System.getenv(serverUrl);
        if (brokerString != null) {
            log.debug("Found " + serverUrl + " in system environment variables: $brokerString")
        } else {
            Path propertiesFilename = makeAccPath("config/apmccctrl.properties")
            FileInputStream is

            log.debug("Reading " + serverUrl + " from $propertiesFilename")
            try {
                // Get the broker URL from the properties file.
                Properties controllerProperties = new Properties()

                is = new FileInputStream(propertiesFilename.toFile())
                controllerProperties.load(is)
                brokerString = controllerProperties.getProperty(serverUrl)

                if(brokerString == null)
                    brokerString = controllerProperties.getProperty(serverUrlShort);
            }
            catch (IOException e) {
                throw new ACCReadAccessException("Failed to read config", propertiesFilename)
            }
            finally {
                if (is != null) {
                    is.close()
                }
            }
        }

        if (!brokerString.endsWith("/")) {
            brokerString += "/"
        }

        return brokerString
    }

    Path applyPackageToAgent() {

        initEnv()
        Optional<Path> existingRelease = existingRelease(this.agentDir.resolve("releases"))
        boolean isLegacyAgent = false;
        boolean hasOldTanukiWrapper = false;
        def licenseChange = false;

        if (existingRelease.isPresent()) {
            logEvent("Found existing release in " + existingRelease.get()+" so not downloading agent")
            log.info("Found existing release in {}", existingRelease.get())
            updateAgentReleaseConf()
        } else{
            final String targetdirName = getTargetdir(this.agentParentDir)
            // Pull the package file from the broker
            logEvent("Downloading Agent Package \"{packageName}:{packageVersion}\"  by Agent Controller \"{agentControllerName}\".")
            long startTime, endTime;

            startTime = getCurrentTimeMillis();
            new FilePuller().copyBrokerFile(broker, partialUrl, archive, authToken, trustSelfSigned, useHttp)
            endTime = getCurrentTimeMillis();
            log.debug("APMIA Package Download Time Taken {} ms", (endTime - startTime))
            logEvent("Successfully downloaded Agent Package \"{packageName}:{packageVersion}\"  by Agent Controller \"{agentControllerName}\"")
            isLegacyAgent = isLegacy();

            if(isLegacyAgent){
                if (!(os ==~ "Windows .*") && hasOldWrapperScript()) {
                    hasOldTanukiWrapper = true
                }
                copyFiles(agentDir, agentDir.resolve(APM_RELEASES), currentReleaseVersion)
            }

            startTime = getCurrentTimeMillis();
            extractArchive(this.agentParentDir, archive)

            endTime = getCurrentTimeMillis();

            if(!hasOldTanukiWrapper){
                licenseChange = wrapperLicenseChange();
                hasOldTanukiWrapper = licenseChange;
            }

            if(licenseChange){
                isLegacyAgent = licenseChange;
            }

            log.debug("APMIA Package Extraction Time Taken {} ms", (endTime - startTime))
            logEvent("Successfully extracted Agent Package at location \""+agentDir+"\" by Agent Controller \"{agentControllerName}\"")

            startTime = getCurrentTimeMillis();
            mergeAgentConfiguration(this.currentAgentDir,agentDir.resolve(APM_RELEASES).resolve(this.releaseVersion))
            endTime = getCurrentTimeMillis();
            log.debug("APMIA Merge Configuration Time Taken {} ms", (endTime - startTime))
            if(isLegacyAgent){
                upgradeLegacyAgentBackup(hasOldTanukiWrapper)
            }
        }

        if(licenseChange || hasOldTanukiWrapper){
            addResponseMessage(generateUninstallInstallMessage());
        } else{
            long startTime = getCurrentTimeMillis();
            restartUpdatedAgent(agentDir, isLegacyAgent)
            long endTime = getCurrentTimeMillis();
            log.debug("APMIA restart Time Taken {} ms", (endTime - startTime))

            if(isLegacyAgent){
                cleanupLegacyUpgrade()
            }
        }
    }

    private String generateUninstallInstallMessage() {
        String agentDirStr = agentDir.toString();
        StringBuffer response = new StringBuffer();
        String oldTanukiWrapperMsg = "The upgraded Infra Agent has incompatible service wrapper. Uninstall and install apmia to take effect.";

        response.append(oldTanukiWrapperMsg).append("\n");
        response.append("Execute script: ").append("\n");
        response.append("cd ").append(agentDirStr).append(" && ");
        if (os ==~ "Windows .*") {
            response.append("./APMIACtrl.bat uninstall").append(" && ");
            response.append("./APMIACtrl.bat install").append("\n");
        } else {
            response.append("./APMIACtrl.sh uninstall").append(" && ");
            response.append("./APMIACtrl.sh install").append("\n");
        }
        return response.toString();
    }

    private long getCurrentTimeMillis() {
        return System.currentTimeMillis();
    }

    private boolean hasOldWrapperScript(){
        def shConfPath = agentDir.resolve("bin").resolve("APMIAgent.shconf");
        if (!Files.exists(shConfPath)) {
            Files.copy(agentDir.resolve(getStartScript()), agentDir.resolve("APMIACtrlLegacy.sh"), StandardCopyOption.REPLACE_EXISTING);
            return true;
        }

        return false;
    }

    private boolean wrapperLicenseChange(){
        def currentWrapperLicense = agentDir.resolve("conf").resolve("wrapper-license.conf");
        def newWrapperLicense = agentDir.resolve(APM_RELEASES).resolve(releaseVersion).resolve("conf").resolve("wrapper-license.conf");
        if (os ==~ "Windows .*") {
            newWrapperLicense =  agentDir.resolve(APM_RELEASES).resolve(releaseVersion).resolve("wrapper").resolve("conf").resolve("wrapper-license.conf");
        }

        if(!Files.exists(currentWrapperLicense) || !Files.exists(newWrapperLicense)){
            return false;
        }
        def currentLicenseContent = currentWrapperLicense.text;
        def newLicenseContent = newWrapperLicense.text;
        return (currentLicenseContent != newLicenseContent);
    }

    private boolean isLegacy(){
        def releaseConfFilePath = agentDir.resolve(AGENT_RELEASE_CONF)
        def releaseConfFile = releaseConfFilePath.toFile()

        return !releaseConfFile.exists()
    }
    private void upgradeLegacyAgentBackup(boolean hasOldTanukiWrapper) {
        def target = agentDir.resolve(APM_RELEASES).resolve(currentReleaseVersion)
        def source = agentDir.resolve(APM_RELEASES).resolve(releaseVersion)

        if (!Files.exists(source)) {
            log.info("The upgrade version is not a supported version.")
            return
        }

        if (os ==~ "Windows .*") {
            copyFiles(source.resolve("wrapper").resolve("conf"), agentDir, "conf" )
            modifyWrapperConfForLegacy()
        } else {
            cleanLibDir(agentDir.resolve("lib"))
            if (hasOldTanukiWrapper) {
                Files.copy(source.resolve("lib").resolve("wrapper.jar"),
                    target.resolve("lib").resolve("wrapper.jar"), StandardCopyOption.REPLACE_EXISTING)
                Files.copy(source.resolve("lib").resolve("libwrapper.so"),
                    target.resolve("lib").resolve("libwrapper.so"), StandardCopyOption.REPLACE_EXISTING)
                Files.copy(source.resolve("wrapper"),
                    target.resolve("wrapper"), StandardCopyOption.REPLACE_EXISTING)
                Files.copy(agentDir.resolve("conf").resolve("wrapper.conf"),
                    agentDir.resolve("conf").resolve("bkpwrapper.bkp"), StandardCopyOption.REPLACE_EXISTING)
                Files.copy(source.resolve("conf").resolve("wrapper-license.conf"),
                    agentDir.resolve("conf").resolve("wrapper-license.conf"), StandardCopyOption.REPLACE_EXISTING)
                updateSeviceName(source)
            }
        }

        String startScript = getStartScript()
        copyFiles(source.resolve(getWrapperDir()), target, getWrapperDir())

        Path targetBinDir = target.resolve("bin");
        // Create the target directory if it doesn't exist
        if (Files.exists(targetBinDir)) {
            deleteDirectory(targetBinDir)
        }
        copyFiles(source.resolve("bin"), target, "bin")
        Files.copy(source.resolve(startScript), target.resolve(startScript), StandardCopyOption.REPLACE_EXISTING);

        log.info("Current APMIA Version {} is backed up.", currentReleaseVersion)

    }

    private void updateSeviceName(Path targetBase){
        def target = targetBase.resolve("bin").resolve("APMIAgent.shconf")
        def source = agentDir.resolve("bin").resolve("APMIAgent.sh")

        try {
            if (!Files.exists(target) || !Files.exists(source)) {
                return;
            }

            String propertyKey = "APP_NAME";
            String propertyValue = null
            List<String> lines = Files.readAllLines(source);
            for (String line : lines) {
                if (line.startsWith(propertyKey + "=")) {
                    propertyValue = line;
                }
            }

            if (propertyValue == null) {
                return;
            }

            // Read target file content
            List<String> targetFileContent = Files.readAllLines(target);

            // Update or add the property in the target file
            boolean propertyFound = false;
            for (int i = 0; i < targetFileContent.size(); i++) {
                String line = targetFileContent.get(i);
                if (line.startsWith(propertyKey + "=")) {
                    targetFileContent.set(i, propertyValue);
                    propertyFound = true;
                    break;
                }
            }
            if (!propertyFound) {
                targetFileContent.add(propertyValue);
            }

            // Write the updated content back to the target file
            Files.write(target, targetFileContent);
        } catch (IOException e){
            log.warn("Error while tacking backup of service name from APMIAgent.sh version {} ", currentReleaseVersion);
        }

    }

    private void modifyWrapperConfForLegacy() {
        String propertyKey = "wrapper.java.classpath.1"; // The property key to modify
        String newValue = "lib/wrapper*.jar"; // The new value for the property

        Path wrapperPath = agentDir.resolve("conf").resolve("wrapper.conf")

        try{
            List<String> lines = Files.readAllLines(wrapperPath);

            // Modify the property value
            List<String> updatedLines = lines.stream()
                .map(new Function<String, String>() {
                    @Override
                    public String apply(String line) {
                        if (line.startsWith(propertyKey + "=")) {
                            return propertyKey + "=" + newValue;
                        }
                        return line;
                    }
                })
                .collect(Collectors.toList());

            // Write the updated lines back to the file
            Files.write(wrapperPath, updatedLines, StandardOpenOption.TRUNCATE_EXISTING);
        } catch (IOException e) {
            log.warn("Error while updating wrapper.conf {} ", currentReleaseVersion);
            logEvent("Error while updating wrapper.conf " + currentReleaseVersion)
        }

    }

    private void cleanupLegacyUpgrade() {
        if (os ==~ "Windows .*") {
            return;
        }
        deleteDirectory(agentDir.resolve("extensions"))
        deleteDirectory(agentDir.resolve("jre"))
        deleteDirectory(agentDir.resolve("core"))
    }

    private void cleanLibDir(Path directory){
        try {
            Files.walkFileTree(directory, new SimpleFileVisitor<Path>() {
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (!file.getFileName().toString().startsWith("wrapper")) {
                        Files.delete(file); // Delete files not starting with "wrapper"
                    }
                    return FileVisitResult.CONTINUE;
                }
            });

        } catch (IOException e) {
            log.warn("Error while tacking backup of APMIA version {} ", currentReleaseVersion);
            logEvent("Error while tacking backup of APMIA version " + currentReleaseVersion)
        }
    }
    private void deleteDirectory(Path directoryToDelete){
        try {
            Files.walkFileTree(directoryToDelete, new SimpleFileVisitor<Path>() {
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file); // Delete files
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir); // Delete directories
                    return FileVisitResult.CONTINUE;
                }
            });

        } catch (IOException e) {
            log.warn("Error while tacking backup of APMIA version {} ", currentReleaseVersion);
            logEvent("Error while tacking backup of APMIA version " + currentReleaseVersion)
        }
    }

    private void copyFiles(Path sourceLocation, Path newLocation, String sourceToCopy) {
        Path targetDir = newLocation.resolve(sourceToCopy);
        Path sourceDir = sourceLocation;
        try {
            // Create the target directory if it doesn't exist
            if (!Files.exists(targetDir)) {
                Files.createDirectories(targetDir);
            }

            // Copy the source directory recursively to the target directory
            Files.walkFileTree(sourceDir, new SimpleFileVisitor<Path>() {
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    // Exclude the "releases" directory
                    if (dir.equals(sourceDir.resolve("releases"))) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    Path targetDirPath = targetDir.resolve(sourceDir.relativize(dir));
                    Files.createDirectories(targetDirPath);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.copy(file, targetDir.resolve(sourceDir.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            log.warn("Error while tacking backup of APMIA version {} ", currentReleaseVersion);
            log.error("Error occurred: $e")
            logEvent("Error while tacking backup of APMIA version " + currentReleaseVersion)
        }
    }

    private boolean isMergeEnabled(){
        String mergeProperty = System.getenv(ENV_ACC_AGENT_MERGE_ENABLED);
        return mergeProperty != null && !mergeProperty.isEmpty() ?
            Boolean.parseBoolean(mergeProperty) :
            Boolean.parseBoolean(extractOptionalParam("acc.agent.merge.enabled", "true"));
    }

    private void mergeAgentConfiguration(Path currentAgentPath, Path newAgentPath){
        if (!isMergeEnabled()) {
            log.info("Merge configuration is disabled.");
            logEvent("Merge configuration is disabled.")
            return;
        }
        try {
            String javaPath = currentAgentPath.resolve("jre").resolve("bin").resolve("java").toString();
            String mergeFilePath = this.agentDir.resolve("tools").resolve("MergeUtility.jar").toString();
            Path rulesFile = currentAgentPath.resolve("tools").resolve("rules.json")

            ArrayList<String> command = new ArrayList<>();
            command.add(javaPath)
            command.add("-Xmx128m")
            command.add("-jar")
            command.add(mergeFilePath)
            command.add("customerDir=" + currentAgentPath.toString())
            command.add("upgradeDir=" + newAgentPath.toString())
            command.add( "mode=extconfig")
            command.add( "from=" + this.currentReleaseVersion)
            command.add( "to=" + this.releaseVersion)

            if (Files.exists(rulesFile)){
                command.add("rule=" + rulesFile.toString())
            }
            CommandResponse response = triggerCommand(command.toArray() as String[])

            if (response.exitCode != 0) {
                log.error("Merge failed due to- {}", response.response)
                logEvent("Failed to merge agent configurations by Agent Controller \"{agentControllerName}\"")
                return
            }

            log.info("Merge operation completed - {}", response.response)
            logEvent("Successfully merged agent configurations by Agent Controller \"{agentControllerName}\"")
        } catch (Exception e) {
            log.info("Skipping File merge due to some error.", e);
        }
    }

    private void updateAgentReleaseConf() {
        def filePath = this.agentDir.resolve(AGENT_RELEASE_CONF).toString()
        def propertyName = "set.APMIA_RELEASE"
        def newValue = releaseVersion

        // Read the content of the file
        def lines = new File(filePath).readLines()
        // Find and replace the line containing the property
        List<String> updatedLines = lines.stream()
            .map(new Function<String, String>() {
                @Override
                public String apply(String line) {
                    if (line.startsWith(propertyName + "=")) {
                        return propertyName + "=" + newValue;
                    }
                    return line;
                }
            })
            .collect(Collectors.toList());

        // Write the updated lines back to the file
        new File(filePath).write(updatedLines.join('\n'))
    }

    private void restartUpdatedAgent(Path newAgentPath, boolean isLegacyAgent) {
        Path newDeployPath = newAgentPath.resolve(APM_RELEASES).resolve(this.releaseVersion).resolve(DEPLOY_PATH)
        Path currentDeployPath = currentAgentDir.resolve(DEPLOY_PATH)
        def containsAccController = containsAccController(currentDeployPath) || containsAccController(newDeployPath)
        def shouldRestart = !(isLegacyAgent && containsAccController)
        if(shouldRestart){
            String[] restartCommand = getExecutionCommand(newAgentPath, CommandType.RESTART)
            if(containsAccController){
                //Cleaning up here since in restart controller dies
                cleanUp(archive)
                restartCommand = getExecutionCommand(newAgentPath, CommandType.RESTART_WITH_ACC)
            }
            executeCommand(restartCommand)
        } else{
            addResponseMessage(RESTART_MSG)
        }
    }

    private boolean containsAccController(Path newDeployPath){
        return checkExtensionAvailability(newDeployPath, ACC_CONTROLLER)
    }

    private boolean checkExtensionAvailability(Path newDeployPath, String extensionName) {
        File directory = new File(newDeployPath.normalize().toString())
        File[] files = directory.listFiles()
        if (files != null) {
            for (File file : files) {
                if (!file.isDirectory() && file.getName().toLowerCase().contains(extensionName.toLowerCase())) {
                    log.info("{} extension is found in deploy Path {}", extensionName, file.getName())
                    return true
                }
            }
        }
        return false
    }

    String getTargetdir(Path apmiaParentPath) {
        final String targetdirPrefix = this.releaseVersion
        for (int i = 0; ; i++) {
            final String targetdir = targetdirPrefix + ((i == 0) ? "" : ("." + i));
            log.info("Checking if targetdir {} is available", targetdir)
            if (!Files.exists(apmiaParentPath.resolve(targetdir))) {
                return targetdir;
            }
        }
    }

    private Optional<Path> existingRelease(Path apmReleasesPath) {
        log.info("Searching for targetdir in {}", apmReleasesPath)
        if (!Files.exists(apmReleasesPath)){
            log.info("$APM_RELEASES directoy not found. Will create it at {}", apmReleasesPath)
            Files.createDirectory(apmReleasesPath)
        }
        return Files.list(apmReleasesPath)
            .filter(new ProfilePathFilter())
            .findFirst();
    }

    private class ProfilePathFilter implements Predicate<Path> {
        @Override
        public boolean test(Path p) {
            Path profilePath = p.resolve(PROFILE_PATH);
            if (Files.isDirectory(profilePath, LinkOption.NOFOLLOW_LINKS) && !TMP_DIRNAME.equals(profilePath.getFileName().toString())) {
                log.info("Checking IntroscopeAgent.profile in {}", profilePath);
                final Properties props = new Properties();
                final Path profileFile = profilePath.resolve("IntroscopeAgent.profile");
                Reader reader = null;
                try {
                    reader = new FileReader(profileFile.toFile());
                    props.load(reader);
                    final String erServerId = props.getProperty("acc.server.id");
                    final String erPackageId = props.getProperty("acc.package.id");
                    log.info("Loaded properties from {}: serverId {}, packageId {}", profileFile, erServerId, erPackageId);
                    return releaseServerId.equals(erServerId) && releasePackageId.equals(erPackageId);
                } catch (Exception e) {
                    log.warn("Cannot load IntroscopeAgent.profile - skipping the directory {}: {}", p, e);
                    return false;
                } finally {
                    if (reader != null) {
                        try {
                            reader.close();
                        } catch (IOException ex) {
                            // Handle exception while closing the stream
                            ex.printStackTrace();
                        }
                    }
                }

            } else {
                return false;
            }
        }
    }


    Path extractArchive(Path targetDir, Path archive) {
        // unpack archive zip into agentTmpDir
        String base = targetDir.toString();
        String currDir = agentDir.toFile().getName();
        ZipFile zip = null;
        try {
            zip = new ZipFile(archive.toFile())
            Enumeration<? extends ZipEntry> entries = zip.entries();

            while (entries.hasMoreElements()) {
                ZipEntry e = entries.nextElement();
                String name = fixRootName(e.getName(), currDir);
                if (e.isDirectory()) {
                    new File(base, name).mkdirs();
                } else {
                    new File(base, name).getParentFile().mkdirs();

                    def fpath = Paths.get(base, name)
                    if (Files.exists(fpath)) {
                        Files.delete(fpath)
                    }

                    InputStream inputStream = null;
                    try {
                        inputStream = zip.getInputStream(e);
                        Files.copy(inputStream, fpath);
                    } finally {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            } catch (IOException ex) {
                                // Handle exception while closing the stream
                                ex.printStackTrace();
                            }
                        }
                    }

                    if (!this.os.toLowerCase().contains("windows")) {
                        setPermissions(fpath)
                    }

                }
            }

        } catch (IOException e) {
            log.error("extractArchive:: Error occurred: $e")
            logEvent(String.format("Error while unpacking file %s to %s. Error - %s", archive, base, e))
            throw new ACCWriteAccessException(
                String.format("Error while unpacking file %s to %s", archive, base), targetDir);
        } finally {
            if (zip != null) {
                try {
                    zip.close();
                } catch (IOException ex) {
                    // Handle exception while closing the stream
                    ex.printStackTrace();
                }
            }
        }

        Path unpackgedAgentPath = targetDir.resolve(currDir)
        if (!Files.exists(unpackgedAgentPath)) {
            throw new ACCNoEntityException("Failed to extract package $archive", "apmia not found in the package")
        }

        return unpackgedAgentPath;
    }

    private String fixRootName(String name, String currDir) {
        return name.startsWith("apmia") ? name.replaceFirst("apmia", currDir) : name;
    }

    void setPermissions(Path file){

        HashSet sevenFiveFive = new HashSet<>(Arrays.asList("APMIACtrl.sh", "APMIAgent.sh", "java", "jfr", "jrunscript",
            "jexec", "jspawnhelper", "keytool", "rmiregistry"));

        Set<PosixFilePermission> sevenFiveFivePermissions = new HashSet<>(Arrays.asList(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE,
            PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_EXECUTE,
            PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_EXECUTE))

        HashSet sevenSevenFive = new HashSet<>(Arrays.asList("wrapper"));
        Set<PosixFilePermission> sevenSevenFivePermissions = new HashSet<>(Arrays.asList(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE,
            PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE,PosixFilePermission.GROUP_EXECUTE,
            PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_EXECUTE))


        HashSet sevenSixFour = new HashSet<>(Arrays.asList("APMIAgent.shconf", "wrapper.conf", "wrapper-license.conf"));
        Set<PosixFilePermission> sevenSixFourPermissions = new HashSet<>(Arrays.asList(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE,
            PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE,
            PosixFilePermission.OTHERS_READ))

        if(userName != null && !userName.isEmpty()){
            UserPrincipal userPrincipal = FileSystems.getDefault().getUserPrincipalLookupService()
                .lookupPrincipalByName(userName)
            if(userPrincipal != null){
                Files.setOwner(file, userPrincipal);
            }
        }

        if (sevenFiveFive.contains(file.getFileName().toString())) {
            Files.setPosixFilePermissions(file, sevenFiveFivePermissions);
        }

        if (sevenSixFour.contains(file.getFileName().toString())) {
            Files.setPosixFilePermissions(file, sevenSixFourPermissions);
        }

        if (sevenSevenFive.contains(file.getFileName().toString())) {
            Files.setPosixFilePermissions(file, sevenSevenFivePermissions);
        }
    }

    private CommandResponse triggerCommand(String[] cmd) {
        log.info("Trigger operation: {}", cmd[1])
        def p = java.lang.Runtime.getRuntime().exec(cmd);
        p.getOutputStream().close();
        p.getErrorStream().close();

        def s = new Scanner(p.getInputStream());
        StringBuffer response = new StringBuffer();
        /* Return last line of output in case there are headings */
        while (s.hasNextLine()) {
            def curr = s.nextLine();
            if (curr != "")
                response.append(curr);
        }

        s.close();
        int exitCode = p.waitFor();
        p.destroy();
        return new CommandResponse(exitCode,response.toString())
    }

    private void executeCommand(String[] command) {
        CommandResponse response = triggerCommand(command)
        log.info("APMIA restart exit code {}", response.exitCode)
        if (response.exitCode != 0) {
            log.error("APMIA restart failed with exit code {}", response.exitCode)
            logEvent("Agent restart failed for APMIA Agent \"" + agentName+"\"")
            throw new ACCFrameworkException("APMIA restart failed - " + response.response, "APMIA restart failed - " + response.response)
        }else{
            addCustomResponseStatus("8")
            if(response.response != null && response.response.contains("Manual restart")){
                addResponseMessage(RESTART_MSG)
            }
        }
        logEvent("Restarted APMIA Agent \"" + agentName+"\"")
    }

    void addResponseMessage(String msg){
        log.info("Response message: {}", )
        responseProp.put("plugin.message", msg)
        addCustomResponseStatus("7")
        logEvent(msg)
    }

    void addCustomResponseStatus(String status){
        responseProp.put("plugin.extStatus", status)
    }

    String[] getExecutionCommand(Path unpackedAgentPath, CommandType commandType) {
        def cmdLine = getStartScript()

        def cmdPath = unpackedAgentPath.resolve(cmdLine)
        if (!Files.exists(cmdPath)) {
            throw new ACCNoEntityException("Failed to find $cmdPath", "APMIA install script not found in the package")
        }

        def operation
        switch (commandType){
            case CommandType.RESTART:
                operation = "restart"
                break
            case CommandType.RESTART_WITH_ACC:
                operation = "restart_service"
                break
            default:
                throw new ACCBadParamException("Unsupported operation : ","$commandType")
        }

        return ([cmdPath.toRealPath().toString(), operation]).toArray(new String[0])
    }

    private String getWrapperDir() {
        if (os ==~ "Windows .*") {
            return "wrapper"
        }
        return "conf"
    }

    private String getStartScript() {
        if (os ==~ "Windows .*") {
            return "APMIACtrl.bat"
        }
        return "APMIACtrl.sh"
    }

    String extractParam(String param) {

        String value = requestJson[param]

        log.debug("{}={}", param, value)

        if (value == null) {
            throw new ACCBadParamException("Missing value", param)
        }
        return value
    }

    String extractOptionalParam(String param, String defaultValue) {
        String value = requestJson[param]
        return (value == null) ? defaultValue : value
    }

    Path extractDirParam(String param) {
        String path = extractParam(param)

        // Add / to Unix path in case it's missing.
        if (!path.startsWith("/") && !extractParam("os.name").contains("Windows")) {
            path = "/" + path
        }
        Path value = Paths.get(path)

        if (!Files.exists(value)) {
            throw new ACCBadParamException("Parameter defined, but file/directory $value does not exist", param)
        }
        return value
    }

    void logEvent(String message){
        eventLog.push(new EventLog(message,getCurrentTimeMillis()))
    }

    def handler(String request) {
        long startTime = getCurrentTimeMillis();

        def errs = []

        try {
            log.debug("Raw request: $request")

            // Parse JSON Request, save in the object
            this.requestJson = new groovy.json.JsonSlurper().parseText(request)
            logEvent("Received upgrade plugin request for Agent \"Package {packageName}:{packageVersion}\" by Agent Controller \"{agentControllerName}\"");
            applyPackageToAgent()

            cleanUp(archive)
        }
        catch (ACCException e) {
            log.error("Error occurred: $e")
            errs.push(e.getAccError())
            logEvent("Agent upgrade is failed due to error "+e.getMessage())
            cleanUp(archive)
        }
        catch (Exception e) {

            log.error("Caught Exception $e")
            errs.push(new ACCFrameworkException("Internal Error", e.getMessage()).getAccError())
            logEvent("Agent upgrade is failed due to error "+e.getMessage())
            cleanUp(archive)
        }

        String response = genJson(errs)

        log.debug("Response: $response")

        long endTime = getCurrentTimeMillis();

        long timeTaken = endTime - startTime
        log.info("APMIA Upgrade Time Taken {} ms", timeTaken)
        logEvent("APMIA Upgrade Time Taken " + timeTaken + "ms")
        return response
    }

    @CompileDynamic
    String genJson(errs) {
        // Create the json message
        def json = new JsonBuilder()
        def logs = eventLog.collect { log ->
            [
                message: log.message,
                date: log.timeStamp
            ]}

        def root = json {
            version "1.0"
            className "KeyValuePairs"
            errors errs
            eventLogs logs.reverse()
            properties responseProp
        }

        return json.toString()
    }

    @CompileDynamic
    void cleanUp(Path file) {
        try {
            if (file != null) {
                log.info("cleanup: removing $file")
                Files.deleteIfExists(file)
            }
        }
        catch (IOException e) {
            log.debug("Failed to delete $e")
        }

        if (agentTmpDir != null) {
            log.info("cleanup: removing dir $agentTmpDir")
            if (Files.exists(agentTmpDir)) {
                Files.walk(agentTmpDir)
                    .sorted(Comparator.reverseOrder())
                    .each { path ->
                        Files.delete((Path)path)
                    }
            }
            agentTmpDir = null
        }
    }
}


if (binding.variables.containsKey("request")) {
    logger.info("Running applyApmiaPackageRelease.groovy")

    ap = new ApplyApmiaBSPackage()
    response = ap.handler(request)
}
else {
println("Running applyApmiaPackageRelease.groovy - TEST MODE")
    request = "{"
    request += String.format('"acc.controller.dir":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[0]))
    request += String.format('"acc.broker.url":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[3]))
    request += String.format('"acc.package.url":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[2]))
    request += String.format('"acc.agent.installpath":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[1]))
    request += String.format('"acc.package.release.agent.version":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[4]))
    request += String.format('"acc.package.release.server.id":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[5]))
    request += String.format('"acc.package.release.package.id":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[6]))
    request += String.format('"com.ca.apm.acc.controller.configurationServer.token":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[7]))
request += '"acc.package.id":"1", '
request += '"acc.package.name":"pname", '
request += '"acc.package.version":"pversion", '
    request += String.format('"com.wily.introscope.agentProfile":"%s", ', groovy.json.StringEscapeUtils.escapeJava(args[8]))
    request += String.format('"os.name":"%s" ', groovy.json.StringEscapeUtils.escapeJava(args[9]))
request += "}"

ap = new ApplyApmiaBSPackage()
response = ap.handler(request)

println("Response is:")
println(response)

}
