/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */


import com.ca.apm.acc.messaging.plugins.ZippedContent
import com.ca.apm.acc.messaging.plugins.ZippedContent.ZippedFile
import com.ca.apm.acc.plugin.PluginException
import groovy.io.FileType
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths
import java.util.concurrent.TimeUnit

@Slf4j(value = "LOGGER", category = "com.ca.apm.acc.plugin.GetConfFile")
class GetConfFiles {
    def run(final String request) {
        try {
            return worker(request);
        } catch (final Exception ex) {
            throw new PluginException("GetConfFiles", ex);
        }
    }

    private static def worker(final String request) {
        def response;
        def confFiles = [];

        def inputs = new JsonSlurper().parseText(request);
        final Path agentInstallationPath = Paths.get(inputs["acc.agent.installpath"] as String);
        final Path profilePath = Paths.get(inputs["com.wily.introscope.agentProfile"] as String);
        final Path profileDirPath = profilePath.getParent();
        Path extensionsPath = profileDirPath.resolve((inputs["introscope.agent.extensions.bundles.directory"] as String) ?: "../../extensions");
        if (!Files.exists(extensionsPath)) {
            extensionsPath = profileDirPath.resolve("extensions");
        }
        final Path extensionsPropertiesPath = extensionsPath.resolve("Extensions.profile");

        if (Files.exists(extensionsPropertiesPath)) {
            readFile(extensionsPropertiesPath, agentInstallationPath, confFiles);

            def final traverseOptions = [type: FileType.FILES, nameFilter: ~/^\.deployment$/, maxDepth: 1] as Map<String, Object>;
            extensionsPath.traverse(traverseOptions) { final Path dotDeploymentPath ->
                final Path bundlePropertiesPath = dotDeploymentPath.resolveSibling("bundle.properties");
                readFile(bundlePropertiesPath, agentInstallationPath, confFiles);
            }
        }

        // Create the json message
        // default constructor must be used for backward compatibility with 11.x
        final ZippedContent zippedContent = new ZippedContent();
        zippedContent.setVersion("1.0");
        zippedContent.setClassName("ZippedContent");
        zippedContent.setFiles(confFiles);
        def json = new JsonBuilder(zippedContent);
        response = json.toString()
        return response
    }

    private static void readFile(Path filePath, Path baseDir, List confFiles) {
        if (!Files.exists(filePath)) {
            LOGGER.warn("{} does not exist", filePath);
            return;
        }

        final ZippedFile extensionsPropertiesInfo = new ZippedFile().with {
            it.filePath = filePath.toAbsolutePath().normalize().toString();
            // We do not actually read files here.
            // We just fill a record here and ACC Controller will read the file and send its contents.
            it.fileContent = null;
            it.fileSize = filePath.size();
            it.modified = Files.getLastModifiedTime(filePath).to(TimeUnit.MILLISECONDS);
            it.fileType = filePath.getFileName().toString();
            // ZippedFile in older ACC Controllers does not know this property.
            if (it.hasProperty("baseDir")) {
                it.baseDir = baseDir.toString();
            }
            return it;
        };
        confFiles.add(extensionsPropertiesInfo);
    }
}

def handler() {
    if (binding.variables.containsKey("request")) {
        logger.info("Running getConfFiles.groovy");

        handler = new GetConfFiles();
        response = handler.run(request);
        return response;
    }
}

handler();
