/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin;

import groovy.util.logging.Slf4j
import java.nio.file.Files
import java.nio.file.LinkOption
import java.nio.file.Path
import java.nio.file.Paths

@Slf4j
class BaseInfo {

    final String eol = System.getProperty("line.separator");

    /**
     * Reads all 'properties' from a file. Property is a line like 'key=value'. Lines which don't
     * conform are ignored.
     * Nothing clever here as the target files are supposed to be dummy.
     */
    Map<String, String> getPropertiesFromLines(String[] lines, String separator) {
        Map<String, String> map = [:]

        for (String line : lines) {

            log.trace("line {}", line)

            line = line.trim();

            int separatorIndex = line.indexOf(separator);

            if (separatorIndex > 0 && (line.length() - 1) > separatorIndex) {
                String key = line.substring(0, separatorIndex).trim();
                String value = line.substring(separatorIndex + 1).trim().replaceAll("\"", "");

                log.debug("setting: {}={}", key, value)
                map[key] = value
            }
        }
        return map;
    }

    boolean isEmpty(String s) {
        return s == null || s.isEmpty();
    }

}


// Reference os.{name,value,arch} : http://lopica.sourceforge.net/os.html

/**
 * <p>A utility class that allows getting information on underlying Linux distribution.
 * There are two common ways of obtaining release information. One is by using systemd's
 * specification of /etc/os-release file.
 *
 * <p>Second is the Linux Standard Base (LSB) specification of lsb_release tool with its /etc/*release
 * files.
 *
 * <p>This utility uses both methods and the former takes precedence. As the result, three values can
 * be retrieved: name, version, and pretty name.
 * However, it's not guaranteed that all would be, so consumers of that class should check for
 * null values. Note that {@link #isKnownDistro} returning true indicates at least 'pretty name'
 * being available.
 *
 * @see <a href="http://linuxmafia.com/faq/Admin/release-files.html">release-files documentation</a>
 * @see <a href="https://www.freedesktop.org/software/systemd/man/os-release.html>os-release documentation</a>
 *
 * @author turmi05
 *
 */
@Slf4j
class LinuxInfo extends BaseInfo {

    protected long SCRIPT_TIMEOUT = 3000; // ms

    private String name;
    private String version;
    private String prettyName;

    /**
     * @return True, if at lease pretty name of the distro is known.
     */
    public boolean isKnownDistro() {
        return getPrettyName() != null;
    }

    /**
     * @return Distro name as reported by the OS or as extracted from pretty name. Can be null.
     */
    public String getName() {
        return getName(false);
    }

    /**
     * @param addLinuxSufix If true, the original name will be suffixed with 'Linux', but only if it
     *        doesn't consist it already.
     * @return Distro name as reported by the OS or as extracted from pretty name. Can be null.
     */
    public String getName(boolean addLinuxSufix) {
        String decoratedName = name;
        if (addLinuxSufix && name != null && !name.toLowerCase(Locale.US).contains("linux")) {
            decoratedName += " Linux";
        }
        return decoratedName;
    }


    /**
     * @return Distro version as reported by the OS or as extracted from pretty name. May be null,
     *         as well. This is free form version, i.e. '7.8 (wheezy)'.
     */
    public String getVersion() {
        return version;
    }

    /**
     * @return Full name of the Linux distro as reported by the OS or as a merge of name and
     *         version, i.e. 'Red Hat Enterprise Linux Server release 6.5 (Santiago)'.
     */
    public String getPrettyName() {
        return prettyName;
    }

    public boolean getInfo() {
        boolean success = false;

        try {
            // Try reading /etc/os-release file first as it consists most comprehensive date
            if (!(success = getFromOsReleaseFile())) {
                // then run /usr/bin/lsb_release script
                if (!(success = getFromOsReleaseScript())) {
                    // then read /etc/lsb-release in case the script is not installed
                    if (!(success = getFromLsbReleaseFile())) {
                        // and finally try other *release files
                        success = getFromDistroReleaseFile();
                    }
                }
            }
        } catch (Exception e) {
            log.debug("Unable to resolve release information of this Linux distribution.", e);
        }

        log.debug("{} linux info", success ? "Successfully retrieved" : "Failed to get");

        return success;
    }

    /**
     * Get OS identification using systemd specification (many modern distros use that).<br/>
     *
     * @see <a href="https://www.freedesktop.org/software/systemd/man/os-release.html">os-release documentation</a>
     * @return True, if OS information have been retrieved successfully. False otherwise.
     * @throws Exception In any case
     */
    private boolean getFromOsReleaseFile() throws Exception {

        // Sample content of /etc/os-release in Slackware 14.1:
        // NAME=Slackware
        // VERSION="14.1"
        // ID=slackware
        // VERSION_ID=14.1
        // PRETTY_NAME="Slackware 14.1"
        // ANSI_COLOR="0;34"
        // CPE_NAME="cpe:/o:slackware:slackware_linux:14.1"
        // HOME_URL="http://slackware.com/"
        // SUPPORT_URL="http://www.linuxquestions.org/questions/slackware-14/"
        // BUG_REPORT_URL="http://www.linuxquestions.org/questions/slackware-14/"

        Path osRelease =
                getFirstExistingFile(true, getPath("/etc/os-release"),
                        getPath("/usr/lib/os-release"));
        if (osRelease != null) {
            log.debug(osRelease.toString() + " found. Will read it to get Linux distro info.");
            try {
                def props = getPropertiesFromLines(osRelease.getText("UTF-8").split(eol), "=");

                return resolveValues(props.get("ID"), props.get("NAME"), props.get("VERSION"),
                        props.get("PRETTY_NAME"));
            } catch (Exception e) {
                log.debug(String.format("Unable to parse %s file", osRelease.toString()), e);
            }
        }
        return false;
    }

    /**
     * Get OS identification using Linux Standard Base (LSB) specification (many distros has
     * lsb_release script and /etc/lsb-release).<br/>
     * See {@link http
     * ://refspecs.linuxbase.org/LSB_3.1.1/LSB-Core-generic/LSB-Core-generic/lsbrelease.html} for
     * more information.
     *
     * @return True, if OS information have been retrieved successfully. False otherwise.
     */
    private boolean getFromOsReleaseScript() {

        // Sample output of '/usr/bin/ls_release -a' script:
        // LSB Version:
        // :core-4.0-amd64:core-4.0-ia32:core-4.0-noarch:graphics-4.0-amd64:graphics-4.0-ia32:graphics-4.0-noarch:printing-4.0-amd64:printing-4.0-ia32:printing-4.0-noarch
        // Distributor ID: RedHatEnterpriseServer
        // Description: Red Hat Enterprise Linux Server release 5.8 (Tikanga)
        // Release: 5.8
        // Codename: Tikanga

        // TODO: Are there any other possible locations of the script?
        final Path script = getFirstExistingExecutable(getPath("/usr/bin/lsb_release"));
        if (script != null) {

            log.debug(script.toString() + " found. Will try to run it to get Linux distro info.");

            String command = script.toString() + " -a"
            log.debug("command is {}", command)

            def lines = command.execute().text.split(eol)

            def properties = getPropertiesFromLines(lines, ":");

            return resolveValues(properties.get("Distributor ID"), null, properties.get("Release"), properties.get("Description"));

        }
        return false;
    }

    /**
     * Fallback routine to parse /etc/lsb-release file if the lsb_release script fails.
     *
     * @return
     * @throws Exception
     */
    private boolean getFromLsbReleaseFile() throws Exception {

        // Sample contnent of /etc/lsb-release in Ubuntu
        // DISTRIB_ID=Ubuntu
        // DISTRIB_RELEASE=12.10
        // DISTRIB_CODENAME=quantal
        // DISTRIB_DESCRIPTION="Ubuntu 12.10 (quantal)"

        Path lsbRelease =
                getFirstExistingFile(true, getPath("/etc/lsb-release"), getPath("/etc/lsb_release"));
        if (lsbRelease != null) {
            log.debug(lsbRelease.toString() + " found. Will read it to get Linux distro info.");
            try {

                def props = getPropertiesFromLines(lsbRelease.getText("UTF-8").split(eol) , "=");

                return resolveValues(props.get("DISTRIB_ID"), null, props.get("DISTRIB_RELEASE"),
                        props.get("DISTRIB_DESCRIPTION"));
            } catch (Exception e) {
                log.debug(String.format("Unable to parse %s file", lsbRelease.toString()), e);
            }
        }
        return false;
    }

    private boolean getFromDistroReleaseFile() throws Exception {
        File[] releaseFiles = getPath("/etc").toFile().listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                Path path = Paths.get(dir.getAbsolutePath(), name);
                boolean isRegularFile =
                        Files.isRegularFile(path, LinkOption.NOFOLLOW_LINKS);
                return (name.matches(".*release") || name.matches("slackware-version")
                        || name.matches("debian_version")) && !"lsb-release".equals(name) && !"lsb_release".equals(name) && isRegularFile;
            }
        });
        if (releaseFiles == null) {
            log.debug("Unable to find any of known Linux distro release files.");
            return false;
        }

        if (releaseFiles.length == 1) {
            log.debug(releaseFiles[0].getAbsolutePath() + " found. Will read it to get Linux distro info.");
            String fileName = releaseFiles[0].getName();
            Path filePath = releaseFiles[0].toPath();

            def lines = filePath.getText("UTF-8").split(eol)

            if (lines == null || lines.size() == 0) {
                log.debug(String
                        .format("File %s is not a proper Linux distro release file.",
                                filePath.toString()));
                return false;
            }
            String tempPrettyName = null;
            String tempVersion = null;
            String tempName = null;
            String tempId = null;
            if ("debian_version".equals(fileName)) {
                // Sample outputs:
                // 5.0.1
                tempId = "debian";
                tempName = "Debian GNU/Linux";
                tempVersion = lines[0];
            } else {
                tempPrettyName = lines[0];
                if ("suse-release".equalsIgnoreCase(fileName)) {
                    tempId = "suse";
                    // SuSE-release also consists of VERSION 'property'
                    def props = getPropertiesFromLines(lines, "=");
                    tempVersion = props.get("VERSION");
                } else if ("redhat-release".equalsIgnoreCase(fileName)) {
                    tempId = "redhat";
                }
            }
            return resolveValues(tempId, tempName, tempVersion, tempPrettyName);

        } else if (releaseFiles.length > 1) {
            // TODO on ubuntu I have 2 files: /etc/{os,lsb}_release
            log.debug(String
                    .format("Ambiguous release files found..."));
        }
        return false;
    }

    private Path getFirstExistingFile(boolean acceptLinks, Path... paths) {
        Path found = null;
        for (Path path : paths) {
            if (Files.exists(path)
                    && (acceptLinks ? Files.isRegularFile(path) : Files.isRegularFile(path,
                            LinkOption.NOFOLLOW_LINKS))) {
                found = path;
                break;
            }
        }
        return found;
    }

    private Path getFirstExistingExecutable(Path... paths) {
        Path found = null;
        for (Path path : paths) {
            if (Files.exists(path) && Files.isRegularFile(path) && Files.isExecutable(path)) {
                found = path;
                break;
            }
        }
        return found;
    }

    /**
     * @param id Distributor ID. This value may help to format the values. I.e. Red Hat uses
     *        'release' separator in its pretty name.
     */
    private boolean resolveValues(String id, String name, String version, String prettyName) {

        if (isEmpty(name) && isEmpty(prettyName)) {
            log.debug(String
                    .format("Unable to resolve distro information based on id=[%s],name=[%s],version=[%s],prettyName=[%s]",
                            id, name, version, prettyName));
            return false;
        }

        // We don't resolve version from names and we take it as it is.
        this.version = isEmpty(version) ? null : version;

        if (!isEmpty(name) && !isEmpty(prettyName)) {
            this.name = name;
            this.prettyName = prettyName;
        } else {
            if (isEmpty(prettyName)) {
                this.name = name;
                // Makeup pretty name from name and (optionally) version
                this.prettyName = name + (isEmpty(version) ? "" : (" " + version));
            } else {
                // The name is empty here, but we have the prettyName

                this.prettyName = prettyName;
                // Here we can try to format name and version knowing the distribution ID
                // Some known distributor IDs: slackware, debian, ubuntu, RedHatEnterpriseServer,
                // RedHatEnterpriseWS, RedHatEnterpriseClient, SUSE LINUX, openSUSE, LinuxMint
                // See also https://wiki.egi.eu/wiki/HOWTO05_How_to_publish_the_OS_name
                if (!isEmpty(id)) {
                    int index;

                    if (id.toLowerCase(Locale.US).contains("suse") && !isEmpty(version)) {
                        // Try to extract name from pretty name using version as the separator
                        if ((index = prettyName.indexOf(version)) > -1) {
                            this.name = prettyName.substring(0, index).trim();
                        }
                    } else if ((id.toLowerCase(Locale.US).contains("redhat") || id.toLowerCase(Locale.US).contains("centos"))
                            && (index = prettyName.indexOf("release")) > -1) {
                        // Get name and version, if the pretty name consists of 'release' separator
                        this.name = prettyName.substring(0, index).trim();
                        if (isEmpty(this.version)) {
                            this.version = prettyName.substring(index + "release".length()).trim();
                        }
                    }
                    else {
                        // TODO Check this with Miro. On my Ubuntu, name is blank, id isn't
                        this.name = id;
                    }
                }
            }
        }

        log.debug(String
                .format("Base values id=[%s],name=[%s],version=[%s],prettyName=[%s] resolved to name=[%s],version=[%s],prettyName=[%s]",
                        id, name, version, prettyName, this.name, this.version, this.prettyName));
        return true;
    }

    /**
     * Provides ability for stubbing.
     */
    protected Path getPath(String path) {
        String sp = System.getProperty("osinfo.root")

        if (sp != null) {
            Path tp = Paths.get(sp, path)
            log.debug("have test path {}", tp)
            return tp;
        }

        return Paths.get(path);
    }

}

////////////////////////////////////////////////////////////////////////////////////////////////////



/**
 *
 * @author hilja07
 *
 */
@Slf4j
class WindowsInfo extends BaseInfo {
    final String eol = System.getProperty("line.separator");

    String osName, osVersion;

    WindowsInfo(String osName, String osVersion) {
        this.osName = osName;
        this.osVersion = osVersion;
    }

    public boolean getInfo() {

        String command = getPath("wmic").toString() + " os get Caption,CSDVersion,Version /value";
        log.debug("command is {}", command);

        def map = getPropertiesFromLines(command.execute().text.split(eol), "=")

        log.debug("Caption: {}", map["Caption"]);
        log.debug("CSDVersion: {}", map["CSDVersion"]);
        log.debug("Version: {}", map["Version"]);

        if (isEmpty(map["Caption"])) {
            // This should not happen
            log.warn("Did not find Caption from wmic")
            return false;
        }

        osName = whiteString(map["Caption"]);

        if (!isEmpty(map["CSDVersion"])) {

            if (!isEmpty(map["Version"])) {
                osVersion = map["CSDVersion"] + ", " + map["Version"];
            }
            else {
                osVersion = map["CSDVersion"];
            }
        }
        else {
            if (!isEmpty(map["Version"])) {
                osVersion = map["Version"];
            }
        }

        log.debug("Successfully retrieved Windows info");

        return true;
    }

    public String getName() {
        return osName;
    }

    public String getVersion() {
        return osVersion;
    }

    /** Strip out non-allowed characters */
    String whiteString(String black) {

        String white = black.replaceAll("\\([RC]\\)", "")

        return white.replaceAll("[^\\w\\d\\s/\\[\\]\\(\\).,-_@;: ]", "")
    }

    /**
     * Provides ability for stubbing.
     */
    protected Path getPath(String path) {
        String sp = System.getProperty("osinfo.root")

        if (sp != null) {
            Path tp = Paths.get(sp, path)
            log.debug("have test path {}", tp)
            return tp;
        }

        return Paths.get(path);
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 *
 * @author hilja07
 *
 */
@Slf4j
class OsInfo
{
    /**
     * Handle the request
     */

    String osName, osVersion, osArch;

    def requestJson;

    void setOsInfo() {

        String osName, osVersion, osArch;

        if ("linux".equalsIgnoreCase(this.osName)) {

            LinuxInfo linuxInfo = new LinuxInfo();

            // this logic should be encapsulated within LinuxInfo. Ref: updateAgentOsInfo
            if (linuxInfo.getInfo() && linuxInfo.isKnownDistro())
            {
                osName = linuxInfo.getName(true);
                if (osName != null && linuxInfo.getVersion() != null) {
                    // By default os.version is set to kernel version for Linux
                    String kernel = this.osVersion;

                    log.debug("Using constructed name with osName: {} kernel: {}", osName, kernel)

                    String kernelSuffix = (kernel.isEmpty()) ? "" : (", kernel " + kernel);
                    osVersion = linuxInfo.getVersion() + kernelSuffix;
                }
                else {
                    log.debug("Using pretty name")
                    osName = linuxInfo.getPrettyName();
                }
            }
        }
        else if (this.osName.toLowerCase(Locale.US).startsWith("windows")) {

            WindowsInfo windowsInfo = new WindowsInfo(this.osName, this.osVersion);

            if (windowsInfo.getInfo()) {
               osName    = windowsInfo.getName();
               osVersion = windowsInfo.getVersion();
            }

        } else {
            log.debug("Unknown base platform type: {}", this.osName)
        }

        //  Save new if set
        if (osName == null) {
            log.debug("Discovered no extra details for osName {}", this.osName)
        }
        else {
            log.debug("Discovered extra details for osName: {} -> {}", this.osName, osName)
            this.osName = osName;
        }

        if (osVersion == null) {
            log.debug("Discovered no extra details for osVersion {}", this.osVersion)
        }
        else {
            log.debug("Discovered extra details for osVersion: {} -> {}", this.osVersion, osVersion)
            this.osVersion = osVersion;
        }

        if (osArch == null) {
            log.debug("Discovered no extra details for osArch {}", this.osArch)
        }
        else {
            log.debug("Discovered extra details for osArch: {} -> {}", this.osArch, osArch)
            this.osArch = osArch;
        }
    }

    /**
     * Extract the given param from the json request.   If does not exist try and get the
     * value as a java system property
     */
    String extractParam(String param) {

        String value = requestJson[param]

        if (value != null) {
            log.debug("[json] {} = {}", param, value);
        }
        else {
            value = System.getProperty(param);
            log.debug("[system] {} = {}", param, value);
        }

        return value
    }

    String handler(String request) {

        log.debug("Raw request: $request")

        if (request == null || request.isEmpty()) {
            request = "{}"
        }

        // Parse JSON Request, save in the object
        requestJson = new groovy.json.JsonSlurper().parseText(request)

        osName    = extractParam("os.name");
        osVersion = extractParam("os.version");
        osArch    = extractParam("os.arch");

        setOsInfo();

        log.debug("Running on {}, {}, {}", osName, osVersion, osArch);

        // Create the json message
        def json = new groovy.json.JsonBuilder();
        def root = json {

            "version"  "1.0"
            "className" "KeyValuePairs"

            properties (
                "acc.agent.override.osName": osName,
                "acc.agent.override.osVersion": osVersion,
                "acc.agent.override.osArch": osArch)
        }

        String response = json.toString();

        log.debug("Response: $response");

        return response;
    }

}

////////////////////////////////////////////////////////////////////////////////////////////////////

if (binding.variables.containsKey("request")) {

    logger.debug("Running osinfo.groovy");

    OsInfo osi = new OsInfo();
    response = osi.handler(request);
}
else {

    /*
     * To run in test mode, a logging implementation has to be supplied on the cp,
     * so for example call with something like this:
     *
     * groovy -cp /usr/share/java/slf4j-simple.jar:/usr/share/java/slf4j-api.jar -Dorg.slf4j.simpleLogger.defaultLogLevel=debug ./osinfo.groovy
     */

    println("Running osinfo.groovy - TEST MODE");

	//println(System.getProperties())

    String osName, osVersion, osArch;

    osName = (args.length > 0) ? args[0] : null;
    osVersion = (args.length > 1) ? args[1] : null;
    osArch = (args.length > 2) ? args[2] : null;
    testRoot = (args.length > 3) ? args[3] : null;

    request = "{"

    if (osName != null) {
        request += String.format('"os.name":"%s", ', osName)
    }

    if (osVersion != null) {
        request += String.format('"os.version":"%s", ', osVersion)
    }

    if (osArch != null) {
        request += String.format('"os.arch":"%s", ', osArch)
    }

    request += "}"

    OsInfo osi = new OsInfo();
    response = osi.handler(request);

}





