/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream
import org.apache.commons.io.IOUtils;

import javax.net.ssl.HostnameVerifier
import javax.net.ssl.HttpsURLConnection
import javax.net.ssl.SSLContext
import javax.net.ssl.X509TrustManager
import java.nio.charset.StandardCharsets
import java.nio.file.Files
import java.nio.file.NoSuchFileException
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.StandardCopyOption
import java.nio.file.StandardOpenOption
import java.util.regex.Matcher
import java.util.regex.Pattern



@Slf4j
@CompileStatic
class ACCException extends Exception {
    String errno
    String[] arguments

    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }

    void setParams(String ... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }

    def getAccError() {
        return ["errno":this.errno, "arguments":arguments]
    }
}

@CompileStatic
class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

@CompileStatic
class ACCWriteAccessException extends ACCException {

    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

@CompileStatic
class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom.toAbsolutePath().normalize().toString())
    }
}

@CompileStatic
class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

@Slf4j
class UpdateTrust {
    def requestJson;
    def logger;

    void httpGet(URL url, Path to, boolean trustSelfSigned) {

        // Write the file to a temporary file first
        Path temp = Paths.get(to.toString() + ".tmp." + Thread.currentThread().getId() + "." + System.nanoTime())

        InputStream is
        try {
            URLConnection urlConnection = url.openConnection()
            if (urlConnection instanceof HttpURLConnection) {
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "text/plain")
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "application/octet-stream")
            }
            if (urlConnection instanceof HttpsURLConnection) {
                HttpsURLConnection connection = (HttpsURLConnection) urlConnection
                if (trustSelfSigned) {
                    def nullTrustManager = [
                            checkClientTrusted: { chain, authType -> },
                            checkServerTrusted: { chain, authType -> },
                            getAcceptedIssuers: { null }
                    ]

                    SSLContext sc = SSLContext.getInstance("SSL")
                    sc.init(null, [nullTrustManager as X509TrustManager] as X509TrustManager[], null)
                    connection.setSSLSocketFactory(sc.getSocketFactory())

                    def nullHostnameVerifier = [
                            verify: { hostname, session -> true }
                    ]
                    connection.setHostnameVerifier(nullHostnameVerifier as HostnameVerifier)
                }
            }
            is = urlConnection.getInputStream()
        }
        catch (IOException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }

        try {
            Files.deleteIfExists(temp)
            Files.copy(is, temp)
            Files.move(temp, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
        }
        catch (NoSuchFileException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
        finally {
            try {
                Files.deleteIfExists(temp)
            }
            catch (Exception ignore) {
                log.info("Failed to delete $temp")
            }

            is.close()
        }
    }

    String extractParam(String param) {

        String value = requestJson[param]

        log.debug("{}={}", param, value)

        if (value == null) {
            throw new ACCBadParamException("Missing value", param)
        }
        return value
    }

    Path extractDirParam(String param) {
        String path = extractParam(param)

        // Add / to Unix path in case it's missing.
        if (!path.startsWith("/") && !extractParam("os.name").contains("Windows")) {
            path = "/" + path
        }
        Path value = Paths.get(path)

        if (!Files.exists(value)) {
            throw new ACCBadParamException("Parameter defined, but file/directory $value does not exist", param)
        }
        return value
    }

    String extractOptionalParam(String param, String defaultValue) {
        String value = this.requestJson[param]
        return (value == null) ? defaultValue : value
    }

    def getFileContent(Path filePath){
        byte[] bytesContent = java.nio.file.Files.readAllBytes(filePath);
        new java.lang.String(bytesContent, "UTF-8");
    }

    def saveFile (Path filePathArg, String contentArg) {
        Files.write(filePathArg, contentArg.getBytes("UTF-8"), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);
    }


    def commentProfile(String fileLoc, String property) {
        Path filePath = Paths.get(fileLoc)
        String content = getFileContent(filePath)
        log.info("Updating agent profile {} to comment {}", fileLoc, property)
        def propQuoted = Pattern.quote(property)
        Matcher matcher = Pattern.compile("(?<!\\\\\\r?\\n)^([\\t ]*)" + propQuoted + "=(.*)", Pattern.MULTILINE).matcher(content)
        if (matcher.find()) {
            log.info("Found property to comment")//debug
            StringBuffer sb = new StringBuffer()
            matcher.appendReplacement(sb, "#\$0")
            matcher.appendTail(sb)
            content = sb.toString()
            saveFile(filePath, content)
        } else {
            log.warn("Property to comment {} not present", property)
        }
    }

    def setOrAddProperty(String fileLoc, String property, String value) {
        Path filePath = Paths.get(fileLoc)
        String content = getFileContent(filePath)
        log.info("Updating agent profile {} to set {}", fileLoc, property)
        def propQuoted = Pattern.quote(property)
        Matcher matcher = Pattern.compile("(?<!\\\\\\r?\\n)^([\\t ]*)" + propQuoted + "=(.*)", Pattern.MULTILINE).matcher(content)
        if (matcher.find()) {
            log.info("Found property to change")
            StringBuffer sb = new StringBuffer()
            matcher.appendReplacement(sb, (property + "=" + value).replace("\$","\\\$"))
            matcher.appendTail(sb)
            content = sb.toString()
            saveFile(filePath, content)
        } else {
            Matcher matcherUncomment = Pattern.compile("(?<!\\\\\\r?\\n)^([#\\t ]*)" + propQuoted + "=(.*)", Pattern.MULTILINE).matcher(content)
            if (matcherUncomment.find()) {
                log.info("Found property to uncomment")
                StringBuffer sb = new StringBuffer()
                matcherUncomment.appendReplacement(sb, (property + "=" + value).replace("\$","\\\$"))
                matcherUncomment.appendTail(sb)
                content = sb.toString()
                saveFile(filePath, content)
            } else {
                log.info("Adding as a new property at the end")
                content += System.lineSeparator() + property + "=" + value + System.lineSeparator()
                saveFile(filePath, content)
            }
        }
    }

    /**
     * Read the given properties file to get the broker url
     * (configurationServer.url=thebrokername)
     */
    String readBrokerUrl() {

        String serverUrl = "com.ca.apm.acc.controller.configurationServer.url";
        String serverUrlShort = "configurationServer.url";

        String brokerString = System.getenv(serverUrl);
        if (brokerString != null) {
            log.debug("Found " + serverUrl + " in system environment variables: $brokerString")
        } else {
            Path propertiesFilename = makeAccPath("config/apmccctrl.properties")
            FileInputStream is

            log.debug("Reading " + serverUrl + " from $propertiesFilename")
            try {
                // Get the broker URL from the properties file.
                Properties controllerProperties = new Properties()

                is = new FileInputStream(propertiesFilename.toFile())
                controllerProperties.load(is)
                brokerString = controllerProperties.getProperty(serverUrl)

                if(brokerString == null)
                    brokerString = controllerProperties.getProperty(serverUrlShort);
            }
            catch (IOException e) {
                throw new ACCReadAccessException("Failed to read config", propertiesFilename)
            }
            finally {
                if (is != null) {
                    is.close()
                }
            }
        }

        if (!brokerString.endsWith("/")) {
            brokerString += "/"
        }

        return brokerString
    }


    def run(request) {
        log.info("Running updateTrust")
        this.requestJson = new groovy.json.JsonSlurper().parseText(request)
        String trustUrl = this.requestJson["updateTrust.url"]
        String newHash = this.requestJson["updateTrust.newHash"]
        Boolean forceRemove = this.requestJson["updateTrust.forceRemove"]
        String storePassword = this.requestJson["updateTrust.storePassword"]
        String agentProfile = this.requestJson["com.wily.introscope.agentProfile"]
        boolean trustSelfSigned = extractOptionalParam("com.ca.apm.acc.cs.trustSelfSignedCerts", "true").toBoolean()
        log.debug("trustSelfSigned: {}", trustSelfSigned)
        def accDir = Paths.get(extractOptionalParam("acc.controller.dir", "."))
        log.debug("accDir: {}", accDir.toAbsolutePath().normalize())
        def agentDir = extractDirParam("acc.agent.installpath")
        log.info("agentDir: {}", agentDir.toAbsolutePath().normalize())
        def agentTrustJks = agentDir.resolve("trust.jks")

        String b = requestJson["acc.broker.url"]
        String brokerUrlStr = ((b != null) ? b : readBrokerUrl())

        if (forceRemove == Boolean.TRUE) {
            log.info("Deleting trust store")
            agentTrustJks.toFile().delete()
        } else {
            log.info("Got url = {}", trustUrl)
            URL url = new URL(trustUrl)
            URL brokerUrl = new URL(brokerUrlStr)
            log.info("Using url = {}", trustUrl)
            String file = url.getFile()
            if (!file.startsWith("/apm/appmap/")) {
                file = "/apm/appmap" + file
            }
            url = new URL(brokerUrl.getProtocol(), brokerUrl.getHost(), brokerUrl.getPort(), file)
            httpGet(url, agentTrustJks, trustSelfSigned)
            log.info("Download done.")
        }

        def p = new Properties()
        p.load(new InputStreamReader(new FileInputStream(agentProfile), StandardCharsets.UTF_8))
        setOrAddProperty(agentProfile, "acc.package.truststore.hash", newHash)
        if (newHash.startsWith("h")) {
            setOrAddProperty(agentProfile, "agentManager.tls.validateHostname", "false")
        } else {
            setOrAddProperty(agentProfile, "agentManager.tls.validateHostname", "true")
        }
        int i = 1
        while(true) {
            def key = "agentManager.url." + i
            def value = p.getProperty(key)
            if (value == null) {
                break
            }
            if ((value.startsWith("https://") || value.startsWith("ssl://") || value.startsWith("wss://"))
                && !newHash.startsWith("0")) {
                setOrAddProperty(agentProfile, "agentManager.trustStore." + i, "{ApmIAHome}/trust.jks")
                setOrAddProperty(agentProfile, "agentManager.trustStorePassword." + i, storePassword)
            } else {
                commentProfile(agentProfile, "agentManager.trustStore." + i)
                commentProfile(agentProfile, "agentManager.trustStorePassword." + i)
            }
            i++
        }
        if (agentDir.resolve("extensions").toFile().exists()) {
            String extensions = this.requestJson["introscope.agent.extensions.bundles.load"]
            if (extensions) {
                def split = extensions.split(",")
                def accControllerName = split.find { it -> it.startsWith("acc-controller-") }
                def accControllerPath = agentDir.resolve("extensions").resolve(accControllerName)
                if (accControllerPath.toFile().exists()) {
                    updateAccControllerTrust(accControllerPath, newHash, storePassword)
                }
            } else {
                Path uiaAccDir = agentDir.resolve("probes").resolve("custom").resolve("acc_controller");
                if (uiaAccDir.toFile().exists()) {
                    updateAccControllerTrust(uiaAccDir, newHash, storePassword)
                    def accPropertiesPath = uiaAccDir.resolve("bundle.properties")
                    if (accPropertiesPath.toFile().exists()) {
                        def accControllerPath = agentDir.resolve("extensions").resolve("deploy")
                        def deployDir = accControllerPath.toFile()
                        def accControllerTgz = deployDir.list {dir, name ->
                            name.startsWith("acc-controller-") && name.endsWith(".tar.gz")}
                        if (accControllerTgz != null && accControllerTgz.size() > 0) {
                            def targetTarGz = accControllerPath.resolve(accControllerTgz[0])
                            String content = getFileContent(accPropertiesPath)
                            def targetTarGzFile = targetTarGz.toFile()
                            def targetTarGzTmpFile = new File(targetTarGzFile.toString() + ".tmp")
                            updateTarGzContents(targetTarGzFile, targetTarGzTmpFile, "bundle.properties", content)
                            log.info("Replace original acc controller tar.gz")
                            Files.move(targetTarGzTmpFile.toPath(), targetTarGzFile.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
                        }
                    }
                }
            }
        }
        log.info("Property update done.")
    }

    void updateAccControllerTrust(Path accControllerPath, String newHash, String storePassword) {
        log.info("Found acc-controller {}", accControllerPath)
        def accPropertiesPath = accControllerPath.resolve("bundle.properties")
        if (accControllerPath.toFile().exists()) {
            def accPropertiesPathStr = accPropertiesPath.toString()
            if (newHash.startsWith("0")) {
                commentProfile(accPropertiesPathStr, "com.ca.apm.acc.controller.certificate.trustStore")
                commentProfile(accPropertiesPathStr, "com.ca.apm.acc.controller.certificate.trustStore.password")
                setOrAddProperty(accPropertiesPathStr, "com.ca.apm.acc.controller.certificate.ignore", "true")
            } else {
                setOrAddProperty(accPropertiesPathStr, "com.ca.apm.acc.controller.certificate.trustStore", "{ApmIAHome}/trust.jks")
                setOrAddProperty(accPropertiesPathStr, "com.ca.apm.acc.controller.certificate.trustStore.password", storePassword)
                setOrAddProperty(accPropertiesPathStr, "com.ca.apm.acc.controller.certificate.ignore", "false")
            }
        }
    }

    void updateTarGzContents(File readTarGz, File writeTarGz, String overwritePath, String contents) {
        log.info("Update tar.gz contents source: {} target: {}", readTarGz, writeTarGz)
        try (FileInputStream fis = new FileInputStream(readTarGz)) {
            try (GzipCompressorInputStream gcis = new GzipCompressorInputStream(fis)) {
                try (TarArchiveInputStream tais = new TarArchiveInputStream(gcis)) {
                    try (FileOutputStream fos = new FileOutputStream(writeTarGz)) {
                        try (GzipCompressorOutputStream gcos = new GzipCompressorOutputStream(fos)) {
                            try (TarArchiveOutputStream taos = new TarArchiveOutputStream(gcos)) {
                                taos.setLongFileMode(TarArchiveOutputStream.LONGFILE_POSIX);
                                TarArchiveEntry tae;
                                while ((tae = tais.getNextEntry()) != null) {
                                    def entryName = tae.getName()
                                    if (overwritePath.equals(entryName)) {
                                        log.info("Found entry to update")
                                        tae = new TarArchiveEntry(overwritePath);
                                        def bytes = contents.getBytes(StandardCharsets.UTF_8)
                                        tae.setSize(bytes.length)
                                        taos.putArchiveEntry(tae)
                                        taos.write(bytes)
                                        taos.closeArchiveEntry()
                                    } else {
                                        taos.putArchiveEntry(tae)
                                        IOUtils.copy(tais, taos, 65536)
                                        taos.closeArchiveEntry()
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

new UpdateTrust().run(request)

