/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

import com.ca.apm.acc.plugin.PluginException
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j

@Slf4j(value = "LOGGER", category = "com.ca.apm.acc.plugin.GetDynamicExtensionDir")
class GetDynamicExtensionDir {
    static def run(final String request) {
        try {
            return worker(request);
        } catch (final Exception ex) {
            throw new PluginException("GetDynamicExtensionDir", ex);
        }
    }

    private static def worker(final String request) {
        def response;

        def inputs = new JsonSlurper().parseText(request);
        def filename = inputs["introscope.agent.extensions.bundles.directory"] ?: "../../extensions";
        def profile  = inputs["com.wily.introscope.agentProfile"];
        def isDynamic = inputs["introscope.agent.extensions.bundles.mode"];

        /* We're actually only interested in the deployed extensions folder */
        //filename += "/deploy";

        def r = [:];
        r.version = "1.0"
        r.className = "DirectoryList"

        def errors = [];

        def extdir;

        if (isDynamic != null) {
            try {
                extdir = new java.io.File(filename);

                /* If it's relative then it's relative to location of profile */
                if (!extdir.isAbsolute()) {

                    extdir = java.nio.file.Paths.get(new java.io.File(profile).getParent(), extdir.getPath()).toFile();

                    LOGGER.debug("extdir now=$extdir");

                    /* If it's still relative then it's relative to 'user.dir' */
                    if (!extdir.isAbsolute()) {
                        extdir = java.nio.file.Paths.get(inputs["user.dir"] ?: "", extdir.getPath()).toFile();

                        LOGGER.debug("And now it is=$extdir");
                    }
                }

                LOGGER.info("Extension directory at $extdir");

                /* Normalize path when sending response. */
                r.path = java.nio.file.Paths.get(extdir.getPath()).normalize() as String;

                /*
                 * Check it is a directory, no-such-file otherwise. Note listFiles
                 * returns 'null' for all errors.
                 */
                if (!extdir.isDirectory()) {
                    LOGGER.info("$extdir is not a directory - this must be an older agent");
                    def error = new Error();
                    error.arguments = [extdir.getPath()];
                    error.errno = "ENOENT";
                    errors.add(error);
                }
                else {

                    extdir = java.nio.file.Paths.get(extdir.toString(), "deploy").toFile();
                    r.path = java.nio.file.Paths.get(extdir.toString()).normalize() as String;

                    LOGGER.info("Deploy directory at $extdir");

                    if (!extdir.isDirectory()) {
                        LOGGER.info("$extdir.getPath() is not a directory - this must be an older agent");
                        def error = new Error();
                        error.arguments = [extdir.getPath()];
                        error.errno = "ENOENT";
                        errors.add(error);
                    }
                    else {

                        def dirList = extdir.listFiles();

                        if (dirList != null) {

                            LOGGER.debug("Total entries=$dirList.length");

                            def entries = [];
                            for (def i = 0; i < dirList.length; i++) {
                                if (dirList[i].getName() != ".lock" && dirList[i].getName() != ".lockProfile"){
                                    def one = [:];
                                    one.name = new String(dirList[i].getName());
                                    def aFile = new java.io.File(extdir.getPath() + "/" + one.name);
                                    one.version = "n/a";
                                    if (dirList[i].getName().endsWith(".tar.gz")) {
                                        /* Handle snapshot versions - will split to 4 - release versions to 3 */
                                        def res = dirList[i].getName().split("-");
                                        if (res != null ){
                                            if( res.length > 3 && res[res.length-1].startsWith("SNAPSHOT")) {
                                                one.version = (res[res.length-2] + "-" + res[res.length-1]).replace(".tar.gz", "");
                                            } else if( res.length > 3){
                                                one.version = (res[res.length-1]).replace(".tar.gz", "");
                                            } else if (res.length > 2){
                                                one.version = res[2].replace(".tar.gz", "");
                                            }
                                        }
                                    }
                                    one.size = aFile.length();
                                    one.modified = aFile.lastModified();
                                    entries.add(one);
                                }
                            }

                            r.directoryEntries = entries;
                        }
                    }
                }
            } catch(e) {
                LOGGER.debug("Caught error $e");
                def error = new Error();
                error.arguments = [extdir.getPath()];
                if (e instanceof java.nio.file.InvalidPathException ||
                        e instanceof java.nio.file.NoSuchFileException) {
                    error.errno = "ENOENT";
                }
                else if (e instanceof java.nio.file.AccessDeniedException) {
                    error.errno = "EACCESS";
                }
                else if (e instanceof java.io.IOException) {
                    error.errno = "EIOERROR";
                }
                LOGGER.error("Had $error.errno reading $extdir");
                errors.add(error);
            }
        } else {
            LOGGER.info("introscope.agent.extensions.bundles.mode was not found for agent; will not scan for dynamic extensions.");
        }

        r["errors"] = errors;
        def json = new JsonBuilder(r);
        response = json.toString()
        return response;
    }

    static class Error {
        String errno;
        String[] arguments;
    }
}

def handler() {
    if (binding.variables.containsKey("request")) {
        logger.info("Running GetDynamicExtensionDir.groovy");

        handler = new GetDynamicExtensionDir();
        response = handler.run(request);
        return response;
    }
}

handler();
