/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

import java.nio.file.*
import java.io.*
import java.util.*
import java.util.logging.Logger
import java.lang.*

// Response constants
CLASS_NAME = "FileContent"
VERSION = "1.0"
FILE_TYPE = "com.wily.introscope.agentProfile";

// Log level property
LOG_LVL_PROPERTY = "log4j.logger.IntroscopeAgent"
LOG_LVL_PROPERTY_LOGBACK = "introscope.agent.log.level.root"

// EPAgent type
AGENT_TYPE_EPAGENT = "ep-agent"

// dotnet type
AGENT_TYPE_DOTNET = ".NetAgent"

// Error constants
EINTROSCOPEPROPERTYNOTFOUND = "EINTROSCOPEPROPERTYNOTFOUND"
EINVALIDINTROSCOPEVALUE = "EINVALIDINTROSCOPEVALUE"
ENOENT = "ENOENT"
EIOERROR = "EIOERROR"
EWRITEACCESS = "EWRITEACCESS"
EACCESS = "EACCESS"

// An array of errors
errors = []
// Possible log lvl values
logLvl = ["ERROR", "VERBOSE", "TRACE", "INFO", "WARN", "DEBUG", "FATAL", "ALL", "OFF"]

/**
 * Make a JSON structure containing the Response message and return as a String
 */
def makeResponse(errorsArg=[], filePathArg=null, fileTypesArg=null, modifiedArg=null) {
    json = new groovy.json.JsonBuilder()
    if (errors.size() == 0) {
        root = json {
            version VERSION
            className CLASS_NAME
            filePath filePathArg
            fileType fileTypesArg
            modified modifiedArg
            errors[]
        }
    }
    else {
        root = json {
            version VERSION
            className CLASS_NAME
            errors errorsArg
        }
    }
    json.toString()
}

/**
 * Retrieves the content of the file and writes to the string
 * @param filePath A {@link Path} of a file
 * @return The file content 
 */
def getFileContent(filePath){
    byte[] bytesContent = java.nio.file.Files.readAllBytes(filePath);
    new java.lang.String(bytesContent, "UTF-8");
}

/**
 * Adds an error to the errors class member
 * @param name An error name 
 */
def addError(name, args=[]) {
    error = ["errno": name, "arguments" : args];
    errors.add(error);
}

/**
 * Updates intrscope.profile's properties. 
 * @param fileLoc The location of a file
 * @param property The property name that value needs to be update
 * @param value new property value
 */
def updateProfile(fileLoc, property, value, agentType) {

    Path filePath = java.nio.file.Paths.get(fileLoc);
    String content = getFileContent(filePath)
    pattern = java.util.regex.Pattern;

    if (property == LOG_LVL_PROPERTY) {
        logger.info("Updating agent profile $fileLoc to loglevel to $value");
        // The value for log level is case insensitive
        validValue = logLvl.contains(value.toString().toUpperCase(Locale.US));
        if (validValue) {
            if (agentType == AGENT_TYPE_DOTNET){
                // dotnet agent
                logger.info("Updating DotNet agent profile");

                // property can start with a whitespace, but not with new the line or horizontal tab
                // replaces the value up to special characters i.e. ',',
                java.util.regex.Matcher matcher = pattern.compile("^(?![(\\r|\\n)])^(^|[\\s]*)introscope.agent.log.level=[a-zA-Z]*",
                        pattern.MULTILINE).matcher(content);
                if (matcher.find() == true) {
                    content = matcher.replaceAll("introscope.agent.log.level=" + value)
                    saveFile(filePath, content)
                }
                else {
                    // In case there are no matches, add the property not found error, but for EPA
                    addError(EINTROSCOPEPROPERTYNOTFOUND,  ["introscope.agent.log.level", fileLoc])
                }

            } else if (agentType == AGENT_TYPE_EPAGENT){
                // ep agent type

                // property can start with a whitespace, but not with new the line or horizontal tab
                // replaces the value up to special characters i.e. ',',
                java.util.regex.Matcher matcher = pattern.compile("^(?![(\\r|\\n)])^(^|[\\s]*)log4j.logger.EPAgent=[a-zA-Z]*",
                        pattern.MULTILINE).matcher(content);
                if (matcher.find() == true) {
                    content = matcher.replaceAll("log4j.logger.EPAgent=" + value)
                    saveFile(filePath, content)
                }
                else {
                    // In case there are no matches, add the property not found error, but for EPA
                    addError(EINTROSCOPEPROPERTYNOTFOUND,  ["log4j.logger.EPAgent", fileLoc])
                }

            } else {
                // java/infra agent types

                // iterate over all log level properties (log4j, logback)
                def logApplied = false;
                for( logProp in [LOG_LVL_PROPERTY, LOG_LVL_PROPERTY_LOGBACK] ) {
                    // property can start with a whitespace, but not with new the line or horizontal tab
                    // replaces the value up to special characters i.e. ',',
                    java.util.regex.Matcher matcher = pattern.compile("^(?![(\\r|\\n)])^(^|[\\s]*)${logProp}=[a-zA-Z]*",
                            pattern.MULTILINE).matcher(content);
                    if (matcher.find() == true) {
                        content = matcher.replaceAll("${logProp}=" + value)
                        saveFile(filePath, content)
                        logger.info("Updated agent profile $fileLoc to loglevel ($logProp) to $value")
                        logApplied = true
                        break
                    }
                }

                if (!logApplied) {
                    // if not log4j log level property, then try

                    // In case there are no matches, add the property not found error
                    addError(EINTROSCOPEPROPERTYNOTFOUND,  [property, fileLoc])
                }
            }
        }
        else {
            // In case the log level property value is not valid, add the invalid introscope value error
            addError(EINVALIDINTROSCOPEVALUE,  [value, LOG_LVL_PROPERTY, fileLoc])
        }
        
    }
    else {
        logger.info("Updating agent profile $fileLoc to set $property to $value");
        // Matches only requested properties, replace whole value
        java.util.regex.Matcher matcher = pattern.compile("^(?![(\\r|\\n)])^(^|[\\s]*)"+property.toString()+"=(.*)", pattern.MULTILINE).matcher(content);
        if (matcher.find() == true) {
            logger.debug("Changing property other than $LOG_LVL_PROPERTY");
            content = matcher.replaceAll(property + "=" + value);
            saveFile(filePath, content)
        }
        else {
            addError(EINTROSCOPEPROPERTYNOTFOUND,  [property, fileLoc])
        }
    }
}

/**
 * Saves the content to the specified file path
 * @param filePathArg A {@link Path} of a file  
 * @param contentArg A {@link String} content to be written to the file
 */
def saveFile (filePathArg, contentArg) {
    Files.write(filePathArg, contentArg.getBytes("UTF-8"), java.nio.file.StandardOpenOption.TRUNCATE_EXISTING);
}

/**
 * Gets the last modification time of the file
 * @param fileLoc A {@link String} file location
 * @return The {@link Long} modification time
 */
def getLastFileModification(fileLoc) {
    File updatedFile = new java.io.File(fileLoc);
    modified = updatedFile.lastModified();
}

/**
 * Returns the JSON request  object if the request is valid, otherwise null 
 * @param request A {@link String} request 
 * @return the JSON request  object if the request is valid, otherwise null 
 */
def getValidRequest(request) {
    if (request != null) {
        request = new groovy.json.JsonSlurper().parseText(request)
        if (request && request["acc.request.property"]  && request["com.wily.introscope.agentProfile"]
        && request["acc.request.value"] != null) {
            // replaces new lines and trims empty spaces
            request["acc.request.value"] = request["acc.request.value"].replaceAll("\n","").trim()

            File apFile = new java.io.File(request["com.wily.introscope.agentProfile"]);
            // Check whether the given path is absolute
            if (!apFile.isAbsolute()) {
                if (request["user.dir"]) {
                    request["com.wily.introscope.agentProfile"] = java.nio.file.Paths.get(request["user.dir"], request["com.wily.introscope.agentProfile"]).toString();
                    logger.debug("absolute path $request['com.wily.introscope.agentProfile']");
                }
                else {
                    return null
                }
            }
            if (request["acc.agent.type"] != null){
            	request["acc.agent.type"] = request["acc.agent.type"].replaceAll("\n","").trim()
            }
            return request
        }
        return null
    }
    return null
}

/**
 * Handles the class logic and returns the JSON response in string  
 */
def init() {

    // Default 'empty' response
    response="{}"
    String prop;
    String value;
    String agentType;

    try {
        // Parse JSON Request
        request = getValidRequest (request)
        if (request) {
            prop = request["acc.request.property"]
            value = request["acc.request.value"]
            agentType = request["acc.agent.type"]
            fileLoc = request["com.wily.introscope.agentProfile"]
            logger.debug("Updating the $fileLoc");
            updateProfile(fileLoc, prop, value, agentType)
            response = makeResponse(errors, fileLoc, FILE_TYPE, getLastFileModification(fileLoc))
        }
    } catch(Exception e){
        logger.debug("Error while updating the property", e)

        if (e instanceof java.nio.file.NoSuchFileException) {
            error= ["errno": ENOENT, "arguments" : [fileLoc, e.toString()]]
        }
        else if (e instanceof java.nio.file.AccessDeniedException) {
            File apFile = new java.io.File(fileLoc);
            if (apFile.canRead()) {
                // write access permision
                error= ["errno": EWRITEACCESS, "arguments" : [fileLoc, e.toString()]]
            }
            else {
                // read access permision
                error= ["errno": EACCESS, "arguments" : [fileLoc, e.toString()]]
            }
        }
        else if (e instanceof java.io.IOException) {
            error= ["errno": EIOERROR, "arguments" : [fileLoc, e.toString()]]
        }
        // Other exceptions will be catched on the plugin framework lvl

        errors.add(error);
        response = makeResponse(errors)
        response = json.toString();
    } finally{
        logger.debug("setIntroscopeProperty: Plugin Response $response");
        response
    }
}

init()
