/*
 * Copyright (c) 2025 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.json.JsonBuilder
import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import java.nio.charset.StandardCharsets
import java.nio.file.Files
import java.nio.file.LinkOption
import java.nio.file.NoSuchFileException
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.StandardCopyOption
import java.nio.file.attribute.PosixFilePermission
import java.util.regex.MatchResult
import java.util.stream.Collectors
import java.util.zip.GZIPInputStream
import java.util.zip.ZipEntry
import java.util.zip.ZipFile
import javax.net.ssl.HostnameVerifier
import javax.net.ssl.HttpsURLConnection
import javax.net.ssl.SSLContext
import javax.net.ssl.X509TrustManager
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream

@Slf4j
@CompileStatic
class ACCException extends Exception {
    String errno
    String[] arguments

    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }

    void setParams(String ... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }

    def getAccError() {
        return ["errno":this.errno, "arguments":arguments]
    }
}

@CompileStatic
class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

@CompileStatic
class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom.toAbsolutePath().normalize().toString())
    }
}

@CompileStatic
class ACCWriteAccessException extends ACCException {

    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

@CompileStatic
class ACCNoEntityException extends ACCException {

    ACCNoEntityException(String msg, enitity) {
        super(msg, "ENOENT")
        setParams(enitity.toString())
    }
}

@CompileStatic
class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

@CompileStatic
class ACCFrameworkException extends ACCException {
    ACCFrameworkException(String msg, String info) {
        super(msg, "EFRAMEWORKERROR")
        setParams(info)
    }
}


@Slf4j
@CompileStatic
class FilePuller  {

    /**
     * Fetch the URL and write the file to a temporary file before moving
     * it atomically into place to avoid half-written files etc
     */
    void httpGet(URL url, Path to, String token, boolean trustSelfSigned, boolean useHttp) {
        // Write the file to a temporary file first
        Path temp = Paths.get(to.toString() + ".tmp")

        InputStream is
        try {
            URLConnection urlConnection = url.openConnection()
            if (urlConnection instanceof HttpURLConnection) {
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "text/plain")
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "application/octet-stream")
            }

            if (useHttp) {
                is = urlConnection.getInputStream()
            } else if (urlConnection instanceof HttpsURLConnection) {
                HttpsURLConnection connection = (HttpsURLConnection)urlConnection
                connection.setRequestProperty("Authorization", "Bearer " + token)
                if(trustSelfSigned)
                {
                    def nullTrustManager = [
                            checkClientTrusted: { chain, authType ->  },
                            checkServerTrusted: { chain, authType ->  },
                            getAcceptedIssuers: { null }
                    ]

                    SSLContext sc = SSLContext.getInstance("SSL")
                    sc.init(null, [nullTrustManager as  X509TrustManager] as  X509TrustManager[], null)
                    connection.setSSLSocketFactory(sc.getSocketFactory())

                    def nullHostnameVerifier = [
                            verify: { hostname, session -> true }
                    ]
                    connection.setHostnameVerifier(nullHostnameVerifier as HostnameVerifier)
                }
                is = connection.getInputStream()
            } else {
                ((HttpURLConnection) urlConnection).setRequestProperty("Authorization", "Bearer " + token)
                is = urlConnection.getInputStream()
            }
        }
        catch (IOException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }

        try {
            Files.deleteIfExists(temp)
            Files.copy(is, temp)
            Files.move(temp, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
        }
        catch (NoSuchFileException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
        finally {
            try {
                Files.deleteIfExists(temp)
            }
            catch (Exception ignore) {
                log.info("Failed to delete $temp")
            }

            is.close()
        }
    }

    /**
     * Copy (or pull) file from the broker
     * and write locally
     */
    void copyBrokerFile(URL broker, String from, Path to, String token, boolean trustSelfSigned, boolean useHttp) {

        URL url = new URL(broker.toString() + from)

        log.debug("copyBrokerFile: $url -> $to")
        httpGet(url, to, token, trustSelfSigned, useHttp)
    }
}

@Slf4j
@CompileStatic
class Archive {

    Path archive

    Archive(Path archive) {
        this.archive = archive
    }

    Path unzip(Path outDir, Path... toExtracts) {
        log.info("Unzipping $toExtracts from $archive to $outDir")

        ZipFile zipFile = null
        Path extracted = null
        try {
            zipFile = new ZipFile(archive.toFile())

            ZipEntry entry = null
            toExtracts.each { toExtract ->
                if (entry == null) {
                    log.debug("Trying to unzip $toExtract from $archive to $outDir")
                    entry = zipFile.getEntry(toExtract.toString().replace("\\", "/"))
                    if (entry == null) {
                        log.debug("Failed to find file in zip with /.  Re-trying with \\")
                        entry = zipFile.getEntry(toExtract.toString().replace("/", "\\"))
                    }
                    if (entry != null) {
                        extracted = (Path)toExtract
                    }
                }
            }

            if (entry == null) {
                throw new ACCNoEntityException("Failed to unzip $toExtracts", "$toExtracts was not found in zip file")
            }
            else {
                Path dest = outDir.resolve(extracted.getFileName())
                if (!Files.exists(dest.getParent())) {
                    Files.createDirectories(dest.getParent())
                }
                Files.deleteIfExists(dest)

                // Unzip the file
                Files.copy(zipFile.getInputStream(entry), dest)

                return dest
            }
        }
        finally {
            if (zipFile != null) {
                zipFile.close()
            }
        }
    }

    Path untar(Path outFile, Path toExtract){
        log.info("Untarring $toExtract from $archive to $outFile")

        TarArchiveInputStream tarInput = null
        try {

            tarInput = new TarArchiveInputStream(new GZIPInputStream(new FileInputStream(archive.toFile())));

            TarArchiveEntry currentEntry = tarInput.getNextTarEntry();
            TarArchiveEntry entry = null
            while(null != currentEntry) {

                if(currentEntry.getName().equals(toExtract.toString().replace("\\", "/"))){
                    entry = currentEntry;
                    break
                }else if(currentEntry.getName().equals(toExtract.toString().replace("/", "\\"))){
                    entry = currentEntry;
                    break
                }

                currentEntry = tarInput.getNextTarEntry()
            }

            if (null == entry) {
                log.info("File $toExtract was not found in tar file $archive")
                return null
            }else {
                log.debug("Found $entry.name with size $entry.size in $archive")

                Files.createDirectories(outFile.getParent())
                Files.deleteIfExists(outFile)

                // Untar the file
                Files.copy(tarInput, outFile)

                return outFile
            }
        }
        finally {
            if (tarInput != null) {
                tarInput.close()
            }
        }
    }
}


@Slf4j
@CompileStatic
class ApplyDotNetBSPackage {
    public static final String TMP_DIRNAME = "tmp"
    public static final String PACKAGETMP_FILENAME = "package.tmp"
    def requestJson

    Path archive

    URL broker
    String partialUrl
    String authToken
    boolean trustSelfSigned
    boolean useHttp
    Path accDir, agentDir, userDir, agentParentDir, agentTmpDir
    String os
    String releaseVersion, releaseServerId, releasePackageId
    List<String> releasesLocked = new ArrayList<>()

    void initEnv() {

        log.debug("Running on: {}", System.getProperty("os.name"))

        accDir = Paths.get(extractOptionalParam("acc.controller.dir", "."))
        log.debug("accDir: {}", accDir.toAbsolutePath().normalize())

        agentDir = extractDirParam("acc.agent.installpath")
        log.info("agentDir: {}", agentDir.toAbsolutePath().normalize())

        // Make sure we can resolve this path before we start.
        log.debug("Profile path={}", makeUserPath(extractParam("com.wily.introscope.agentProfile")));

        agentParentDir = extractDirParam("agent.parent.dir");
        log.info("agentParentDir: {}", agentParentDir.toAbsolutePath().normalize())

        releaseVersion = extractParam("acc.package.release.agent.version");
        log.info("releaseVersion: {}", releaseVersion)

        releaseServerId = extractParam("acc.package.release.server.id");
        log.info("releaseServerId: {}", releaseServerId)

        releasePackageId = extractParam("acc.package.release.package.id");
        log.info("releasePackageId: {}", releasePackageId)

        log.info("Collecting locked releases")
        for (int i = 1; ; i++) {
            final String releaseId = extractOptionalParam("acc.package.release.locked.release." + i, null);
            if (releaseId == null) {
                break;
            }
            final Path lrProductCodeFile = this.agentParentDir.resolve(releaseId).resolve("win-installer-guid.txt")
            final Path lrProfile = this.agentParentDir.resolve(releaseId).resolve("IntroscopeAgent.profile")
            if (!Files.exists(lrProductCodeFile) && !Files.exists(lrProfile)) {
                log.info("Ignoring locked release {} - not valid release", releaseId)
            } else {
                this.releasesLocked.add(releaseId)
            }
        }
        if (this.releasesLocked.isEmpty()) {
            log.info("No locked releases")
        } else {
            log.info("Locked releases: {}", this.releasesLocked)
        }

        partialUrl = extractParam("acc.package.url")

        String b = requestJson["acc.broker.url"]
        setBroker((b != null) ? b : readBrokerUrl())

        authToken = extractParam("com.ca.apm.acc.controller.configurationServer.token")

        trustSelfSigned = extractParam("com.ca.apm.acc.cs.trustSelfSignedCerts").toBoolean()

        String useHttpProp = requestJson["com.ca.apm.acc.cs.useHttp"]
        if(useHttpProp == null) {
            useHttp = false
        } else {
            useHttp = useHttpProp.toBoolean()
        }

        log.debug("broker: {}", this.broker)
        log.debug("partialUrl: {}", this.partialUrl)

        os = requestJson["os.name"]
        log.info("OS: {}", os)

        // for Windows - only 2 dotnet agent installations are allowed
        if (os ==~ "Windows.*") {
            if (this.releasesLocked.size() > 1) {
                throw new ACCFrameworkException("There are already 2 releases of .NET agent present in the system. Please delete at least one and retry the action.",
                        "There are already 2 releases of .NET agent present in the system. Please delete at least one and retry the action.")
            }
        }

        agentTmpDir = File.createTempDir("dotnet_acc").toPath()
        agentTmpDir = agentTmpDir.toRealPath()

        archive = agentTmpDir.resolve(PACKAGETMP_FILENAME)
        log.debug("archive: {}", this.archive)
    }

    void setBroker(String b) {
        this.broker = new URL((b.endsWith("/")) ? b : b + "/" )
    }

    Path makeAccPath(String other) {
        return (other != null) ? accDir.resolve(other) : accDir
    }

    Path makeUserPath(String other) {

        if (other != null) {
            Path pathOther = Paths.get(other)
            if (pathOther.isAbsolute()) {
                return pathOther;
            }
        }

        /*
         * Need to resolve the paths against userDir.
         * Note that we only try this when the other path is relative
         * because we might not have permissions to actually read user.dir
         */
        if (userDir == null) {
            userDir = extractDirParam("user.dir")
            log.debug("userDir: {}", userDir.toAbsolutePath().normalize())
        }

        return (other != null) ? userDir.resolve(other) : userDir
    }

    /**
     * Read the given properties file to get the broker url
     * (configurationServer.url=thebrokername)
     */
    String readBrokerUrl() {

        String serverUrl = "com.ca.apm.acc.controller.configurationServer.url";
        String serverUrlShort = "configurationServer.url";

        String brokerString = System.getenv(serverUrl);
        if (brokerString != null) {
            log.debug("Found " + serverUrl + " in system environment variables: $brokerString")
        } else {
            Path propertiesFilename = makeAccPath("config/apmccctrl.properties")
            FileInputStream is

            log.debug("Reading " + serverUrl + " from $propertiesFilename")
            try {
                // Get the broker URL from the properties file.
                Properties controllerProperties = new Properties()

                is = new FileInputStream(propertiesFilename.toFile())
                controllerProperties.load(is)
                brokerString = controllerProperties.getProperty(serverUrl)

                if(brokerString == null)
                    brokerString = controllerProperties.getProperty(serverUrlShort);
            }
            catch (IOException e) {
                throw new ACCReadAccessException("Failed to read config", propertiesFilename)
            }
            finally {
                if (is != null) {
                    is.close()
                }
            }
        }

        if (!brokerString.endsWith("/")) {
            brokerString += "/"
        }

        return brokerString
    }




    Path applyPackageToAgent() {

        initEnv()

        // Pull the package file from the broker
        new FilePuller().copyBrokerFile(broker, partialUrl, archive, authToken, trustSelfSigned, useHttp)

        final Path unpackedAgentPath = extractArchive(archive)

        removeUnlockedReleases()

        final String targetdir = getTargetdir()

        installArchive(unpackedAgentPath, targetdir)
    }

    void removeUnlockedReleases() {
        log.info("Removing unused releases from {}", this.agentParentDir)
        final List<String> releases2remove = Files.list(this.agentParentDir).filter { p ->
            if (Files.isDirectory(p, LinkOption.NOFOLLOW_LINKS) && !TMP_DIRNAME.equals(p.getFileName().toString())) {
                final String releaseName = p.getFileName().toString()
                return !this.releasesLocked.contains(releaseName)
            }
            return false
        }.map(p->p.getFileName().toString()).collect(Collectors.toList())
        if (releases2remove.isEmpty()) {
            log.info("No unused releases found in {}", this.agentParentDir)
        } else {
            log.info("Removing unused releases in {}", releases2remove)
            for (final String releaseName : releases2remove) {
                final Path releasePath = this.agentParentDir.resolve(releaseName)
                log.info("Removing release directory {}", releasePath)

                if (os ==~ "Windows.*") {
                    // uninstall the release directory if there is a product code file
                    // start /wait msiexec /l*v log.log /x {A4B59B6B-8208-4766-8FD8-82B5B424EC5C} APM_DEBUG_INSTALL=1 /quiet
                    // get product code from win-installer-guid.txt
                    final Path productCodeFile = this.agentParentDir.resolve(releaseName).resolve("win-installer-guid.txt")
                    if (Files.exists(productCodeFile)) {
                        try (final Scanner scanner = new Scanner(productCodeFile)) {
                            final Optional<MatchResult> res = scanner.findAll("ProductCode=(.*)").findFirst();
                            if (res.isPresent() && res.get().groupCount() == 1) {
                                final String productCode = res.get().group(1)

                                final String scriptFile = this.agentTmpDir.resolve("run.bat").toString()
                                try (PrintWriter pw = new PrintWriter(new FileOutputStream(scriptFile))) {
                                    pw.format("start /wait msiexec /l*v log.log /x %s APM_DEBUG_INSTALL=1 /quiet", productCode)
                                    pw.println()
                                    pw.println("IF %ERRORLEVEL% NEQ 0 EXIT /B %ERRORLEVEL%")
                                }

                                StringBuffer response = execCmd(new String[] {scriptFile}, this.agentTmpDir)

                                log.info("Uninstall response: {}", response)

                                // clean up directory
                                cleanUpDirectory(this.agentParentDir.resolve(releaseName))
                            }
                        }
                    } else {
                        log.warn("Release {} won't be uninstalled, because of missing win-installer-guid.txt file", releaseName)
                    }

                } else {
                    try {
                        Files.walk(releasePath)
                                .sorted(Comparator.reverseOrder())
                                .map(Path::toFile)
                                .forEach(File::delete);
                    } catch (Exception e) {
                        log.warn("Error occurred during removing release directory", e)
                    }
                }
            }
        }
    }

    String getTargetdir() {
        log.info("Searching for targetdir in {}", this.agentParentDir)
        final Optional<Path> existingRelease = Files.list(this.agentParentDir).filter { p ->
            {
                if (Files.isDirectory(p, LinkOption.NOFOLLOW_LINKS) && !TMP_DIRNAME.equals(p.getFileName().toString())) {
                    log.info("Checking IntroscopeAgent.profile in {}", p)
                    final Properties props = new Properties();
                    final Path profileFile = p.resolve("IntroscopeAgent.profile")
                    try (final Reader reader = new FileReader(profileFile.toFile())) {
                        props.load(reader);
                        final String erServerId = props.getProperty("acc.server.id")
                        final String erPackageId = props.getProperty("acc.package.id")
                        log.info("Loaded properties from {}: serverId {}, packageId {}", profileFile, erServerId, erPackageId)
                        return this.releaseServerId.equals(erServerId) && this.releasePackageId.equals(erPackageId);
                    } catch (Exception e) {
                        log.warn("Cannot load IntroscopeAgent.profile - skipping the directory $p: $e");
                        return false;
                    }
                } else {
                    return false;
                }
            }
        }.findFirst();
        if (existingRelease.isPresent()) {
            log.info("Found existing release in {}", existingRelease.get())
            return existingRelease.get().toRealPath().getFileName().toString();
        }

        final String targetdirPrefix = "wily_" + this.releaseVersion
        for (int i = 0; ; i++) {
            final String targetdir = targetdirPrefix + ((i == 0) ? "" : ("." + i));
            log.info("Checking if targetdir {} is available", targetdir)
            if (!Files.exists(agentParentDir.resolve(targetdir))) {
                return targetdir;
            }
        }
    }


    Path extractArchive(Path archive) {
        // unpack archive zip into agentTmpDir
        String base = agentTmpDir.toString();

        try (ZipFile zip = new ZipFile(archive.toFile())) {

            Enumeration<? extends ZipEntry> entries = zip.entries();

            while (entries.hasMoreElements()) {
                ZipEntry e = entries.nextElement();
                String name = e.getName()

                if (e.isDirectory()) {
                    new File(base, name).mkdirs();
                } else {
                    new File(base, name).getParentFile().mkdirs();

                    def fpath = Paths.get(base, name)
                    try (final InputStream inputStream = zip.getInputStream(e)) {
                        Files.copy(inputStream, fpath);
                    }

                    if ("install.sh".equals(fpath.getFileName().toString())) {
                        Files.setPosixFilePermissions(fpath, new HashSet<>(Arrays.asList(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.OWNER_READ)));
                    }
                }
            }

        } catch (IOException e) {
            throw new ACCWriteAccessException(String.format("Error while unpacking file %s to %s", archive, base), agentTmpDir);
        }

        Path unpackgedAgentPath = agentTmpDir.resolve("dotnetagent")
        if (!Files.exists(unpackgedAgentPath)) {
            throw new ACCNoEntityException("Failed to extract package $archive",  "dotnetagent not found in the package")
        }
        return unpackgedAgentPath;
    }

    void installArchive(Path unpackedAgentPath, String targetdir) {
        String[] cmds;
        if (os ==~ "Windows.*") {
            final String installCmd = "IntroscopeDotNetAgentInstall64.msi"
            final Path cmdPath = unpackedAgentPath.resolve(installCmd);
            if (!Files.exists(cmdPath)) {
                throw new ACCNoEntityException("Failed to find $cmdPath",  "install binary not found in the package")
            }

            final String scriptFile = unpackedAgentPath.resolve("run.bat").toString()
            try (PrintWriter pw = new PrintWriter(new FileOutputStream(scriptFile))) {
                pw.format("start /wait msiexec /l*v log.log /i IntroscopeDotNetAgentInstall64.msi INSTALLDIR=\"%s\" WILY=\"%s\" APM_DEBUG_INSTALL=1 /quiet",
                        this.agentParentDir.toString(), this.agentParentDir.resolve(targetdir).toString())
                pw.println()
                pw.println("IF %ERRORLEVEL% NEQ 0 EXIT /B %ERRORLEVEL%")
            }

            cmds = new String[] {scriptFile}

        } else {
            final String installCmd = "./install.sh"
            final Path cmdPath = unpackedAgentPath.resolve(installCmd);
            if (!Files.exists(cmdPath)) {
                throw new ACCNoEntityException("Failed to find $cmdPath",  "install script not found in the package")
            }
            cmds = new String[]{ installCmd,
                    "--targetdir=" + targetdir,
                    "--agent-home=" + agentParentDir.toString()}
        }

        StringBuffer response = execCmd(cmds, unpackedAgentPath)

        log.info("Install response: {}", response)

        if (os ==~ "Windows.*") {
            final Path logPath = unpackedAgentPath.resolve("log.log");
            if (Files.exists(logPath)) {
                log.info("Copying log file {}", logPath)
                try (final Scanner scanner = new Scanner(new File(logPath.toString()), StandardCharsets.UTF_16)) {
                    Optional<MatchResult> res = scanner.findAll("Property\\(S\\): LOGS = (.*)").findFirst();
                    if (res.isPresent() && res.get().groupCount() == 1) {
                        final String logDir = res.get().group(1)
                        Files.copy(logPath, Paths.get(logDir).resolve(logPath.getFileName()))
                        log.info("Log file copied to {}", logDir)

                    } else {
                        log.info("Cannot copy log file - log dir (LOGS) not identified")
                    }
                }
            } else {
                log.info("Log file not found")
            }
        }
    }

    private StringBuffer execCmd(String[] cmds, Path execPath) {
        def p = new ProcessBuilder(cmds)
                .directory(execPath.toFile())
                .start();

        p.getOutputStream().close();
        p.getErrorStream().close();

        def s = new Scanner(p.getInputStream());
        StringBuffer response = new StringBuffer();
        /* Return last line of output in case there are headings */
        while (s.hasNextLine()) {
            def curr = s.nextLine();
            if (curr != "")
                response.append(curr);
        }

        s.close();
        int exitCode = p.waitFor();
        log.info("Dotnet installer exit code {}", exitCode)
        p.destroy();
        if (exitCode != 0) {
            log.error("Dotnet installer failed with exit code {}", exitCode)
            throw new ACCFrameworkException("DotNet installer failed with exit code " + exitCode, "DotNet installer failed with exit code " + exitCode)
        }
        response
    }

    /**
     * Extract the given param from the json request, throwing an ACCBadParamException
     * if the value does not exist
     */
    String extractParam(String param) {

        String value = requestJson[param]

        log.debug("{}={}", param, value)

        if (value == null) {
            throw new ACCBadParamException("Missing value", param)
        }
        return value
    }

    String extractOptionalParam(String param, String defaultValue) {
        String value = requestJson[param]
        return (value == null) ? defaultValue : value
    }

    Path extractDirParam(String param) {
        String path = extractParam(param)

        // Add / to Unix path in case it's missing.
        if (!path.startsWith("/") && !extractParam("os.name").contains("Windows")) {
            path = "/" + path
        }
        Path value = Paths.get(path)

        if (!Files.exists(value)) {
            throw new ACCBadParamException("Parameter defined, but file/directory $value does not exist", param)
        }
        return value
    }

    def handler(String request) {
        def errs = []

        try {
            log.debug("Raw request: $request")

            // Parse JSON Request, save in the object
            this.requestJson = new groovy.json.JsonSlurper().parseText(request)

            applyPackageToAgent()

            cleanUp()
        }
        catch (ACCException e) {
            log.error("Error occurred: $e")
            errs.push(e.getAccError())
            cleanUp()
        }
        catch (Exception e) {
            /*
             * An exceptional exception occurred, so log an error
             */

            log.error("Caught Exception $e")
            log.error("exception", e)
            e.printStackTrace()

            errs.push(new ACCFrameworkException("Internal Error", e.getMessage()).getAccError())
            cleanUp()
        }

        String response = genJson(errs)

        log.debug("Response: $response")

        return response
    }

    @CompileDynamic
    String genJson(errs) {
        // Create the json message
        def json = new JsonBuilder()

        def root = json {
            version "1.0"
            className "KeyValuePairs"
            errors errs
        }

        return json.toString()
    }

    def cleanUp(Path file) {
        try {
            if (file != null) {
                log.info("cleanup: removing $file")
                Files.deleteIfExists(file)
            }
        }
        catch (IOException e) {
            log.debug("Failed to delete $e")
        }
    }

    def cleanUp() {
        cleanUp(archive)
        if (agentTmpDir != null) {
            log.info("cleanup: removing dir $agentTmpDir")
            cleanUpDirectory(agentTmpDir)
            agentTmpDir = null
        }
    }

    def cleanUpDirectory(Path directory) {
        if (directory != null) {
            log.info("cleanup: removing dir $directory")
            if (Files.exists(directory)) {
                Files.walk(directory)
                        .sorted(Comparator.reverseOrder())
                        .map(Path::toFile)
                        .forEach(File::delete);
            }
        }
    }
}


if (binding.variables.containsKey("request")) {
    logger.info("Running applyDotNetPackageRelease.groovy")

    ap = new ApplyDotNetBSPackage()
    response = ap.handler(request)
}
else {

    /*
     * To run in test mode, a logging implementation has to be supplied on the cp,
     * so for example call with something like this:
     *
     * groovy -cp /usr/share/java/slf4j-simple.jar:/usr/share/java/slf4j-api.jar
     *     -Dorg.slf4j.simpleLogger.defaultLogLevel=debug ./applyPackage.groovy
     *         /home/hilja07/temp/acc/APMCommandCenterController/
     *         /home/hilja07/temp/acc/wily/
     *         acc-controller-package-99.99.accEagle-SNAPSHOT.tar
     *         http://accdemowin01.ca.com:8888
     *
     * I needed to install this:
     * sudo apt-get install libslf4j-java
     */

    println("Running applyDotNetPackageRelease.groovy - TEST MODE")

    String accDir
    String agentDir
    String partialUrl
    URL broker
    try {
        accDir = Paths.get(args[0]).toAbsolutePath().toString()      // acc install dir
        agentDir = Paths.get(args[1]).toAbsolutePath().toString()    // agent install dir
        partialUrl = Paths.get(args[2])  // partial path to upgraded controller package - catted to broker url
        broker = new URL(args[3])
    }
    catch (ArrayIndexOutOfBoundsException e) {
    }

    request = "{"
    request += String.format('"acc.controller.dir":"%s", ', groovy.json.StringEscapeUtils.escapeJava( accDir) )  // this property only exists for testing
    request += String.format('"acc.broker.url":"%s", ', groovy.json.StringEscapeUtils.escapeJava( broker.toString().toString() ) )      // this property only exists for testing
    request += String.format('"acc.package.url":"%s", ', groovy.json.StringEscapeUtils.escapeJava( partialUrl.toString() ) )
    request += String.format('"acc.agent.installpath":"%s", ', groovy.json.StringEscapeUtils.escapeJava( agentDir) )
    request += '"acc.package.id":"1", '
    request += '"acc.package.name":"pname", '
    request += '"acc.package.version":"pversion", '
    request += '"com.wily.introscope.agentProfile":"c://sw//groovy//TomcatAcc//wily//core//config//IntroscopeAgent.profile", '

    request += '"os.name":"Windows" '
    request += "}"

    ap = new ApplyDotNetBSPackage()
    response = ap.handler(request)

    println("Response is:")
    println(response)

}
