/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j
import groovy.xml.XmlSlurper
import java.nio.file.AccessDeniedException
import java.nio.file.Files
import java.nio.file.NoSuchFileException

/**
 * make a JSON structure containing the Response message and return as a String
 */
@Slf4j
class WebsphereInfo {
    /**
     * Java File.exists() and Files.exists() both
     * return 'false' for a 'permission denied' error.
     */
    static def fileExists(file) {
        try {
            Files.readAllBytes(new File(file).toPath())
            log.debug(file + " read in")
            return true
        } catch (AccessDeniedException e) {
            log.debug("Had " + e + " exception for " + file)
            return true
        } catch (NoSuchFileException e) {
            log.debug("File $file not found")
            return false
        }
        log.debug("file does not exist")
        return false
    }

    static def makeResponse(errs = [], si = null, sn = null, sb = null) {
        def json = new JsonBuilder()
        json (
            version: "1.0",
            className: "KeyValuePairs",
            properties: [
                    "acc.agent.appserver.name": si,
                    "acc.agent.appserver.version": sn,
                    "acc.agent.appserver.built": sb
            ],
            errors: errs
        )
        return json.toString()
    }

    /**
     * make a JSON structure containing the Response message and return as a String
     */
    static def makeErrorResponse(errs = []) {
        def json = new JsonBuilder()
        json (
            version: "1.0",
            className: "KeyValuePairs",
            properties: [],
            errors: errs
        )
        return json.toString()
    }

    static def tryProductFile(productFileName, errors) {
        FileInputStream productFile = null
        // Load xml from WAS.product and extract version information
        try {
            if (fileExists(productFileName)) {
                log.debug("in fileExists if")
                productFile = new FileInputStream(productFileName)
                def parser = new XmlSlurper()
                // Configure parser to ignore DTD
                parser.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false)
                parser.setFeature("http://xml.org/sax/features/namespaces", false)
                parser.setFeature("http://apache.org/xml/features/disallow-doctype-decl", false)
                def product = parser.parse(productFile)
                def name = product.@name.text()
                log.debug("name=$name")
                def version = product.version.text()
                def build = product.'build-info'.@date.text()

                // Clear any previous errors
                errors.clear()

                return makeResponse([], name, version, build)
            } else {
                log.error("File $productFileName does not exist")
                def anErr = ["errno": "EAPPSERVER", "arguments": ["WebSphere", "File $productFileName does not exist"]]
                errors.add(anErr)
            }
        } catch (Exception e) {
            log.error("Error loading or accessing Websphere properties", e)
            def anErr = ["errno": "EAPPSERVER", "arguments": ["WebSphere", e.toString()]]
            errors.add(anErr)
        } finally {
            if (productFile) {
                productFile.close()
            }
        }

        return null
    }

    def handler(request) {
        // Default response if we do not find any signs of WebSphere.
        def json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: []
        )
        def response = json.toString()

        if (request) {
            // Parse JSON Request
            def inputs = new JsonSlurper().parseText(request)

            def errors = []

            /*
             * Check the java system properties in the given order to derive the Websphere
             * installation directory.   item[0] is the java system property containing
             * the Websphere root directory. item[1] is the subdirectories with the
             * Websphere root to look for WAS.product
             */
            def candidatePaths = [
                    ["was.install.root", ["properties/version", "product/properties/version"]],
            ]

            /*
             * For each base directory look for WAS.product in these sub-directories, in the given order
             */
            for (item in candidatePaths) {
                def base = inputs[item[0]]
                if (base != null) {
                    log.info("Websphere discovered at $base")
                    for (subdir in item[1]) {
                        response = tryProductFile("$base/$subdir/WAS.product", errors)

                        if (response) {
                            break
                        }
                    }
                }
            }

            /*
             * If we have any errors replace the response with an error response
             */
            if (errors) {
                response = makeErrorResponse(errors)
            }
        }

        return response
    }

}

response = new WebsphereInfo().handler(request)
