/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.google.gson.stream.JsonReader
import groovy.transform.CompileStatic;
import org.apache.commons.compress.archivers.ArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream
import org.apache.commons.compress.utils.IOUtils

import java.nio.file.StandardCopyOption
import java.util.zip.GZIPInputStream
import java.util.zip.GZIPOutputStream

import java.nio.file.FileVisitResult
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.Files
import java.nio.file.SimpleFileVisitor
import java.nio.file.attribute.BasicFileAttributes

import java.text.MessageFormat
import java.util.logging.ConsoleHandler
import java.util.logging.FileHandler
import java.util.logging.Level
import java.util.logging.LogRecord
import java.util.logging.Logger
import java.util.logging.Formatter;

@CompileStatic
interface IMetadataEntry{
    /**
     * Returns the name of the metadata key.
     *
     * @return the name of the metadata key
     */
    String getKey()
}

@CompileStatic
enum PreferenceType {
    CURRENT, UPGRADE, INVALID

    static PreferenceType getType(String type) {
        switch (type.toUpperCase()) {
            case "C":
                return CURRENT
            case "U":
                return UPGRADE
            default:
                return INVALID
        }
    }
}

@CompileStatic
enum MergeableType {
    PROFILE, PBD, PBL, INVALID

    static MergeableType getType(String type) {
        switch (type.toLowerCase()) {
            case "profile":
                return PROFILE
            case "pbd":
                return PBD
            case "pbl":
                return PBL
            default:
                return INVALID
        }
    }
}

@CompileStatic
class MergeException extends Exception
{
    private static final long serialVersionUID = 1L;

    public MergeException()
    {
        super();
    }

    public MergeException(String message)
    {
        super(message);
    }

    MergeException(Throwable cause)
    {
        super(cause);
    }

    MergeException(String message, Throwable cause)
    {
        super(message, cause);
    }

}

@CompileStatic
class NoMergeCandidatesException extends MergeException{
    NoMergeCandidatesException(String message)
    {
        super(message);
    }
}

@CompileStatic
interface IRulesEngine {
    boolean isDelete(MergeableType type, String fileName)
    boolean isOverwrite(MergeableType type, String fileName)
    boolean isMerge(MergeableType type, String fileName)
    boolean isEntryDelete(MergeableType type, IMetadataEntry entry)
    PreferenceType getPreferredType(MergeableType type, IMetadataEntry entry)
    Key getRenamedEntry(MergeableType type, IMetadataEntry entry)
    Set<IMetadataEntry> getAllRenamedEntries(MergeableType type)
}

@CompileStatic
interface IMergeable<T extends IMergeable<?>>{
    //T merge(T other);
    T merge(T other, IRulesEngine engine);
    void write(Writer writer) throws IOException;
}

@CompileStatic
class Key implements IMetadataEntry {
    private String fKey
    private String fGroup

    Key(String key) {
        fKey = key
        fGroup = parseGroup(key)
    }

    private String parseGroup(String key) {
        String group = key
        int index = key.lastIndexOf('.')
        if (index > 0) {
            String last = key.substring(index+1)
            if (last.length()==1){
                String temp = key.substring(0, index)
                if(temp.lastIndexOf('.') > 0){
                    index = temp.lastIndexOf('.')
                }
            }
            group = key.substring(0, index)
        }

        return group
    }

    String getKey() {
        return fKey
    }

    String getGroup() {
        return fGroup
    }

    int hashCode() {
        int result = 17

        int hash = (fKey == null) ? 0 : fKey.hashCode()
        result = (31 * result) + hash

        return result
    }

    boolean equals(Object obj) {
        boolean retValue = false
        if (obj instanceof Key) {
            Key otherObj = (Key) obj
            if (fKey != null && fKey.equals(otherObj.fKey)) {
                retValue = true
            }
        }
        return retValue
    }

    String toString() {
        StringBuilder strBldr = new StringBuilder("Class: Key")
        strBldr.append("\n\tKey:   " + fKey).append("\n\tGroup: " + fGroup)
        return strBldr.toString()
    }
}

@CompileStatic
class Line {
    private String fContent
    private int    fIndex

    Line(String content, int index)
    {
        fContent = content
        fIndex = index
    }
    String getContent() { return fContent }
    int getIndex() { return fIndex }
    String toString() { return fContent }
}

@CompileStatic
class Value {
    private String fValue
    private boolean fCommented
    private List<Line> fComments = []
    private int fIndex
    private boolean fIsOld = false

    Value(String value, boolean commented, List<Line> comments, int lineNumber) {
        fValue = value
        fCommented = commented
        fIndex = lineNumber
        fComments.clear()
        for (Line aLine : comments) {
            Line line = new Line(aLine.getContent(), aLine.getIndex())
            fComments.add(line)
        }
    }

    Value(String value, boolean commented, List<Line> comments, int lineNumber, boolean isOldValue) {
        fValue = value
        fCommented = commented
        fIndex = lineNumber
        fIsOld = isOldValue
        fComments.clear()
        for (Line aLine : comments) {
            Line line = new Line(aLine.getContent(), aLine.getIndex())
            fComments.add(line)
        }
    }

    String getValue() {
        return fValue
    }

    boolean isCommented() {
        return fCommented
    }

    List<Line> getComments() {
        return fComments
    }

    int getLineNumber() {
        return fIndex
    }

    boolean isOldValue() {
        return fIsOld
    }

    int hashCode() {
        int result = 17
        int hash = (fValue == null) ? 0 : fValue.hashCode()
        result = (31 * result) + hash
        return result
    }

    boolean equals(Object obj) {
        boolean retValue = false
        if (obj instanceof Value) {
            Value otherObj = (Value) obj
            if (fValue != null && fValue.equals(otherObj.fValue)) {
                retValue = true
            }
        }
        return retValue
    }

    String toString() {
        String comments = getCommentString()
        StringBuilder strBldr = new StringBuilder("Class: Value")
        strBldr.append("\n\tValue:     $fValue")
                .append("\n\tCommented: $fCommented")
                .append("\n\tComments:  $comments")
        return strBldr.toString()
    }

    private String getCommentString() {
        StringBuilder strBldr = new StringBuilder()
        for (Line line : fComments) {
            strBldr.append("\n").append(line)
        }
        return strBldr.toString()
    }
}

@CompileStatic
class KMergeConstants
{
    public static final String kNewline                 = System.getProperty("line.separator");
    public static final String kProfileType             = "profile";
    public static final String kPblType                 = "pbl";
    public static final String kPbdType                 = "pbd";
    public static final String kPropertiesType          = "properties";
    public static final String kSkipMergeTag            = "Skip.Merge";
    public static final String kMergeExceptionMessage   = "Failed to merge {0} because {1}.";
    public static final int    kHeaderDelimitDeterminer = 10;
    public static final String kMergeLogFile            = "mergeutility.log";
    public static final String kMergedOutputFolderName  = "mergedoutput";
    public static final String kWorkingDirectoryName = ".acc";
    public static final String kExtensionPropertyFile = "bundle.properties";
    public static final String kTmpFolder = "tmp";
    public static final String kExtensionTarGZ = "gz";
}

@CompileStatic
class MergeCustomMessageFormatter extends Formatter
{
    private static final MessageFormat messageFormat = new MessageFormat(
            "{0,date,dd/mm/yyyy hh:mm:ss} {1}: {2}");

    public MergeCustomMessageFormatter()
    {
        super();
    }

    @Override
    public String format(LogRecord record)
    {
        Object[] arguments = new Object[3];
        arguments[0] = new Date(record.getMillis());
        arguments[1] = "[Merge.Utility]";
        arguments[2] = record.getMessage() + "\n";
        return messageFormat.format(arguments);
    }

}

@CompileStatic
class UpgradeUtil
{
    /**
     *
     * @param aReader
     * @return
     * @throws IOException
     */
    static String readFileReaderIntoString(Reader aReader)
            throws IOException
    {
        int ch;
        StringBuilder strContent = new StringBuilder("");

        while ((ch = aReader.read()) != -1)
        {
            strContent.append((char) ch);
        }
        aReader.close();

        return strContent.toString();

    }

    /**
     * Checks if a line consists of '#' character only
     *
     * @param content
     * @return
     */
    static boolean isCompleteHashedLine(String content)
    {
        if (UpgradeUtil.isBlank(content)) return false;
        int i;
        for (i = 0; i < content.length(); i++)
        {
            if ((content.charAt(i) != '#' || Character.isWhitespace(content
                    .charAt(i))))
            {
                return false;
            }
        }
        // FIXME this is hacky.better logic?
        if (i >= KMergeConstants.kHeaderDelimitDeterminer)
        {
            return true;
        }
        return false;

    }

    /**
     * TODO use wily stringutil instead implementation of stringUtils.isBlank()
     * Dip- may not do this mentioned above.if it is going to be a standalone
     * jar, then we should avoid dependencies.
     *
     * @param str
     * @return
     */
    static boolean isBlank(String str)
    {
        int strLen;
        if (str == null || (strLen = str.length()) == 0)
        {
            return true;
        }
        for (int i = 0; i < strLen; i++)
        {
            if ((Character.isWhitespace(str.charAt(i)) == false))
            {
                return false;
            }
        }
        return true;
    }

    static String getFileExtension(String filename)
    {
        String ext = null;
        int dotPos = filename.lastIndexOf(".");
        if (0 < dotPos && dotPos <= filename.length() - 2)
        {
            ext = filename.substring(dotPos + 1);
        }

        return ext;
    }
}

@CompileStatic
class AgentDirectoryResolver {
    static boolean isFileAvailableAt(String dirPath, final String fileName){
        File directory = new File(dirPath);
        if (directory.isDirectory()){
            File[] children = directory.listFiles(new FilenameFilter() {
                @Override
                public boolean accept(File dir, String name) {
                    return name.equalsIgnoreCase(fileName);
                }
            });

            return children.length > 0;
        }
        return false;
    }
}

@CompileStatic
class JsonWriter {
    static Gson gson;
    static void WriteObjectToFileAsJson(Object obj, String fileName, String path) throws IOException{
        if (gson == null){
            gson = new Gson();
        }

        String json = gson.toJson(obj);
        writeInternal(json,path+"/"+fileName);

    }

    static void writeInternal(String text, String destination) throws IOException{
        FileWriter writer = null;
        BufferedWriter bufferedWriter = null;
        try
        {
            writer = new FileWriter(destination);
            bufferedWriter = new BufferedWriter(writer);
            writer.write(text);
            writer.flush();
        } catch (IOException e)
        {
            throw e;
        } finally
        {
            if (writer != null)
            {
                try
                {
                    writer.close();
                } catch (IOException e)
                {
                    // ignore exception
                }
            }
            if (bufferedWriter != null)
            {
                try
                {
                    bufferedWriter.close();
                } catch (IOException e)
                {
                    // ignore exception
                }
            }
        }
    }
}

@CompileStatic
abstract class ARule
{

    private MergeableType type;
    private String name;

    ARule(MergeableType type, String name)
    {
        this.type = type
        this.name = name
    }

    MergeableType getType()
    {
        return type;
    }

    /**
     * Returns the name of metadata file or metadata entry in a file.
     *
     * @return name of metadata file or metadata entry in a file
     */
    String getName()
    {
        return name;
    }



    int hashCode()
    {
        int result = 17;

        int hash = (type == null) ? 0 : type.hashCode();
        result = (31 * result) + hash;
        hash = (name == null) ? 0 : name.hashCode();
        result = (31 * result) + hash;

        return result;
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;

       if (obj instanceof ARule)
        {
            ARule otherObj = (ARule) obj;
            if ((type != null && type.equals(otherObj.type))
                    && (name != null && name.equals(otherObj.name)))
            {
                retValue = true;
            }
        }

        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: ARule");
        strBldr.append("\n\tType: " + type).append("\n\tName: " + name);

        return strBldr.toString();
    }
}

@CompileStatic
class RuleDeleteEntry extends ARule
{

    RuleDeleteEntry(MergeableType type, String name)
    {
        super(type, name);
    }

    int hashCode()
    {
        return super.hashCode();
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;

        if (obj instanceof RuleDeleteEntry)
        {
            RuleDeleteEntry otherObj = (RuleDeleteEntry) obj;
            retValue = super.equals(otherObj);
        }

        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: RuleDeleteEntry");
        strBldr.append("\n" + super.toString());

        return strBldr.toString();
    }
}

@CompileStatic
class RuleDeleteFile extends ARule
{
    RuleDeleteFile(MergeableType type, String name)
    {
        super(type, name);
    }

    int hashCode()
    {
        return super.hashCode();
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;

        if (obj instanceof RuleDeleteFile)
        {
            RuleDeleteFile otherObj = (RuleDeleteFile) obj;
            retValue = super.equals(otherObj);
        }

        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: RuleDeleteFile");
        strBldr.append("\n" + super.toString());

        return strBldr.toString();
    }
}


class RuleNoMergeFile extends ARule
{
    RuleNoMergeFile(MergeableType type, String name)
    {
        super(type, name);
    }

    int hashCode()
    {
        return super.hashCode();
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;
        if (obj instanceof RuleNoMergeFile)
        {
            RuleNoMergeFile otherObj = (RuleNoMergeFile) obj;
            retValue = super.equals(otherObj);
        }
        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: RuleNoMergeFile");
        strBldr.append("\n" + super.toString());
        return strBldr.toString();
    }
}


class RuleOverwriteFile extends ARule
{
    RuleOverwriteFile(MergeableType type, String name)
    {
        super(type, name);
    }

    int hashCode()
    {
        return super.hashCode();
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;
        if (obj instanceof RuleOverwriteFile)
        {
            RuleOverwriteFile otherObj = (RuleOverwriteFile) obj;
            retValue = super.equals(otherObj);
        }
        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: RuleOverwriteFile");
        strBldr.append("\n" + super.toString());

        return strBldr.toString();
    }
}


class RulePreferredEntry extends ARule
{
    private PreferenceType preference = PreferenceType.CURRENT;

    RulePreferredEntry(MergeableType type, String name)
    {
        super(type, name);
    }

    RulePreferredEntry(MergeableType type, String name, PreferenceType preference)
    {
        this(type, name);
        this.preference = preference;
    }

    PreferenceType getPreference()
    {
        return preference;
    }

    int hashCode()
    {
        return super.hashCode();
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;

        if (obj instanceof RulePreferredEntry)
        {
            RulePreferredEntry otherObj = (RulePreferredEntry) obj;
            retValue = super.equals(otherObj);
        }

        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: PreferenceEntryRule");
        strBldr.append("\n" + super.toString()).append(
                "\n\tPreference: "
                        + preference);
        return strBldr.toString();
    }
}

@CompileStatic
class RuleRenamedEntry extends ARule{
    private Key renamed;
    RuleRenamedEntry(MergeableType type, String name,
                     String renamed)
    {
        super(type, name);
        this.renamed = new Key(renamed);
    }

    Key getRenamedKey() { return renamed;}

    int hashCode()
    {
        return super.hashCode();
    }

    boolean equals(Object obj)
    {
        boolean retValue = false;

        if (obj instanceof RuleRenamedEntry)
        {
            RuleRenamedEntry otherObj = (RuleRenamedEntry) obj;
            retValue = super.equals(otherObj);
        }

        return retValue;
    }

    String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: RenameEntryRule");
        strBldr.append("\n" + super.toString()).append(
                "\n\tRenamed: "
                        + renamed);
        return strBldr.toString();
    }
}

@CompileStatic
class RulesDTO {
    public List<ARuleDTO> deleteEntryRules = new ArrayList<>()
    public List<RuleRenamedEntryDTO> renamedEntryRules = new ArrayList<>()
    public List<RulePreferredEntryDTO> preferredEntryRules = new ArrayList<>()
    public List<ARuleDTO> noMergeFileRules = new ArrayList<>()
    public List<ARuleDTO> overwriteFileRules = new ArrayList<>()
    public List<ARuleDTO> deleteFileRules = new ArrayList<>()
}

@CompileStatic
class ARuleDTO
{
    public String name
    public String type
}
@CompileStatic
class RuleRenamedEntryDTO
{
    public String name
    public String renamed
    public String type
}
@CompileStatic
class RulePreferredEntryDTO
{
    public String name
    public String preference
    public String type
}

@CompileStatic
class Rules
{
    private Map<MergeableType, Set<RuleDeleteFile>> fDeleteFileRules;
    private Map<MergeableType, Set<RuleOverwriteFile>> fOverwriteFileRules;
    private Map<MergeableType, Set<RuleNoMergeFile>> fNoMergeFileRules;
    private Map<MergeableType, Set<RuleDeleteEntry>> fDeleteEntryRules;
    private Map<MergeableType, Map<String, RulePreferredEntry>> fPreferredEntryRules;
    private Map<MergeableType, Map<IMetadataEntry, RuleRenamedEntry>> fRenameEntryRules;

    public Rules()
    {
        fDeleteFileRules = new HashMap<MergeableType, Set<RuleDeleteFile>>();
        fOverwriteFileRules = new HashMap<MergeableType, Set<RuleOverwriteFile>>();
        fNoMergeFileRules = new HashMap<MergeableType, Set<RuleNoMergeFile>>();
        fDeleteEntryRules = new HashMap<MergeableType, Set<RuleDeleteEntry>>();
        fPreferredEntryRules = new HashMap<MergeableType, Map<String, RulePreferredEntry>>();
        fRenameEntryRules = new HashMap<MergeableType, Map<IMetadataEntry, RuleRenamedEntry>>();
    }

    boolean hasDeleteFileRule(MergeableType type, String name)
    {
        boolean retValue = false;
        Set<RuleDeleteFile> rules = fDeleteFileRules.get(type);
        if (rules != null)
        {
            retValue = rules.contains(new RuleDeleteFile(type, name));
        }

        return retValue;
    }

    void addDeleteFileRule(RuleDeleteFile rule)
    {
        MergeableType type = rule.getType();
        Set<RuleDeleteFile> rules = fDeleteFileRules.get(type);
        if (rules == null)
        {
            rules = new HashSet<RuleDeleteFile>();
            fDeleteFileRules.put(type, rules);
        }
        rules.add(rule);
    }

    boolean hasOverwriteFileRule(MergeableType type, String name)
    {
        boolean retValue = false;
        Set<RuleOverwriteFile> rules = fOverwriteFileRules.get(type);
        if (rules != null)
        {
            retValue = rules.contains(new RuleOverwriteFile(type, name));
        }

        return retValue;
    }

    void addOverwriteFileRule(RuleOverwriteFile rule)
    {
        MergeableType type = rule.getType();
        Set<RuleOverwriteFile> rules = fOverwriteFileRules.get(type);
        if (rules == null)
        {
            rules = new HashSet<RuleOverwriteFile>();
            fOverwriteFileRules.put(type, rules);
        }
        rules.add(rule);
    }

    boolean hasNoMergeFileRule(MergeableType type, String name)
    {
        boolean retValue = false;
        Set<RuleNoMergeFile> rules = fNoMergeFileRules.get(type);
        if (rules != null)
        {
            retValue = rules.contains(new RuleNoMergeFile(type, name));
        }

        return retValue;
    }

    void addNoMergeFileRule(RuleNoMergeFile rule)
    {
        MergeableType type = rule.getType();
        Set<RuleNoMergeFile> rules = fNoMergeFileRules.get(type);
        if (rules == null)
        {
            rules = new HashSet<RuleNoMergeFile>();
            fNoMergeFileRules.put(type, rules);
        }
        rules.add(rule);
    }

    boolean hasDeleteEntryRule(MergeableType type, String name)
    {
        boolean retValue = false;
        Set<RuleDeleteEntry> rules = fDeleteEntryRules.get(type);
        if (rules != null)
        {
            retValue = rules.contains(new RuleDeleteEntry(type, name));
        }

        return retValue;
    }

    void addDeleteEntryRule(RuleDeleteEntry rule)
    {
        MergeableType type = rule.getType();
        Set<RuleDeleteEntry> rules = fDeleteEntryRules.get(type);
        if (rules == null)
        {
            rules = new HashSet<RuleDeleteEntry>();
            fDeleteEntryRules.put(type, rules);
        }
        rules.add(rule);
    }

    RulePreferredEntry getPreferredEntryRule(MergeableType type, String name)
    {
        RulePreferredEntry retValue = null;
        Map<String, RulePreferredEntry> rules = fPreferredEntryRules.get(type);
        if (rules != null)
        {
            retValue = rules.get(name.toLowerCase());
        }

        return retValue;
    }

    void addPreferredEntryRule(RulePreferredEntry rule)
    {
        MergeableType type = rule.getType();
        Map<String, RulePreferredEntry> rules = fPreferredEntryRules.get(type);
        if (rules == null)
        {
            rules = new HashMap<String, RulePreferredEntry>();
            fPreferredEntryRules.put(type, rules);
        }
        rules.put(rule.getName(), rule);
    }

    RuleRenamedEntry getRenameEntryRule(MergeableType type, String name)
    {
        RuleRenamedEntry retValue = null;
        Map<IMetadataEntry, RuleRenamedEntry> rules = fRenameEntryRules.get(type);
        if (rules != null)
        {
            retValue = rules.get(new Key(name.toLowerCase()));
        }

        return retValue;
    }

    Set<IMetadataEntry> getAllRenamedEntryRules(MergeableType type){
        Set<IMetadataEntry> allRenameEntryRules = new HashSet<>();
        if (fRenameEntryRules.get(type) != null){
            allRenameEntryRules = fRenameEntryRules.get(type).keySet();
        }
        return allRenameEntryRules;
    }

    void addRenameEntryRule(RuleRenamedEntry rule)
    {
        MergeableType type = rule.getType();
        Map<IMetadataEntry, RuleRenamedEntry> rules = fRenameEntryRules.get(type);
        if (rules == null)
        {
            rules = new HashMap<IMetadataEntry, RuleRenamedEntry>();
            fRenameEntryRules.put(type, rules);
        }
        rules.put(new Key(rule.getName()),rule);
    }
}

@CompileStatic
class RulesParser
{
    private Rules fRules = new Rules();

    public RulesParser()
    {}

    RulesParser(String rulesFile) throws MergeException
    {
        fRules = new Rules();
        if (rulesFile != null && !rulesFile.isEmpty())
        {
            // check for valid file path
            File file = new File(rulesFile);
            if (!file.exists())
            {
                throw new MergeException("ERROR: The rule file \"" + rulesFile
                        + "\" doesn't exist.");
            }
            parse(rulesFile);
        }
    }

    Rules getRules()
    {
        return fRules;
    }

    protected void parse(String rulesFile)
    {
        String ext = UpgradeUtil.getFileExtension(rulesFile)
        if("json".equalsIgnoreCase(ext)){
            parseAsJson(rulesFile)
        }else{
            //no op.
        }
    }

    private void parseAsJson(String rulesFile){
        Gson gson = new GsonBuilder().create()
        JsonReader reader = new JsonReader(new FileReader(rulesFile));
        RulesDTO rules = gson.fromJson(reader,RulesDTO.class)
        for(ARuleDTO rl: rules.deleteEntryRules){
            RuleDeleteEntry rul = new RuleDeleteEntry(rl.type as MergeableType,rl.name)
            fRules.addDeleteEntryRule(rul)
        }
        for(RulePreferredEntryDTO rl: rules.preferredEntryRules){
            RulePreferredEntry rul = new RulePreferredEntry(rl.type as MergeableType,
                    rl.name,rl.preference as PreferenceType)
            fRules.addPreferredEntryRule(rul)
        }
        for(RuleRenamedEntryDTO rl: rules.renamedEntryRules){
            RuleRenamedEntry rul = new RuleRenamedEntry(rl.type as MergeableType, rl.name,rl.renamed)
            fRules.addRenameEntryRule(rul)
        }
        for(ARuleDTO rl: rules.deleteFileRules){
            RuleDeleteFile rul = new RuleDeleteFile(rl.type as MergeableType, rl.name)
            fRules.addDeleteFileRule(rul)
        }
        for(ARuleDTO rl: rules.overwriteFileRules){
            RuleOverwriteFile rul = new RuleOverwriteFile(rl.type as MergeableType, rl.name)
            fRules.addOverwriteFileRule(rul)
        }
        for(ARuleDTO rl: rules.noMergeFileRules){
            RuleNoMergeFile rul = new RuleNoMergeFile(rl.type as MergeableType, rl.name)
            fRules.addNoMergeFileRule(rul)
        }
    }
}

@CompileStatic
class RulesEngine implements IRulesEngine
{
    private Rules fRules;
    private RulesParser fParser;

    /**
     * Constructs a metadata merge rules engine from the given metadata merge
     * rules xml file.
     *
     * @param rulesFile
     *            merged rules xml file file path
     */
    RulesEngine(String rulesFile) throws MergeException
    {
        fParser = new RulesParser(rulesFile);
        addDefaultRules()
    }

    private void addDefaultRules(){
        fRules = fParser.getRules()
        addDeleteEntryRule("agentManager.credential")
        addDeleteEntryRule("agentManager.url.1")
        addDeleteEntryRule("agentManager.url.2")
        addDeleteEntryRule("acc.package.id")
        addDeleteEntryRule("acc.server.id")
        addDeleteEntryRule("acc.package.name")
        addDeleteEntryRule("acc.package.version")
        addDeleteEntryRule("acc.package.truststore.hash")
    }

    private void addDeleteEntryRule(String name){
        fRules.addDeleteEntryRule(new RuleDeleteEntry(MergeableType.PROFILE,name))
    }

    /**
     * Checks if a current file of given type and name is subjected to deletion.
     *
     * @param type
     *            metadata file type
     * @param fileName
     *            name of the metadata file
     * @return true if the current metadata file should be deleted
     */
    boolean isDelete(MergeableType type, String fileName)
    {
        return fRules.hasDeleteFileRule(type, fileName);
    }

    /**
     * Checks if a current file of given type and name is subjected to be
     * overwritten by the new update metadata file.
     *
     * @param type
     *            metadata file type
     * @param fileName
     *            name of the metadata file
     * @return true if the current metadata file should be overwritten by the
     *         new update metadata file
     */
    boolean isOverwrite(MergeableType type, String fileName)
    {
        return fRules.hasOverwriteFileRule(type, fileName);
    }

    /**
     * Checks if a current file of given type and name is subjected to be merged
     * with the new update metadata file.
     *
     * @param type
     *            metadata file type
     * @param fileName
     *            name of the metadata file
     * @return true if the current metadata file should be merged with the new
     *         update metadata file
     */
    boolean isMerge(MergeableType type, String fileName)
    {

        return !fRules.hasNoMergeFileRule(type, fileName);
    }

    /**
     * Checks if a metadata entry of given type and name from a current file is
     * subjected to deletion in the merged metadata file.
     *
     * @param type
     *            metadata entry type
     * @param fileName
     *            name of the metadata entry
     * @return true if the metadaata entry should be deleted
     */
    boolean isEntryDelete(MergeableType type, IMetadataEntry entry)
    {
        return fRules.hasDeleteEntryRule(type, entry.getKey());
    }

    /**
     * For a given type and name of metadata entry, returns the preferred file
     * type which to be f=given preference during merge.
     *
     * @param type
     *            metadata entry type
     * @param fileName
     *            name of the metadata entry
     * @return the preferred file trype that is given preference during merge
     */
    PreferenceType getPreferredType(MergeableType type,
                                           IMetadataEntry entry)
    {
        PreferenceType preferred = PreferenceType.CURRENT;
        RulePreferredEntry rule = fRules.getPreferredEntryRule(type, entry
                .getKey());
        if (rule != null)
        {
            preferred = rule.getPreference();
            if (preferred == PreferenceType.INVALID)
            {
                preferred = PreferenceType.CURRENT;
            }
        }

        return preferred;
    }

    @Override
    Key getRenamedEntry(MergeableType type, IMetadataEntry entry) {
        Key renamedKey = (Key)entry;
        RuleRenamedEntry rule = fRules.getRenameEntryRule(type, entry.getKey());
        if (rule != null)
        {
            renamedKey = rule.getRenamedKey();
        }

        return renamedKey;
    }

    @Override
    Set<IMetadataEntry> getAllRenamedEntries(MergeableType type) {
        return fRules.getAllRenamedEntryRules(type);
    }
}

@CompileStatic
class Profile implements IMergeable<Profile> {
    private List<Line> fHeader
    private Map<String, Key> fGroupMap
    private Map<Key, List<Value>> fEntryMap
    private Map<Key, List<Value>> fFinalMergedMap
    private List<Key> fKeyOrderList
    private ProfileFileMergeDiff fProfileFileMergeDiff

    Profile() {
        fHeader = []
        fGroupMap = [:]
        fEntryMap = [:]
        fKeyOrderList = []
        fFinalMergedMap = [:]
        fProfileFileMergeDiff = new ProfileFileMergeDiff()
    }

    ProfileFileMergeDiff getProfileDiffObject() {
        return this.fProfileFileMergeDiff
    }

    List<Line> getHeader() {
        return Collections.unmodifiableList(fHeader)
    }

    void setHeader(List<Line> header) {
        fHeader.clear()
        header.each { fHeader.add(it) }
    }

    void add(Key key, Value value) {
        fGroupMap[key.getGroup()] = key
        List<Value> values = fEntryMap.get(key) ?: []
        values.add(value)
        fEntryMap[key] = values
    }

    Set<Key> getKeys() {
        return Collections.unmodifiableSet(fEntryMap.keySet())
    }

    static Profile read(BufferedReader reader) throws IOException {
        Profile profile = new Profile()

        int header = 0
        List<Line> comments = []
        String line = null
        int count = 0
        while ((line = reader.readLine()) != null) {
            count++
            if (UpgradeUtil.isCompleteHashedLine(line)) {
                comments.add(new Line(line, count))
                if (header == 1) {
                    profile.setHeader(comments)
                    comments.clear()
                }
                header++
            } else if (UpgradeUtil.isBlank(line)) {
                comments.add(new Line("", count))
            } else if (line.trim().startsWith("#")) {
                boolean success = isCommentedKey(profile, line, comments, count)
                if (success) {
                    comments.clear()
                } else {
                    comments.add(new Line(line, count))
                }
            } else {
                handleKey(profile, line, comments, false, count)
                comments.clear()
            }
        }

        return profile
    }

    void write(Writer writer) throws IOException {
        writer.write(getProfile())
        writer.flush()
    }


    Profile merge(Profile otherProfile, IRulesEngine engine) {
        Profile profile = new Profile()
        profile.setHeader(otherProfile.fHeader)

        handleRenamedKeys(otherProfile, engine)

        Map<Key, List<Value>> finalMergedMap = new LinkedHashMap<>()
        List<Key> keyOrderList = new LinkedList<>()
        for (Key key : fEntryMap.keySet()) {
            keyOrderList.add(key)
        }

        Set<Key> commonKeys = new HashSet<>()
        for (Key key : fEntryMap.keySet()) {
            // if metadata entry needs to be deleted, do not add
            // and skip rest of this loop
            if (engine.isEntryDelete(MergeableType.PROFILE, key)) {
                keyOrderList.remove(key)
                continue
            }

            // if a metadata entry exist on both sides
            if (otherProfile.fEntryMap.containsKey(key)) {
                commonKeys.add(key)
                PreferenceType type = engine.getPreferredType(MergeableType.PROFILE, key)
                if (type == PreferenceType.CURRENT) {
                    handleCommonKey(key, profile, this, otherProfile, true)
                } else if (type == PreferenceType.UPGRADE) {
                    handleCommonKey(key, profile, otherProfile, this, false)
                }
            } else {
                // if no key exist on right side
                UUID keyId = profile.getProfileDiffObject().generatePropertyId()
                List<Value> values = fEntryMap.get(key)
                for (Value value : values) {
                    profile.add(key, value)
                    profile.getProfileDiffObject().setOldProperty(keyId, key.getKey(), value)
                }
            }
        }

        // take care of the remaining keys
        handleNonCollidingKeys(commonKeys, profile, otherProfile, keyOrderList)

        commonKeys.clear()

        for (Key key : keyOrderList) {
            finalMergedMap.put(key, profile.fEntryMap.get(key))
        }
        profile.fEntryMap = finalMergedMap
        return profile
    }

    /**
     * Utility method that handles the addition of keys which are present in
     * both the sides
     *
     * @param key
     * @param profile
     * @param preferredProfile
     * @param otherProfile
     */
    private void handleCommonKey(Key key,
                                 Profile profile,
                                 Profile preferredProfile,
                                 Profile otherProfile,
                                 boolean overwriteNewFileComments)
    {
        List<Value> lvalues = preferredProfile.fEntryMap.get(key);
        List<Value> rvalues = otherProfile.fEntryMap.get(key);

        if(overwriteNewFileComments){
            ArrayList<Line> comments = new ArrayList<>();
            for(Value rv : rvalues){
                comments.addAll(rv.getComments());
            }
            Value lv1 = lvalues.get(0);
            if(lv1 != null){
                Value overwritten = new Value(lv1.getValue(),lv1.isCommented(),comments,lv1.getLineNumber());
                lvalues.set(0,overwritten);
            }
        }

        // enter all values of the key from left side
        boolean uncommentedKeyFlag = false;
        for (Value value : lvalues)
        {
            profile.add(key, value);
            //fProfileFileMergeDiff.setOldProperty(keyId,key.getKey(),value);
            if (!value.isCommented())
            {
                uncommentedKeyFlag = true;
            }
        }

        // enter remaining non-colliding values of the key from left
        // side
        // TODO need to check for uncommented key state


        Key[] tempkeys = new Key[rvalues.size()];
        Value[] tempValues = new Value[rvalues.size()];
        boolean shouldPropertyAddToMergeDiff=false;
        int counter=0;

        for (Value value : rvalues)
        {
            if (!lvalues.contains(value))
            {
                if (!value.isCommented() && uncommentedKeyFlag)
                {
                    Value v = new Value(value.getValue(),
                            true,
                            new ArrayList<Line>(),
                            value.getLineNumber(),
                            true);

                    profile.add(key,v);

                    tempkeys[counter]=key;
                    tempValues[counter]=v;
                    shouldPropertyAddToMergeDiff=true;
                } else
                {
                    profile.add(key, value);

                    tempkeys[counter]=key;
                    tempValues[counter]=value;
                    shouldPropertyAddToMergeDiff=true;
                }
                counter++;
            }
        }

        if (shouldPropertyAddToMergeDiff){
            java.util.UUID keyId = profile.getProfileDiffObject().generatePropertyId();
            for (Value val : lvalues)
            {
                profile.getProfileDiffObject().setOldProperty(keyId,key.getKey(),val);
            }
            for (int i=0;i<tempkeys.length;i++)
            {
                if(tempkeys[i] != null) {
                    profile.getProfileDiffObject().setNewProperty(keyId, tempkeys[i].getKey(), tempValues[i]);
                }
            }
        }
    }

    /**
     * Utility method that handles the addition of those keys which are present
     * in only in the other profile
     *
     * @param commonKeys
     * @param profile
     * @param otherProfile
     * @param keyOrderList
     */
    private void handleNonCollidingKeys(Set<Key> commonKeys,
                                        Profile profile,
                                        Profile otherProfile,
                                        List<Key> keyOrderList)
    {
        // take care of the remaining keys
        for (Key key : otherProfile.fEntryMap.keySet())
        {
            // key which doesn't exist on left side
            if (!commonKeys.contains(key))
            {

                int index = keyOrderList.indexOf(fGroupMap.get(key.getGroup()));
                if (index != -1)
                {
                    keyOrderList.add(index + 1, key);
                } else
                {
                    keyOrderList.add(key);
                }

                fGroupMap.put(key.getGroup(), key);

                List<Value> values = otherProfile.fEntryMap.get(key);
                for (Value value : values)
                {
                    profile.add(key, value);
                    java.util.UUID keyId = profile.getProfileDiffObject().generatePropertyId();
                    profile.getProfileDiffObject().setNewProperty(keyId,key.getKey(),value);
                }
            }
        }
        commonKeys.clear();
    }

    /**
     * This method should be called to free up any resources after which this
     * profile is no longer going to be used.
     */
    public void clear()
    {
        fGroupMap.clear();
        for (List<Value> values : fEntryMap.values())
        {
            values.clear();
        }
        fEntryMap.clear();
    }

    /**
     * Checks whether a commented line is a key-value entry in commented state
     * or just plain comment. If it's a commented key-value entry, insert it
     * into the profile along with the comments and its commented state.
     *
     * @param profile
     *            profile to which this line belongs
     * @param line
     *            line against which this test is performed
     * @param comments
     *            comments associated with the entry
     * @return true if its a commented key-value entry, false if it's just a
     *         comment
     */
    private static boolean isCommentedKey(Profile profile,
                                          String line,
                                          List<Line> comments,
                                          int lineNumber)
    {
        boolean success = false;
        if (line.trim().startsWith("#"))
        {
            line = line.trim();
            // strip off the # from the beginning
            String modLine = "";
            for (int i = 0; i < line.length(); i++)
            {
                if (line.charAt(i) != '#')
                {
                    modLine = line.substring(i);
                    break;
                }
            }

            modLine = modLine.trim();
            String[] tokens = line.trim().split("=");
            if (tokens.length == 2 || modLine.endsWith("="))
            {
                success = handleKey(profile, modLine, comments, true,lineNumber);
            }
        }

        return success;
    }

    /**
     * Utility method that splits an entry to a key value pair and inserts in
     * the profile.
     *
     * @param profile
     *            profile to which this entry belongs
     * @param line
     *            key-value entry
     * @param comments
     *            comments associated with the entry
     * @param commented
     *            commented state of the entry
     * @return true if the entry is a valid key-value entry
     */
    private static boolean handleKey(Profile profile,
                                     String line,
                                     List<Line> comments,
                                     boolean commented,
                                     int lineNumber)
    {
        boolean success = false;
        String[] tokens = line.trim().split("=");

        if (line.trim().endsWith("=") && tokens.length == 1)
        {
            Key key = new Key(tokens[0]);
            Value value = new Value("", commented, comments,lineNumber);
            profile.add(key, value);
// profile.fKeyOrderList.add(key);
            success = true;
        } else if (tokens.length == 2)
        {
            Key key = new Key(tokens[0]);
            Value value = new Value(tokens[1], commented, comments,lineNumber);
            profile.add(key, value);
// profile.fKeyOrderList.add(key);
            success = true;
        }

        return success;
    }

    public String toString()
    {
        StringBuilder strBldr = new StringBuilder("Class: Profile\n");
        strBldr.append(getProfile());
        return strBldr.toString();
    }

    private String getProfile()
    {
        StringBuilder strBldr = new StringBuilder();
        String newLine = System.getProperty("line.separator");
        for (Line line : fHeader)
        {
            strBldr.append(line).append(newLine);
        }
        for (Key key : fEntryMap.keySet())
        {
            List<Value> values = fEntryMap.get(key);
            if (!values.isEmpty())
            {
                for (Value value : values)
                {
                    List<Line> lines = value.getComments();
                    for (Line line : lines)
                    {
                        strBldr.append(line.getContent()).append(newLine);
                    }
                    if (value.isCommented())
                    {
                        strBldr.append("#");
                    }
                    if (value.isOldValue())
                    {
                        strBldr.append("<oldProperty>");
                        strBldr.append(key.getKey() + "=" + value.getValue());
                        strBldr.append("</oldProperty>").append(newLine);;
                    } else {
                        strBldr.append(key.getKey() + "=" + value.getValue())
                                .append(newLine);
                    }
                }
            }
        }
        return strBldr.toString();
    }

    private void handleRenamedKeys(Profile otherProfile, IRulesEngine engine){
        Set<IMetadataEntry> renamedKeys = engine.getAllRenamedEntries(MergeableType.PROFILE);
        if (renamedKeys != null && !renamedKeys.isEmpty()){
            for (IMetadataEntry key: renamedKeys){
                if (this.fEntryMap.containsKey(key)){
                    Key renamedKey = engine.getRenamedEntry(MergeableType.PROFILE,key);
                    List<Value> toPushAgain = this.fEntryMap.get(key);
                    this.fEntryMap.remove(key);
                    this.fEntryMap.put(renamedKey,toPushAgain);
                }
                if (otherProfile.fEntryMap.containsKey(key)){
                    Key renamedKey = engine.getRenamedEntry(MergeableType.PROFILE,key);
                    List<Value> toPushAgain = otherProfile.fEntryMap.get(key);
                    otherProfile.fEntryMap.remove(key);
                    otherProfile.fEntryMap.put(renamedKey,toPushAgain);
                }
            }
        }
    }
}

@CompileStatic
class Config {
    String key
    String value
    int index
    boolean isCommented
}

@CompileStatic
class ConfigDiff {
    UUID id
    long timeStamp
    String fromRelease
    String toRelease
    List<MergedFileDiff> mergedFiles
}

@CompileStatic
class MergedFileDiff {
    String fileKey
    List<PropertiesMerged> propertiesMerged
}

@CompileStatic
class PropertiesMerged{
    ArrayList<Config> from
    ArrayList<Config> to
}

@CompileStatic
abstract class AFileMergeDiff {
    private String fileKey

    void setFileKey(String fileKey) {
        this.fileKey = fileKey
    }

    String getFileKey() {
        return this.fileKey
    }
}

@CompileStatic
class ProfileFileMergeDiff extends AFileMergeDiff {
    private MergedFileDiff fileDiff
    protected Map<UUID, PropertiesMerged> mergedPropertyMap

    ProfileFileMergeDiff() {
        super()
        fileDiff = new MergedFileDiff()
        this.mergedPropertyMap = [:]
        fileDiff.fileKey = this.getFileKey()
    }

    MergedFileDiff getProfileMergeDiff() {
        this.fileDiff.fileKey = this.getFileKey()
        this.fileDiff.propertiesMerged = []
        for (id in mergedPropertyMap.keySet()) {
            this.fileDiff.propertiesMerged.add(this.mergedPropertyMap[id])
        }

        return this.fileDiff
    }

    UUID generatePropertyId() {
        UUID id = UUID.randomUUID()
        PropertiesMerged propertyDetails = new PropertiesMerged(from: [], to: [])
        this.mergedPropertyMap[id] = propertyDetails
        return id
    }

    void setOldProperty(UUID id, String key, Value value) {
        Config from = new Config(
                key: key,
                value: value.isCommented() ? "#${value.getValue()}" : value.getValue(),
                index: value.getLineNumber(),
                isCommented: value.isCommented()
        )
        PropertiesMerged propertyDetails = this.mergedPropertyMap[id]
        propertyDetails.from.add(from)
        this.mergedPropertyMap[id] = propertyDetails
    }

    void setNewProperty(UUID id, String key, Value value) {
        Config to = new Config(
                key: key,
                value: value.isCommented() ? "#${value.getValue()}" : value.getValue(),
                index: value.getLineNumber(),
                isCommented: value.isCommented()
        )
        PropertiesMerged propertyDetails = this.mergedPropertyMap[id]
        propertyDetails.to.add(to)
        this.mergedPropertyMap[id] = propertyDetails
    }
}

class MergeAdministrator {
    /**
     * This method is used to merge both /core/config and /extension.
     * It is called from acc apmia plugin. If mode=extconfig, then also it
     * is invoked (from java -jar command)
     *
     * @param oldDir
     * @param newDir
     * @param mergeRules
     * @param fromRelease
     * @param toRelease
     */
    static String mergeConfigAndExtensions(String oldDir,
                                                  String newDir,
                                                  String mergeRules,
                                                  String fromRelease,
                                                  String toRelease) {
        Map<String,String> result = new HashMap<>();
        StringBuilder resultMsg = new StringBuilder();
        String oldConfigDir = oldDir+File.separator+"core"+File.separator+"config";
        String newConfigDir = newDir+File.separator+"core"+File.separator+"config";

        try {
            if (oldDir.isEmpty() || newDir.isEmpty() || fromRelease.isEmpty() || toRelease.isEmpty()){
                throw new MergeException("Missing one or more required parameters.");
            }

            if (!AgentDirectoryResolver.isFileAvailableAt(newConfigDir, "IntroscopeAgent.profile")) {
                throw new MergeException("IntroscopeAgent.profile not found. " +
                        "Invalid configuration directory passed-" + newConfigDir);
            }

            if (!AgentDirectoryResolver.isFileAvailableAt(oldConfigDir, "IntroscopeAgent.profile")) {
                throw new MergeException("IntroscopeAgent.profile not found. " +
                        "Invalid configuration directory passed-" + oldConfigDir);
            }

            String baseDir = newDir;
            if (newDir.contains("releases")) {
                Path root = Paths.get(newDir);
                while (root.toString().contains("releases")) {
                    root = root.getParent();
                }
                baseDir = root.toString();
            }

            String configResponse =mergeConfigurations(oldConfigDir, newConfigDir, mergeRules,
                    fromRelease, toRelease, baseDir);
            resultMsg.append(configResponse);

            String oldExtnDir = oldDir + File.separator + "extensions";
            String newExtnDir = newDir + File.separator + "extensions";

            if (!AgentDirectoryResolver.isFileAvailableAt(newExtnDir, "Extensions.profile")) {
                throw new MergeException("Extensions.profile not found. " +
                        "Invalid extensions directory passed-" + newExtnDir);
            }
            if (!AgentDirectoryResolver.isFileAvailableAt(oldExtnDir, "Extensions.profile")) {
                throw new MergeException("Extensions.profile not found. " +
                        "Invalid extensions directory passed-" + oldExtnDir);
            }

            String extnResponse = mergeExtensionProperties(oldExtnDir, newExtnDir, mergeRules,
                    fromRelease, toRelease, baseDir);
            resultMsg.append(extnResponse);

            result.put("Status","0");
            result.put("Message",resultMsg.toString());
            return result.toString();
        }
        catch (Exception ex){
            result.put("Status","-1");
            result.put("Message","Error while performing merge action. Error-"+ex.getMessage());
            return result.toString();
        }
    }

    /**
     * Directly pass /extension directory of old and new agent directory to merge extensions.
     *
     * @param oldExtensionsDir
     * @param newExtensionsDir
     * @param mergeRules
     * @param fromRelease
     * @param toRelease
     * @param baseDir
     * @throws MergeException
     */
    static String mergeExtensionProperties(String oldExtensionsDir,
                                                  String newExtensionsDir,
                                                  String mergeRules,
                                                  String fromRelease,
                                                  String toRelease,
                                                  String baseDir) throws MergeException {
        MergeExtensionsImpl mergeImpl = new MergeExtensionsImpl();
        try {
            IRulesEngine engine = new RulesEngine(mergeRules);
            mergeImpl.setTargetRelease(toRelease);
            mergeImpl.initializeMergeDiff(fromRelease, toRelease);
            mergeImpl.resolveWorkingDirectory(baseDir);
            String response = mergeImpl.mergeExtensions(oldExtensionsDir, newExtensionsDir, engine);
            mergeImpl.cleanup();
            return response;
        }catch (NoMergeCandidatesException e){
            return e.getMessage();
        }
        catch (IOException e) {
            mergeImpl.getLogger().log(Level.WARNING, "Working directory unresolved at " + baseDir);
            String msg = MessageFormat.format("Working directory unresolved. Error - {0}",
                    "\n" + e.getMessage());
            throw new MergeException(msg);
        }catch (MergeException e){
            mergeImpl.cleanup();
            mergeImpl.getLogger().log(Level.WARNING, "Caught exception -"+e.getMessage());
            throw e;
        }
    }

    static String mergeConfigurations(String oldConfigDir,
                                             String newConfigDir,
                                             String mergeRules,
                                             String fromRelease,
                                             String toRelease,
                                             String baseDir)throws MergeException {
        MergeConfigurations mergeImpl = new MergeConfigurations();
        try {
            IRulesEngine engine = new RulesEngine(mergeRules);
            mergeImpl.setTargetRelease(toRelease);
            mergeImpl.initializeMergeDiff(fromRelease, toRelease);
            mergeImpl.resolveWorkingDirectory(baseDir);
            String response = mergeImpl.mergeConfig(newConfigDir, oldConfigDir, engine);
            mergeImpl.cleanup();
            return response;
        } catch (NoMergeCandidatesException e){
            return e.getMessage();
        } catch (IOException e) {
            mergeImpl.getLogger().log(Level.WARNING, "Working directory unresolved at " + baseDir);
            String msg = MessageFormat.format("Working directory unresolved. Error - {0}",
                    "\n" + e.getMessage());
            throw new MergeException(msg);
        }catch (MergeException e){
            mergeImpl.cleanup();
            mergeImpl.getLogger().log(Level.WARNING, "Caught exception -"+e.getMessage());
            throw e;
        }
    }

}


class MergeConfigurations extends AMergeBase{
    MergeConfigurations() {
        super(MergeConfigurations.class.getName());
    }

    String mergeConfig(String updateDir,
                              String configDir,
                              IRulesEngine engine) throws MergeException {
        StringBuilder response=new StringBuilder();

        getLogger().info("Reading configurations from "+updateDir);
        Map<String, String> originalFiles = getConfigFiles(updateDir,false);
        Map<String, String> updateFiles = getConfigFiles(updateDir,true);

        getLogger().info("Reading configurations from existing install"+configDir);
        Map<String, String> customerFiles = getConfigFiles(configDir, false);

        if (updateFiles.isEmpty()){
            getLogger().warning("No files found for merge. Path:"+updateDir);
            throw new NoMergeCandidatesException("No files found for merge. Path:"+updateDir);
        }

        if (customerFiles.isEmpty()){
            getLogger().warning("No files found for merge. Path:"+configDir);
            throw new NoMergeCandidatesException("No files found for merge. Path:"+configDir);
        }

        Map<String, String> customerFilesAfterDelete = new HashMap<String, String>();
        customerFilesAfterDelete.putAll(customerFiles);

        // now check for deletion of customer files
        for (String customerFile : customerFiles.keySet()) {
            String ext = UpgradeUtil.getFileExtension(customerFile);
            if (KMergeConstants.kProfileType.equalsIgnoreCase(ext)
                    && engine.isDelete(MergeableType.PROFILE, customerFile)
                    || KMergeConstants.kPblType.equalsIgnoreCase(ext)
                    && engine.isDelete(MergeableType.PBL, customerFile)
                    || KMergeConstants.kPbdType.equalsIgnoreCase(ext)
                    && engine.isDelete(MergeableType.PBD, customerFile)) {
                File custFile = new File(customerFiles.get(customerFile));
                custFile.delete();
                customerFilesAfterDelete.remove(customerFile);
                getLogger().info("Deleted " + customerFile + " successfully.");
            }

        }
        customerFiles = customerFilesAfterDelete;

        //Maintain the collection of updated files. Not every file from old or new installation is updated.
        Map<String, String> trulyUpdated = new HashMap<String,String>();

        mergeInMemory(engine,updateFiles,customerFiles,trulyUpdated);
        response.append("Merged ");

        for(String s: trulyUpdated.keySet()){
            response.append(s).append(" ");
        }
        response.append("in working directory").append("\n");

        try {
            copyFiles(originalFiles,trulyUpdated);
            response.append("Files written to /core/config.").append("\n");
        }catch (IOException e){
            getLogger().log(Level.WARNING,"Exception while copying files:"+trulyUpdated.keySet());
            rollback();
            String msg = MessageFormat.format(KMergeConstants.kMergeExceptionMessage,"\n" + e.getMessage());
            throw new MergeException(msg);
        }

        try
        {
            JsonWriter.WriteObjectToFileAsJson(this.getMergeDiff(),"config.diff",
                    getLogLocation(this.getWorkingDirectory()).toString());
            response.append("config.diff created.");
        }catch (IOException e){
            getLogger().log(Level.WARNING,"Cannot create config.diff");
            String msg = MessageFormat.format("Failed to create config.diff. Error - {0}",
                    "\n" + e.getMessage());
            throw new MergeException(msg);
        }

        customerFiles.clear();
        updateFiles.clear();
        originalFiles.clear();
        customerFilesAfterDelete.clear();
        customerFiles = null;
        updateFiles = null;
        originalFiles = null;
        customerFilesAfterDelete = null;
        return response.toString();
    }
}

class MergeExtensionsImpl extends AMergeBase{

    MergeExtensionsImpl(){
        super(MergeExtensionsImpl.class.getName());
    }


    private String getArchiveFolderName(Path dirname){
        String fileName = dirname.getFileName().toString();
        if (!fileName.contains("tar.gz")){
            getLogger().log(Level.WARNING, dirname.getFileName()+" is not tar archive");
            return "";
        }
        return getExtensionFolderName(fileName);
    }

    private  String getExtensionFolderName(String fileName){
        int count=0;
        for(int i=fileName.length()-1;;i--){
            if ('-' == fileName.charAt(i)){
                count++;
            }
            if (count == 2){
                return fileName.subSequence(0,i).toString();
            }
        }
    }

    String mergeExtensions(String oldExtensionsDir,
                                  String newExtensionsDir,
                                  IRulesEngine engine) throws MergeException {

        StringBuilder response=new StringBuilder();

        //fetch all bundle.properties from newly installed extensions/*.tar.gz and Extensions.profile as key-val pair.
        //key=folderName without hashcode and version number.
        //value=filePath
        getLogger().info("Reading configurations from "+newExtensionsDir);
        Map<String, String> newInstallationFiles = getConfigFilesFromNewInstall(newExtensionsDir);

        if (newInstallationFiles.isEmpty()){
            getLogger().warning("No files found for merge. Path:"+newExtensionsDir);
            throw new NoMergeCandidatesException("No files found for merge. Path:"+newExtensionsDir);
        }

        //get all bundle.properties from currently installed extensions dir as key-val pair.
        //key,val same as above
        getLogger().info("Reading configurations from "+oldExtensionsDir);
        Map<String, String> existingInstallationFiles = getConfigFilesFromExistingInstall(oldExtensionsDir);

        if (existingInstallationFiles.isEmpty()){
            getLogger().warning("No files found for merge. Path:"+oldExtensionsDir);
            throw new NoMergeCandidatesException("No files found for merge. Path:"+oldExtensionsDir);
        }


        //Maintain the collection of updated files. Not every file from old or new installation is updated.
        Map<String, String> trulyUpdated = new HashMap<String,String>();

        getLogger().info("Initiating extensions merge..");
        int mergeFileCount = mergeInMemory(engine, newInstallationFiles, existingInstallationFiles, trulyUpdated);
        response.append("Merged ").append(mergeFileCount).append(" files in working directory").append("\n");
        getLogger().info(response.toString());

        try {
            //Package back to tar.
            patchConfigFilesInWD(trulyUpdated);
            copyFilesToOriginalInstallation(newExtensionsDir);
            response.append("Files written to ").append(newExtensionsDir).append("\n");
        }catch (IOException e){
            //Rollback here
            getLogger().warning("Error while patching tar or copying updated profiles. Error-"+e.getMessage());
            rollback();
            String msg = MessageFormat.format(
                    "Error while patching tar or copying updated profiles. Error-{0}",
                    "\n" + e.getMessage());
            throw new MergeException(msg);
        }

        try
        {
            JsonWriter.WriteObjectToFileAsJson(this.getMergeDiff(),"extensions-config.diff",
                    getLogLocation(this.getWorkingDirectory()).toString());
            response.append("extensions-config.diff created.");
        }catch (IOException e){
            getLogger().log(Level.WARNING,"Cannot create extensions-config.diff");
            String msg = MessageFormat.format("Failed to create extensions-config.diff. Error - {0}",
                    "\n" + e.getMessage());
            throw new MergeException(msg);
        }

        return response.toString();
    }

    @Override
    protected int mergeInMemory(IRulesEngine engine,
                                Map<String, String> updateFiles,
                                Map<String, String> existingFiles,
                                Map<String, String> trulyUpdated) throws MergeException {
        int mergeFileCount=0;
        for (String updateFile : updateFiles.keySet())
        {
            if (existingFiles.containsKey(updateFile)) {
                try
                {
                    getLogger().info("Checking property difference between "+
                            existingFiles.get(updateFile)+" with "+updateFiles.get(updateFile));

                    Profile profile = mergeProfileNoWrite(existingFiles.get(updateFile),
                            updateFiles.get(updateFile), engine);
                    profile.getProfileDiffObject().setFileKey(updateFile);

                    //Write only if properties have changed.
                    if (!profile.getProfileDiffObject().getProfileMergeDiff().propertiesMerged.isEmpty()) {
                        trulyUpdated.put(updateFile, updateFiles.get(updateFile));
                        this.getMergeDiff().mergedFiles.add(profile.getProfileDiffObject().getProfileMergeDiff());
                        write(profile, updateFiles.get(updateFile));
                        getLogger().info("Merged at "+updateFiles.get(updateFile));
                        mergeFileCount++;
                    }else{
                        getLogger().log(Level.ALL,"No property differences found");
                    }
                }catch (IOException e){
                    getLogger().log(Level.WARNING,"Could not merge: "+updateFile);
                    String msg = MessageFormat.format(KMergeConstants.kMergeExceptionMessage,
                            updateFile,
                            "\n" + e.getMessage());
                    throw new MergeException(msg);
                }
                catch (Exception ex){
                    getLogger().log(Level.WARNING,"Could not merge: "+updateFile);
                    String msg = MessageFormat.format(KMergeConstants.kMergeExceptionMessage,
                            updateFile,
                            "\n" + ex.getMessage());
                    throw new MergeException(msg);
                }
            }
        }
        return mergeFileCount;
    }

    private void copyFilesToOriginalInstallation(String newExtensionsDir) throws IOException {
        final File extnsDir = new File(newExtensionsDir);
        final File[] extnsCollection = extnsDir.listFiles();
        Files.walkFileTree(this.getWorkingDirectory(), new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String fullPath = file.toString();
                if ( KMergeConstants.kExtensionTarGZ.equals(UpgradeUtil.getFileExtension(fullPath))) {
                    Path targetPath = extnsDir.toPath().resolve("deploy").resolve(file.getFileName());
                    Files.move(file, targetPath, StandardCopyOption.REPLACE_EXISTING);
                    addRollbackArtifact(targetPath.toFile());
                    getLogger().info("Copied file: "+file.getFileName()+" to :"+targetPath);
                    return FileVisitResult.SKIP_SUBTREE;

                } else if (fullPath.endsWith(".properties")) {
                    Path parentFolder = file.getParent().getFileName();
                    for (File extractedExtension : extnsCollection) {
                        if (extractedExtension.toString().contains(parentFolder.toString())) {
                            Path targetPath = extractedExtension.toPath().resolve(KMergeConstants.kExtensionPropertyFile);
                            Files.move(file, targetPath, StandardCopyOption.REPLACE_EXISTING);
                            addRollbackArtifact(targetPath.toFile());
                            getLogger().info("Copied file: "+extractedExtension.getName()+" to :"+targetPath);
                            return FileVisitResult.SKIP_SUBTREE;
                        }
                    }

                }
                return FileVisitResult.CONTINUE;
            }
        });
    }


    private void patchConfigFilesInWD(Map<String, String> trulyUpdated) throws IOException {

        for (String fileKey: trulyUpdated.keySet()) {
            final String targetFolderName = fileKey.substring(0,fileKey.lastIndexOf("/"));
            final String filePath = trulyUpdated.get(fileKey);

            try {
                Files.walkFileTree(this.getWorkingDirectory(), new SimpleFileVisitor<Path>() {
                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                        String fullPath = file.toString();
                        if (fullPath.endsWith(".tar.gz.bkp") && fullPath.contains(targetFolderName)) {
                            try {
                                updateConfigInTar(targetFolderName, new File(filePath),file.toFile());
                                return FileVisitResult.SKIP_SUBTREE;
                            } catch (MergeException e) {
                                throw new RuntimeException(e);
                            }
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
            } catch (IOException e) {
                e.printStackTrace();
            }
        }


    }

    /**
     * Fetch bundle.properties and Extensions.profile from newly installed extensions directory.
     * @param newDir
     * @return A Map of "extenstionName/fileName" as key and filePath as Value
     * @throws MergeException
     */
    private Map<String,String> getConfigFilesFromNewInstall(String newDir)
            throws MergeException {

        Map<String, String> result = new HashMap<String, String>();

        File extnsDir = new File(newDir);

        getConfigFromExtensions(extnsDir, result, true);

        //get config from deploy folder. Do not add if folder is already exploded under extensions.
        File deployDir = new File(newDir+"/deploy");
        if (deployDir.isDirectory()){
            File[] extensionsCollection = deployDir.listFiles();
            for(File file : extensionsCollection){
                String folderName = getArchiveFolderName(file.toPath());
                if (folderName.isEmpty()){
                    continue;
                }

                if (!result.containsKey(folderName+"/bundle.properties"))
                {
                    String[] fileDetails = getConfigFileFromTar(file.toString());
                    if (fileDetails[0] != null) {
                        result.put(fileDetails[0], fileDetails[1]);
                        getLogger().info("Loaded "+fileDetails[0]+" from :"+fileDetails[1]);
                    }
                }
            }
        }

        return  result;
    }

    private Map<String,String> getConfigFilesFromExistingInstall(String oldDir)
            throws MergeException {
        Map<String, String> result = new HashMap<String, String>();

        File extnsDir = new File(oldDir);
        if (!extnsDir.isDirectory()){
            throw new MergeException("Not a directory of extensions-"+oldDir);
        }

        getConfigFromExtensions(extnsDir, result, false);
        return  result;
    }

    private void getConfigFromExtensions(File extnsDir,
                                         Map<String, String> result,
                                         boolean copyToWorkingDirectory) throws MergeException {

        //Not in current scope.
        //Look for Extension.profile at root.
//        File[] extnsProfile = searchConfig(extnsDir,false);
//        for (File config : extnsProfile) {
//            result.put(config.getName(), config.getPath());
//        }

        //Search in all extensions folders.
        File[] extensionsCollection = extnsDir.listFiles();
        assert extensionsCollection != null;

        try {
            for (File file : extensionsCollection) {
                if (file.isDirectory()) {
                    File[] fileDetails = searchProfiles(file, true);
                    for (File config : fileDetails) {
                        String folderName = getExtensionFolderName(file.getName());
                        if (folderName.isEmpty()) {
                            continue;
                        }
                        if (copyToWorkingDirectory) {
                            Path target = this.getWorkingDirectory().resolve(folderName);
                            Files.createDirectories(target);

                            target = target.resolve(config.getName());
                            Files.deleteIfExists(target);
                            Files.copy(config.toPath(), target);
                            result.put(folderName + "/" + config.getName(), target.toString());
                            takeBackup(folderName,config);
                            getLogger().info("Loaded "+config.getName()+" from :"+target);
                            continue;
                        }
                        result.put(folderName + "/" + config.getName(), config.getPath());
                        getLogger().info("Loaded "+config.getName()+" from :"+config.getPath());
                    }
                }
            }
        }catch (IOException e){
            throw new MergeException("Unable to fetch config from extensions.");
        }
    }

    private String[] getConfigFileFromTar(String dirPath)
            throws MergeException{
        String[] result = new String[2];

        File dir = new File(dirPath);
        if (!dir.exists())
        {
            throw new MergeException(
                    KMergeConstants.kNewline
                            + "ERROR: The archive \""
                            + dirPath
                            + "\" given as input doesn't exist.Please check the input.");
        }


        try{
            //Create extension name directory in .acc
            //For acc-controller-windows-6fb819dcxt1-99.99.0.0.tar.gz, create acc-controller-windows in .acc
            String folderName = getArchiveFolderName(dir.toPath());
            Path target = this.getWorkingDirectory().resolve(folderName);
            Files.createDirectories(target);

            //Backup the existing tar file before operating on it.
            File backupFile = takeBackup(folderName,dir);

            //Open backup file for reading bundle.properties
            TarArchiveInputStream tar = new TarArchiveInputStream(new GZIPInputStream(new FileInputStream(backupFile)));
            ArchiveEntry entry;
            while((entry =tar.getNextEntry())!=null)
            {
                if (KMergeConstants.kExtensionPropertyFile.equals(entry.getName())){
                    target = target.resolve(entry.getName());
                    Files.deleteIfExists(target);
                    Files.copy(tar, target);
                    result[0] = folderName+"/"+entry.getName();
                    result[1] = target.toString();
                    break;
                }
            }

            tar.close();
        }
        catch (FileNotFoundException e) {
            getLogger().log(Level.WARNING,"File is not a valid tar.gz : "+dir);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }

    private void updateConfigInTar(String folderForBackup, File file, File tarName) throws MergeException {

        if (!tarName.exists())
        {
            throw new MergeException(
                    KMergeConstants.kNewline
                            + "ERROR: The archive \""
                            + tarName.toString()
                            + "\" given as input doesn't exist.Please check the input.");
        }

        if (!file.exists()){
            throw new MergeException(
                    KMergeConstants.kNewline
                            + "ERROR: The file \""
                            + file.toString()
                            + "\" given as input doesn't exist.Please check the input.");
        }

        String fileName = file.getName();

        try {
            FileInputStream fs = new FileInputStream(file);
            File tmpFile = File.createTempFile("tmp", null);

            TarArchiveInputStream tarInputStream = new TarArchiveInputStream(
                    new GZIPInputStream(
                            new FileInputStream(tarName)));

            TarArchiveOutputStream tarOutputStream = new TarArchiveOutputStream(
                    new GZIPOutputStream(
                            new FileOutputStream(tmpFile)));
            TarArchiveEntry entry;

            // Step 4: Iterate through entries in the original .tar archive
            while ((entry = tarInputStream.getNextTarEntry()) != null) {
                // Step 5: Check if the current entry is the file to update
                if (entry.getName().equals(fileName)) {
                    // Step 6: Update the content of the file
                    TarArchiveEntry newEntry = new TarArchiveEntry(entry.getName());
                    newEntry.setSize(file.length());
                    tarOutputStream.putArchiveEntry(newEntry);
                    IOUtils.copy(fs, tarOutputStream);
                    tarOutputStream.closeArchiveEntry();
                } else {
                    // Step 7: Copy the original entry to the new .tar archive
                    tarOutputStream.putArchiveEntry(entry);
                    IOUtils.copy(tarInputStream, tarOutputStream);
                    tarOutputStream.closeArchiveEntry();
                }
            }

            tarInputStream.close();
            fs.close();
            tarOutputStream.close();

            //remove .bkp from existing file name and create new file in working directory.
            String targetTar = tarName.toString();
            targetTar = targetTar.substring(0,targetTar.length()-4);
            Files.move(tmpFile.toPath(), new File(targetTar).toPath(), StandardCopyOption.REPLACE_EXISTING);
            getLogger().info("Patched "+targetTar);
        } catch (FileNotFoundException e) {
            getLogger().log(Level.WARNING,"File is not a valid tar.gz : "+tarName);
        } catch (IOException e) {
            getLogger().log(Level.WARNING,"Unable to take backup of "+tarName);
            throw new MergeException("Unable to take backup of "+tarName,e);
        }
    }
}

@CompileStatic
abstract class AMergeBase {
    private static Logger sLogger;
    private Path fWorkingDirectory;
    private ConfigDiff fMergeDiff;
    private String fTargetRelease;
    private List<File> fRollbackArtifacts;
    private HashMap<String, File> fBackedUpArtifacts;

    AMergeBase(String className){
        setupLogger(className);
    }

    private void setupLogger(String className){
        sLogger = Logger.getLogger(className);
        MergeCustomMessageFormatter formatter = new MergeCustomMessageFormatter();
        FileHandler fileHandler = null;
        ConsoleHandler consoleHandler = null;
        try
        {
            fileHandler = new FileHandler(KMergeConstants.kMergeLogFile);
            fileHandler.setFormatter(formatter);
            fileHandler.setLevel(Level.ALL);
            sLogger.addHandler(fileHandler);

            consoleHandler = new ConsoleHandler();
            consoleHandler.setFormatter(formatter);
            consoleHandler.setLevel(Level.ALL);
            sLogger.addHandler(consoleHandler);

            sLogger.setUseParentHandlers(false);

        } catch (Exception e)
        {
            // TODO eat the exception now.
        }
    }

    protected Logger getLogger(){
        return sLogger;
    }

    protected void setTargetRelease(String targetRelease){
        this.fTargetRelease = targetRelease;
    }

    protected String getTargetRelease(){
        return this.fTargetRelease;
    }

    /**
     * Utility method to read any type of config file and return its
     * corresponding object instance via IMergable interface.
     *
     * @param reader
     * @param ext
     * @return
     * @throws IOException
     */
    protected IMergeable readConfig(BufferedReader reader, String ext)
            throws IOException
    {
        IMergeable config = null;
        if (KMergeConstants.kProfileType.equalsIgnoreCase(ext))
        {
            config = Profile.read(reader);
        } //else if (KMergeConstants.kPblType.equalsIgnoreCase(ext))
//        {
//            config = PBL.read(reader);
//        } else if (KMergeConstants.kPbdType.equalsIgnoreCase(ext))
//        {
//            config = PBD.read(reader);
//        }
        return config;
    }

    protected boolean shouldMerge(String updateFile, String ext, IRulesEngine engine) {
        return KMergeConstants.kProfileType.equalsIgnoreCase(ext)
                && engine.isMerge(MergeableType.PROFILE, updateFile)
                || KMergeConstants.kPblType.equalsIgnoreCase(ext)
                && engine.isMerge(MergeableType.PBL, updateFile)
                || KMergeConstants.kPbdType.equalsIgnoreCase(ext)
                && engine.isMerge(MergeableType.PBL, updateFile);
    }

    protected boolean shouldOverwrite(String updateFile, String ext, IRulesEngine engine) {
        return KMergeConstants.kProfileType.equalsIgnoreCase(ext)
                && engine.isOverwrite(MergeableType.PROFILE, updateFile)
                || KMergeConstants.kPblType.equalsIgnoreCase(ext)
                && engine.isOverwrite(MergeableType.PBL, updateFile)
                || KMergeConstants.kPbdType.equalsIgnoreCase(ext)
                && engine.isOverwrite(MergeableType.PBD, updateFile);
    }

    /**
     * Returns a map of configuration files with key being the lower cased
     * config file name, while the value being the file path.
     *
     * @param dirPath
     *            directory location
     * @return map of config files, with key being the lower cased config file
     *         name, while the value being the file path.
     * @throws MergeException
     */
    protected Map<String, String> getConfigFiles(String dirPath)
            throws MergeException
    {
        Map<String, String> result = new HashMap<String, String>();

        File dir = new File(dirPath);
        if (!dir.exists())
        {
            throw new MergeException(
                    KMergeConstants.kNewline
                            + "ERROR: The directory \""
                            + dirPath
                            + "\" given as input doesn't exist.Please check the input.");
        }
        if (dir.isDirectory())
        {
            File[] children = searchConfig(dir,false);

            for (File child : children)
            {
                result.put(child.getName().toLowerCase(), child.getPath());
            }
        }

        return result;
    }



    /**
     * Returns a map of configuration files with key being the lower cased
     * config file name, while the value being the file path.
     *
     * @param dirPath
     *            directory location
     * @return map of config files, with key being the lower cased config file
     *         name, while the value being the file path.
     * @throws MergeException
     */
    protected Map<String, String> getConfigFiles(String dirPath, boolean takeBakup)
            throws MergeException
    {
        Map<String, String> result = new HashMap<String, String>();

        File dir = new File(dirPath);
        if (!dir.exists())
        {
            throw new MergeException(
                    KMergeConstants.kNewline
                            + "ERROR: The directory \""
                            + dirPath
                            + "\" given as input doesn't exist.Please check the input.");
        }
        if (dir.isDirectory())
        {
            File[] children = searchProfiles(dir,false);

            try {
                for (File config : children) {
                    if (takeBakup) {
                        //result.put(folderName + "/" + config.getName(),)
                        Path target = this.getWorkingDirectory().resolve("config");
                        Files.createDirectories(target);

                        target = target.resolve(config.getName());
                        Files.deleteIfExists(target);
                        Files.copy(config.toPath(), target);
                        result.put(dir.getName() + "/" + config.getName().toLowerCase(), target.toString());
                        takeBackup("config", config);
                        getLogger().info("Loaded " + config.getName() + " from :" + target);
                        continue;
                    } else {
                        result.put(dir.getName() + "/" + config.getName().toLowerCase(), config.getPath());
                    }
                }
            }catch (IOException ex){
                getLogger().log(Level.WARNING,"Error while reading configuration files: "+ex.getMessage());
                throw new MergeException("Error while reading configuration files: "+ex.getMessage());
            }
        }

        return result;
    }

    protected File[] searchConfig(File dir, final boolean lookProperties) {
        File[] children = dir.listFiles(new FileFilter() {

            public boolean accept(File pathname)
            {
                boolean retValue = false;
                if (pathname.isFile())
                {
                    String ext = UpgradeUtil.getFileExtension(pathname.getName());
                    if (KMergeConstants.kProfileType.equalsIgnoreCase(ext)
                            || KMergeConstants.kPblType.equalsIgnoreCase(ext)
                            || KMergeConstants.kPbdType.equalsIgnoreCase(ext)
                            || (lookProperties && KMergeConstants.kPropertiesType.equalsIgnoreCase(ext)))
                    {
                        retValue = true;
                    }

                }
                return retValue;
            }
        });
        return children;
    }

    //Removed .pbl and .pbd for now.
    protected File[] searchProfiles(File dir, final boolean lookProperties) {
        File[] children = dir.listFiles(new FileFilter() {

            public boolean accept(File pathname)
            {
                boolean retValue = false;
                if (pathname.isFile())
                {
                    String ext = UpgradeUtil.getFileExtension(pathname.getName());
                    if (KMergeConstants.kProfileType.equalsIgnoreCase(ext)
                            || (lookProperties && KMergeConstants.kPropertiesType.equalsIgnoreCase(ext)))
                    {
                        retValue = true;
                    }

                }
                return retValue;
            }
        });
        return children;
    }


    protected int mergeInMemory(IRulesEngine engine,
                                Map<String, String> updateFiles,
                                Map<String, String> existingFiles,
                                Map<String, String> trulyUpdated) throws MergeException {
        int mergeFileCount=0;
        for (String updateFile : updateFiles.keySet())
        {
            if (existingFiles.containsKey(updateFile)) {
                String ext = UpgradeUtil.getFileExtension(updateFile);
                try {
                    // check for overwrite conditions and after that merge
                    // condition for all the three types of config files
                    if (shouldOverwrite(updateFile, ext, engine)) {
                        BufferedReader overWriteReader = new BufferedReader(
                                new FileReader(
                                        updateFiles
                                                .get(updateFile)));

                        IMergeable ovreWrittenConfig = readConfig(overWriteReader,
                                ext);

                        write(ovreWrittenConfig, existingFiles.get(updateFile));


                        getLogger().info("Overwritten " + updateFile
                                + " successfully.");

                    } else if (shouldMerge(updateFile, ext, engine)) {
                        getLogger().log(Level.ALL, "Attempting to merge " +
                                existingFiles.get(updateFile) + " with " + updateFiles.get(updateFile));

                        Profile profile = mergeProfileNoWrite(existingFiles.get(updateFile),
                                updateFiles.get(updateFile), engine);
                        profile.getProfileDiffObject().setFileKey(updateFile);

                        //Write only if properties have changed.
                        if (!profile.getProfileDiffObject().getProfileMergeDiff().propertiesMerged.isEmpty()) {
                            trulyUpdated.put(updateFile, updateFiles.get(updateFile));
                            this.getMergeDiff().mergedFiles.add(profile.getProfileDiffObject().getProfileMergeDiff());
                            write(profile, updateFiles.get(updateFile));
                            getLogger().log(Level.ALL, "Merged file written in working directory at " + updateFiles.get(updateFile));
                            mergeFileCount++;
                        } else {
                            getLogger().log(Level.ALL, "No property differences found");
                        }
                    }
                }catch (IOException e){
                    e.printStackTrace()
                    getLogger().log(Level.WARNING,"Could not merge: "+updateFile);
                    String msg = MessageFormat.format(KMergeConstants.kMergeExceptionMessage,
                            updateFile,
                            "\n" + e.getMessage());
                    throw new MergeException(msg);
                }
                catch (Exception ex){
                    ex.printStackTrace()

                    getLogger().log(Level.WARNING,"Could not merge: "+updateFile);
                    String msg = MessageFormat.format(KMergeConstants.kMergeExceptionMessage,
                            updateFile,
                            "\n" + ex.getMessage());
                    throw new MergeException(msg);
                }
            }
        }
        return mergeFileCount;
    }

    protected void copyFiles(Map<String, String> originalFiles,
                             Map<String, String> trulyUpdated) throws IOException {
        for(String file: trulyUpdated.keySet()) {
            if (originalFiles.containsKey(file)) {
                File toCopy = new File(trulyUpdated.get(file));
                File targetFile = new File(originalFiles.get(file));
                Files.move(toCopy.toPath(), targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                this.addRollbackArtifact(targetFile);
                getLogger().info("Copied file: "+toCopy.getName()+" , to path: "+targetFile);
            }
        }
    }



    /**
     * Writes a mergeable config to a file.
     *
     * @param config
     *            the config to be written to a file
     * @param destination
     *            the destination file path
     * @throws IOException
     */
    protected void write(IMergeable<?> config, String destination)
            throws IOException
    {
        FileWriter writer = null;
        BufferedWriter bufferedWriter = null;
        try
        {
            writer = new FileWriter(destination);
            bufferedWriter = new BufferedWriter(writer);
            config.write(bufferedWriter as Writer);
        } catch (IOException e)
        {
            throw e;
        } finally
        {
            if (writer != null)
            {
                try
                {
                    writer.close();
                } catch (IOException e)
                {
                    // ignore exception
                }
            }
            if (bufferedWriter != null)
            {
                try
                {
                    bufferedWriter.close();
                } catch (IOException e)
                {
                    // ignore exception
                }
            }
        }
    }

    /**
     * This method takes the paths of customer and update profiles and merges
     * one with the other, and returns the merged profile.
     *
     * @param oldInstallPath
     *            The full path of the file inside config dir
     * @param fUpdateFilePath
     *            The full path of the file inside update dir
     * @throws IOException
     */
    protected Profile mergeProfileNoWrite(String oldInstallPath,
                                          String fUpdateFilePath,
                                          IRulesEngine engine) throws IOException {
        Profile mergedConfig = null;

        BufferedReader rightReader = new BufferedReader(new FileReader(fUpdateFilePath));
        Profile updateConfig = Profile.read(rightReader);
        rightReader.close();

        BufferedReader leftReader = new BufferedReader(new FileReader(oldInstallPath));
        Profile customerConfig = Profile.read(leftReader);
        leftReader.close();
        mergedConfig = customerConfig.merge(updateConfig, engine);
        return mergedConfig;
    }

    protected ConfigDiff initializeMergeDiff(String fromRelease, String toRelease){
        this.fMergeDiff = new ConfigDiff();
        this.fMergeDiff.id = java.util.UUID.randomUUID();
        this.fMergeDiff.timeStamp = System.currentTimeMillis();
        this.fMergeDiff.fromRelease = fromRelease;
        this.fMergeDiff.toRelease = toRelease;
        this.fMergeDiff.mergedFiles = new LinkedList<>();

        return this.fMergeDiff;
    }

    protected ConfigDiff getMergeDiff(){
        return this.fMergeDiff;
    }
    protected void resolveWorkingDirectory(String baseDir) throws IOException {
        if (baseDir.isEmpty()) {
            this.fWorkingDirectory = new File(System.getProperty("user.dir")).toPath();
        } else {
            //Create .acc directory in baseDir
            File outputDir = new File(baseDir);
            this.fWorkingDirectory = outputDir.toPath().resolve(KMergeConstants.kWorkingDirectoryName);
            if (!Files.exists(this.fWorkingDirectory)) {
                Files.createDirectory(this.fWorkingDirectory);
            }
        }
    }

    protected Path getWorkingDirectory(){
        if (this.getTargetRelease().isEmpty()) {
            return this.fWorkingDirectory;
        }
        return this.fWorkingDirectory.resolve(this.getTargetRelease());
    }

    protected Path getLogLocation(Path workingDirectory) throws IOException{
        if (!Files.exists(workingDirectory.resolve(KMergeConstants.kMergedOutputFolderName))
                || !Files.isDirectory(workingDirectory.resolve(KMergeConstants.kMergedOutputFolderName))) {
            return Files.createDirectory(workingDirectory.resolve(KMergeConstants.kMergedOutputFolderName));
        }
        return workingDirectory.resolve(KMergeConstants.kMergedOutputFolderName);
    }

    protected File takeBackup(String folderToPatch, File fileToBackup) throws IOException {

        String bkpFilePath = this.getWorkingDirectory().
                resolve(folderToPatch).
                resolve(fileToBackup.getName()).toString() +".bkp"
        File bkpFile = new File( bkpFilePath);

        Files.deleteIfExists(bkpFile.toPath());
        Files.copy(fileToBackup.toPath(), bkpFile.toPath());

        if (this.fBackedUpArtifacts == null){
            this.fBackedUpArtifacts = new HashMap<>();
        }

        this.fBackedUpArtifacts.put(fileToBackup.toString(),bkpFile);
        return bkpFile;
    }

    protected void rollback() {
        getLogger().warning("Merge was unsuccessful.Rollback requested.");
        if (this.fRollbackArtifacts == null) {
            getLogger().warning("No artifacts found for rollback");
            return;
        }
        if (this.fBackedUpArtifacts == null) {
            getLogger().warning("Back up artifacts not found");
            return;
        }
        for (File artifact : fRollbackArtifacts) {
            if (this.fBackedUpArtifacts.containsKey(artifact.toString())) {
                File originalFile = this.fBackedUpArtifacts.get(artifact.toString());
                try {
                    Files.move(originalFile.toPath(), artifact.toPath(), StandardCopyOption.REPLACE_EXISTING);
                    getLogger().warning("Reverted merge operation on file "+artifact);
                } catch (IOException e) {
                    getLogger().warning("Error during rollback :"+e);
                }
            }
        }
    }

    protected void addRollbackArtifact(File artifact){
        if (this.fRollbackArtifacts == null){
            this.fRollbackArtifacts = new ArrayList<>();
        }
        this.fRollbackArtifacts.add(artifact);
    }

    protected List<File> getRollbackArtifacts(){
        return this.fRollbackArtifacts;
    }

    protected void cleanup() throws MergeException {
        try {
            final Path wd = this.getWorkingDirectory();
            Files.walkFileTree(this.getWorkingDirectory(), new SimpleFileVisitor<Path>() {
                @Override
                FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (!file.toString().contains(KMergeConstants.kMergedOutputFolderName)) {
                        Files.delete(file); // Delete files
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (!(KMergeConstants.kMergedOutputFolderName.equals(dir.toFile().getName()) ||
                            wd.compareTo(dir)==0)) {
                        Files.delete(dir); // Delete directories
                    }
                    return FileVisitResult.CONTINUE;
                }
            });

        } catch (IOException e) {
            throw new MergeException("Error during cleanup. Error- "+e);
        }
    }

}

//String customerDir="/root/Downloads/apmia_old_v1/apmia"
//String upgradeDir="C:\\Users\\aj895306\\Downloads\\new-apmia\\apmia\\releases\\99.99.0.0"
//String fromRelease="10.8"
//String toRelease="22.3"
//
//String response = MergeAdministrator.mergeConfigAndExtensions(customerDir,
//        upgradeDir,
//        mergeRulePath,
//        fromRelease,
//        toRelease);
//System.out.println(response);
