
/*
 * Copyright (c) 2025 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */


import groovy.json.JsonBuilder
import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import java.nio.file.DirectoryStream
import java.nio.file.Files
import java.nio.file.NoSuchFileException
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.StandardCopyOption
import java.util.zip.ZipEntry
import java.util.zip.ZipFile

@Slf4j
@CompileStatic
class ACCException extends Exception {
    String errno
    String[] arguments
 
    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }
    
    void setParams(String ... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }
    
    def getAccError() {
        return ["errno":this.errno, "arguments":arguments]
    }
}

@CompileStatic
class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

@CompileStatic
class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom.toAbsolutePath().normalize().toString())
    }
}

@CompileStatic
class ACCWriteAccessException extends ACCException {
    
    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

@CompileStatic
class ACCNoEntityException extends ACCException {
    
    ACCNoEntityException(String msg, enitity) {
        super(msg, "ENOENT")
        setParams(enitity.toString())
    }
}

@CompileStatic
class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

@CompileStatic
class ACCFrameworkException extends ACCException {
    ACCFrameworkException(String msg, String info) {
        super(msg, "EFRAMEWORKERROR")
        setParams(info)
    }
}


@Slf4j
@CompileStatic
class FilePuller  {

    /**
     * Fetch the URL and write the file to a temporary file before moving
     * it atomically into place to avoid half-written files etc
     */
    void httpGet(URL url, Path to) {
    
        // Write the file to a temporary file first
        Path temp = Paths.get(to.toString() + ".tmp")

        InputStream is
        try {
            is = url.openConnection().getInputStream()
        }
        catch (IOException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }

        try {
            Files.deleteIfExists(temp)
            Files.copy(is, temp)
            Files.move(temp, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
        }
        catch (NoSuchFileException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
        finally {
            try {
                Files.deleteIfExists(temp)
            }
            catch (Exception ignore) {
                log.info("Failed to delete $temp")
            }

            is.close()
        }
    }

    /**
     * Copy (or pull) file from the broker
     * and write locally
     */
    void copyBrokerFile(URL broker, String from, Path to) {
        
        URL url = new URL(broker.toString() + from)
        
        log.info("copyBrokerFile: $url -> $to")
        httpGet(url, to)
    }
}

@Slf4j
@CompileStatic
class Archive {

    Path archive

    Archive(Path archive) {
        this.archive = archive
    }

    Path untar(Path outDir, Path toExtract) {
        log.info("untarring $toExtract from $archive to $outDir")

        ProcessBuilder pb = new ProcessBuilder("tar", "xf", "$archive", toExtract.toString())

        pb.directory(outDir.toFile())
        pb.redirectErrorStream(true)

        log.info("command is {}", pb.command())

        Process proc = pb.start()

        int rc = proc.waitFor()
        
        log.info("rc is $rc")

        def output = proc.text
        
        if (rc != 0) {
            log.debug("command output was $output")
        }

        Path dest = outDir.resolve(toExtract)

        if (!Files.exists(dest)) {
            throw new ACCNoEntityException("Failed to untar $toExtract", output)
        }

        return dest
    }

    Path unzip(Path outDir, Path toExtract) {
        log.info("unzipping $toExtract from $archive to $outDir")
        
        ZipFile zipFile = new ZipFile(archive.toFile())
        
        ZipEntry entry = zipFile.getEntry(toExtract.toString().replace("\\", "/"))

        if (entry == null) {
            log.warn("Failed to find file in zip with /.  Re-trying with \\")
            entry = zipFile.getEntry(toExtract.toString().replace("/", "\\"))
        }
        
        if (entry == null) { 
            throw new ACCNoEntityException("Failed to unzip $toExtract", "$toExtract was not found in zip file")
        }
        else {
            Path dest = outDir.resolve(toExtract)
            Files.createDirectories(dest.getParent())
            Files.deleteIfExists(dest)

            // Unzip the file
            Files.copy(zipFile.getInputStream(entry), dest)

            return dest
        }

        return null
    }

    /**
     * tar/zip agnostic unpacker
     */
    Path unpack(Path outDir, Path toExtract) {
        Path of

        String fn = archive.getFileName().toString()
        if (fn.endsWith(".tar")) {
            of = untar(outDir, toExtract)
        }
        else if (fn.endsWith(".zip")) {
            of = unzip(outDir, toExtract)
        }
        else {
            log.warn("Unknown archive extension for $fn.  Will try to untar it")
            of = untar(outDir, toExtract)
        }

        return of
    }
}


@Slf4j
@CompileStatic
class ControllerUpgrader {

    Path tempDir

    Path archive
    Path upgradeJar, packageArchive
    
    URL broker
    String partialUrl
    Path accDir

    /*
     * Don't do anything that can fail in the constructor
     */
    ControllerUpgrader(URL broker, String partialUrl, Path accDir) {
        this.broker = broker
        this.accDir = accDir
        this.partialUrl = partialUrl
    }

    void initEnv() {

        log.info("Running on: {}", System.getProperty("os.name"))

        tempDir = Paths.get(System.getProperty("java.io.tmpdir")).resolve("CA/acc/plugins")
    
        if (accDir == null) {
            accDir = getAccDir()
        }

        log.info("accDir: {}", this.accDir.toAbsolutePath().normalize())

        if (partialUrl == null) {
            this.partialUrl = getPartialUrl()
        }

        setBroker((broker != null) ? broker : readBrokerUrl())

        log.info("broker: {}", this.broker)
        log.info("partialUrl: {}", this.partialUrl)
        log.info("tempDir: {}", this.tempDir)

        archive = myTemp("download").resolve(Paths.get(partialUrl).getFileName())
    }

    void setBroker(URL broker) {
        String b = broker.toString()
        if (!b.endsWith("/")) {
            b += "/"
        }
        this.broker = new URL(b)        
    }

    Path tryAccDir(Path candidate) {
        DirectoryStream<Path> ds;
        try {
            ds = Files.newDirectoryStream(candidate.resolve("bin"), "apmccctrl-*.ver")
            for (Path path : ds) {
                log.info("Found {}", path)
                return candidate
            }
        }
        catch (NoSuchFileException ignore) {
            log.debug("$ignore")
        }
        finally {
            if (ds != null) {
                ds.close()
            }
        }

        return null
    }
    
    /**
     * Derive ACC base directory from the cwd
     */
    Path getAccDir() {
        log.info("Looking for ACC install signature")
        
        Path cwd = Paths.get(".").toAbsolutePath().normalize()

        Path acc = tryAccDir(cwd)

        if (acc == null) {
            acc = tryAccDir(cwd.getParent())
        }

        if (acc != null) {
            return acc
        }

        log.warn("Could not derive ACC base path properly. Falling back to using {}", cwd.toAbsolutePath().normalize())
        return cwd 
    }

    Path makeAccPath(String other) {
        return (other != null) ? accDir.resolve(other) : accDir
    }

    /**
     * Read the given properties file to get the broker url
     * (configurationServer.url=thebrokername)
     */
    URL readBrokerUrl() {

		String serverUrl = "com.ca.apm.acc.controller.configurationServer.url";
		String serverUrlShort = "configurationServer.url";
		
        String brokerString = System.getenv(serverUrl);
        if (brokerString != null) {
            log.info("Found " + serverUrl + " in system environment variables: $brokerString")
        } else {
            Path propertiesFilename = makeAccPath("config/apmccctrl.properties")
            FileInputStream is

            log.info("Reading " + serverUrl + " from $propertiesFilename")
            try {
                // Get the broker URL from the properties file.
                Properties controllerProperties = new Properties()

                is = new FileInputStream(propertiesFilename.toFile())
                controllerProperties.load(is)
                brokerString = controllerProperties.getProperty(serverUrl)
				
				if(brokerString == null)
					brokerString = controllerProperties.getProperty(serverUrlShort);
            }
            catch (IOException e) {
                throw new ACCReadAccessException("Failed to read config", propertiesFilename)
            }
            finally {
                if (is != null) {
                    is.close()
                }
            }
        }

        if (!brokerString.endsWith("/")) {
            brokerString += "/"
        }

        return new URL(brokerString)
    }
   
    boolean isWin() {
        return System.getProperty("os.name").toLowerCase(Locale.US).contains("windows")
    }
    
    /**
     * Derive the name to the package  - it could be a zip or tar depending on what platform
     * we are running on.
     */
    String getPartialUrl() {

        String packagePath = "package/acc-controller-package"

        String sys = System.getProperty("os.name")

        String ext = isWin() ? ".zip" : ".tar"

        return packagePath + ext
    }

    void run(String ... args) throws IOException  {

        log.info("Running $args")

		new Timer().schedule( {
			ProcessBuilder pb = new ProcessBuilder(args)
			log.info("command is {}", pb.command())

			/*
			 * Ensure we set the working directory of the process to something other than
			 * the controller directory, to avoid the upgrade failing
			 * due to the controller dir having a handle open on it
			 * This is especially critical for Windows.
			 */
			pb.directory(tempDir.toFile())

			Process proc = pb.start()
		} as TimerTask, 2000)
    }

    Path extractUpgradeTool(Path archive) {

        Path upgradeTool = Paths.get("APMCommandCenterController/lib/acc-upgradetool.jar")

        // This is where the upgradetool.jar is unpacked to
        Path outDir = myTemp("download")

        return new Archive(archive).unpack(outDir, upgradeTool)
    }

    /**
     *  Find local tar
     */
    String whichTar() {
        String TAR_PATH="which tar".execute().getText().trim()

        if (!TAR_PATH) {
            TAR_PATH="/bin/tar"
            log.warn("Could not find path to tar binary, assuming $TAR_PATH")
        }

        return TAR_PATH 
    }

    Path controllerUpgrade() {

        initEnv()

        // Pull the package file from the broker
        FilePuller fp = new FilePuller()
        fp.copyBrokerFile(broker, partialUrl, archive)

        // extract upgradetool.jar out of the file
        upgradeJar = extractUpgradeTool(archive)

        String JAVA_HOME = System.getProperty("java.home")
        log.info("JAVA_HOME is $JAVA_HOME")

        // Now need to spawn the upgrade tool with our jvm details. 
        // perhaps the upgrade script should do that. At least we have the option of being updated
        // or whatever
        String INSTALL_DIR = makeAccPath(null).toAbsolutePath().normalize().toString()
        
        if (isWin()) {
            
            run("$JAVA_HOME\\bin\\java.exe",
                "-DLOG_LEVEL=INFO",
                "-DINSTALL_DIR=$INSTALL_DIR",
                "-DVALIDATE_RUNS_CORRECTLY",
                "-DREMOVE_ARCHIVE",
                "-jar", upgradeJar.toAbsolutePath().toString(),
                archive.toAbsolutePath().toString())
        }
        else {
            String TAR_PATH = whichTar()

            run("$JAVA_HOME/bin/java",
                "-DLOG_LEVEL=INFO",
                "-DTAR_PATH=$TAR_PATH",
                "-DINSTALL_DIR=$INSTALL_DIR",
                "-DVALIDATE_RUNS_CORRECTLY",
                "-DREMOVE_ARCHIVE",
                "-jar", upgradeJar.toAbsolutePath().toString(),
                archive.toAbsolutePath().toString())
        }

        // Note, am relying on the CS will wait for plugins to finish running before shutting down.
    }

    
    Path myTemp(String otherDir) {
        Path temp = tempDir.resolve(otherDir)
        Files.createDirectories(temp)
        return temp
    }

    def handler() {
        def errs = []

        try {
            controllerUpgrade()
        }
        catch (ACCException e) {
            errs.push(e.getAccError())
            cleanUp()
        }
        catch (Exception e) {
            /*
             * An exceptional exception occurred, so log an error
             */

            log.error("Caught Exception $e")
            e.printStackTrace()

            errs.push(new ACCFrameworkException("Internal Error", e.getMessage()).getAccError())
            cleanUp()
        }

        String response = genJson(errs)

        log.debug("Response: $response")

        return response
    }

    @CompileDynamic
    String genJson(errs) {
        // Create the json message
        def json = new JsonBuilder()
    
        def root = json {
            version "1.0"
            className "KeyValuePairs"
            errors errs
        }

        return json.toString()
    }
    
    def cleanUp(Path file) {
        try {
            if (file != null) {
                log.info("cleanup: removing $file")
                Files.deleteIfExists(file)
            }
        }
        catch (IOException e) {
            log.debug("Failed to delete $e")
        }
    }

    def cleanUp() {
        cleanUp(archive)
        cleanUp(upgradeJar)
        cleanUp(tempDir)
    }
}



if (binding.variables.containsKey("request")) {
    logger.info("Running controllerUpgrade.groovy")
    upgrade = new ControllerUpgrader(null, null, null)
    response = upgrade.handler()
}
else {

    /*
     * To run in test mode, a logging implementation has to be supplied on the cp, 
     * so for example call with something like this:
     * 
     * groovy -cp /usr/share/java/slf4j-simple.jar:/usr/share/java/slf4j-api.jar 
     *     -Dorg.slf4j.simpleLogger.defaultLogLevel=debug ./controllerUpgrade.groovy 
     *         http://hilja07-linux3:80 acc-controller-package-99.99.accEagle-SNAPSHOT.tar /home/hilja07/temp/acc/APMCommandCenterController/
     */

    println("Running controllerUpgrade.groovy - TEST MODE")

    Path accDir
    String partialUrl
    URL broker
    try {
        accDir = Paths.get(args[0])      // acc install dir
        partialUrl = Paths.get(args[1])  // partial path to upgraded controller package - catted to broker url
        broker = new URL(args[2])
    }
    catch (ArrayIndexOutOfBoundsException e) {
    }

    upgrade = new ControllerUpgrader(broker, partialUrl, accDir)

    response = upgrade.handler()

    println("Response is:")    
    println(response)

}





