/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */


import com.ca.apm.acc.messaging.plugins.ZippedContent
import com.ca.apm.acc.plugin.PluginException
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths

import static groovy.io.FileType.FILES

@Slf4j(value = "LOGGER", category = "com.ca.apm.acc.plugin.GetAgentLog")
class GetAgentLog {
    static def run(final String request) {
        try {
            return worker(request);
        } catch (final Exception ex) {
            throw new PluginException("GetAgentLog", ex);
        }
    }

    private static def worker(final String request) {
        def response;
        def logFiles = [];

        def inputs = new JsonSlurper().parseText(request);
        String logFilename = inputs["log4j.appender.logfile.File"];
        if (logFilename==null) {
            logFilename = inputs["introscope.agent.logfile"];
        }
        String logDirProp = inputs["introscope.agent.log.directory"];
        String userDir = inputs["user.dir"];
        String agentProfile = inputs["com.wily.introscope.agentProfile"];
        final Path profilePath = Paths.get(agentProfile != null ? agentProfile : "");
        Path logDir = null;

        if (logFilename != null) {
            LOGGER.debug("Log filename {}", logFilename);
            /*
             * The logfile can have a property explicitly in the name
             * such as /tmp/introscope.${name.to.use}.log.
             * Replace properties with values, use empty string if property is
             *  undefined, the same as agent does.
             */
            def parts = logFilename =~ /\$\{[A-Za-z0-9.]*}/;
            parts.each {
                String p ->
                    def lookup = p.replaceAll(/\$\{|\}/, "");
                    LOGGER.debug("Property ${lookup} = ${inputs[lookup]}");
                    String replacement = inputs[lookup] ?: "";
                    logFilename = logFilename.replace(p, replacement);
            }
            LOGGER.debug("New filename: $logFilename");

            logFilename = getDefaultNaming(logFilename, inputs);
            logFilename = getAbsolutePath(Paths.get(logFilename), profilePath, Paths.get(userDir != null ? userDir : ""));
            if (logFilename != null) {
                logDir = Paths.get(logFilename).getParent().toAbsolutePath();
            }

        }
		/*
		 * logDirProp will take preference when provided
		 */
		if (logDirProp != null ) {
            LOGGER.debug("Log directory {}", logDirProp);
            logDir = Paths.get(getAbsolutePath(Paths.get(logDirProp), profilePath, Paths.get(userDir != null ? userDir : ""))).toAbsolutePath();

        }
        if (logDir == null) {
            throw new PluginException("Cannot identify agent log dir - neither log4j.appender.logfile.File nor introscope.agent.log.directory is defined");
        }

        // gather log files recursively from the given directory
        LOGGER.debug("Fetching all files from $logDir");
        if (logDir.toFile().exists()) {
            logDir.toFile().eachFileRecurse(FILES) {
                if (it.name.toString() ==~ /.*\.log([.][0-9]*)?$/ && !(it.name.toString() ==~ /.*apmccctrl.log.*/)) {
                    getFileDetails(logFiles, it.absolutePath);
                }
            }
        }

        // Create the json message
        // default constructor must be used for backward compatibility with 11.x
        final ZippedContent zippedContent = new ZippedContent();
        zippedContent.setVersion("1.0");
        zippedContent.setClassName("ZippedContent");
        zippedContent.setFiles(logFiles);
        def json = new JsonBuilder(zippedContent);
        response = json.toString()
        return response
    }

/*
 * Get all file details needed for a log file
 */
    static def getFileDetails(files, filename) {
        LOGGER.info("Fetching " + filename);
        def openFile = Paths.get(filename).toFile();
        if (!openFile.exists()) {
            LOGGER.info("File $openFile does not exist or is not accessible");
        }
        def aFile = [:];
        aFile.filePath = filename as String;
        aFile.fileType = "log4j.appender.logfile.File";

        /* 'permissioned denied' gives 0 modified time. */
        if (openFile.lastModified() != 0) {
            aFile.modified = openFile.lastModified();
        }

        /* actual file content is automatically added by the controller. */
        files.add(aFile);
    }

/*
 * logfile and autoprobe share common 'renaming'
 */
    static String getDefaultNaming(filename, inputs) {
        String logNamingOffFlag = inputs["introscope.agent.disableLogFileAutoNaming"];
        String agentNamingFlag = inputs["introscope.agent.agentAutoNamingEnabled"];
        String agentName = inputs["acc.agent.name"];
        String appender = inputs["introscope.agent.agentNameSystemPropertyKey"];
        String wilyName = inputs["com.wily.introscope.agent.agentName"];
        String infix = inputs[appender];

        LOGGER.debug("appender=${appender}");
        LOGGER.debug("infix=${infix}");
        LOGGER.debug("wilyName=${wilyName}");

        /* Simplest case - log autonaming is disabled */
        if (isTrue(logNamingOffFlag)) {
        }
        /* Next case - 'system property' is used to name file */
        else if (infix != null) {
            /* non "A-Za-z0-9" gets translated to '_'  */
            infix = infix.replaceAll(/\W/, "_");
            filename = appendLogName(filename, infix);

        }
        /* Next case - agent autonaming is 'on' in which case use 'agent name' */
        else if (isTrue(agentNamingFlag)) {
            agentName = agentName.replaceAll(/\W/, "_");
            filename = appendLogName(filename, agentName);
        }
        /* Next case - 'com.wily.introscope.agent.agentName' is set. Still use agentName. */
        else if (wilyName != null) {
            agentName = agentName.replaceAll(/\W/, "_");
            filename = appendLogName(filename, agentName);
        }
        return filename;
    }

    static boolean isTrue(value) {
        return value ? value.toBoolean() : false;
    }

/*
 * Add string at end of filename, before extension
 */
    static String appendLogName(file, inf) {
        Path p = Paths.get(file);
        String fn = p.getFileName();
        int li = fn.lastIndexOf(".");
        fn = (li >= 0) ? (fn.substring(0, li) + "." + inf + "." + fn.substring(li + 1)) : (fn + "." + inf);
        String parent = p.getParent();
        file = parent ? Paths.get(parent, fn) : Paths.get(fn);
        return file;
    }

    /**
     * Determine full path to a relative file. Can be relative to agent profile itself.
     * <p>
     * If relative path then it's relative to {@code IntroscopeAgentProfile.properties}. If that's
     * also relative, then resolve that relative to {@code user.dir} (startup dir of app server)
     * and resolve from there.
     *
     * @param findThis absolute or relative path of a file we are looking for
     * @param profile absolute or relative path of the {@code IntroscopeAgentProfile.properties} file.
     * @param userDir Agent installation directory, e.g., the {@code wily} directory
     */
    static String getAbsolutePath(Path findThis, Path profile, Path userDir) {
        if (findThis.isAbsolute()) {
            return findThis.toString();
        }

        // File is not absolute. So resolve it against profile path first.

        if (!profile.isAbsolute()) {
            // Profile path is not absolute either. Resolve it against the user dir.
            profile = userDir.resolve(profile);
        }

        // Resolve file against profile parent directory.

        def profileParent = profile.getParent();
        if (profileParent != null) {
            def thisRelativeToProfileDir = profileParent.resolve(findThis);
            if (Files.exists(thisRelativeToProfileDir)) {
                return thisRelativeToProfileDir.toString();
            }
        }

        // Try to find the file against the user dir.
        return userDir.resolve(findThis).toAbsolutePath().toString();
    }
}

def handler() {
    if (binding.variables.containsKey("request")) {
        logger.info("Running GetAgentLog.groovy");

        handler = new GetAgentLog();
        response = handler.run(request);
        return response;
    }
}

handler();
