/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

logger.debug("Running getPBDs.groovy")

errs = []

def addIOError(java.io.File file) {
    try {
        if (java.nio.file.Files.isDirectory(file.toPath())) {
            java.nio.file.Files.newDirectoryStream(file.toPath()).close();
        } else {
            java.nio.file.Files.readAllBytes(file.toPath())
        }
    } catch (java.nio.file.NoSuchFileException nsfe) {
        anErr = ["errno":"ENOENT", "arguments":[file.toString()]]
        errs.add(anErr)
        logger.debug(nsfe.getMessage())
        return true
    } catch (java.nio.file.AccessDeniedException ade) {
        anErr = ["errno":"EACCESS", "arguments":[file.toString()]]
        errs.add(anErr)
        logger.debug(ade.getMessage())
        return true
    } catch (java.io.IOException ioe) {
        anErr = ["errno":"EIOERROR", "arguments":[file.toString()]]
        errs.add(anErr)
        logger.debug(ioe.getMessage())
        return true
    }
    return false
}

/*
 * Return extension from filename
 */
def getExtension(filename) {
    return filename.lastIndexOf(".").with {it != -1 ? filename.substring(it) : ""}
}

def getDirectives(File agentProfile) {

    // Read directives list from agent profile
    def props = new Properties()
    try {
        agentProfile.withInputStream {
            stream -> props.load(stream)
        }
    } catch (java.io.IOException ioe) {
        addIOError(agentProfile)
        return null
    }

    pbdpbl = props["introscope.autoprobe.directivesFile"]?.trim()
    
    logger.debug("Get directives file introscope.autoprobe.directivesFile=" + pbdpbl);

    if (pbdpbl == null || pbdpbl == "") {
        // Malformed IntroscopeAgent.profile - missing introscope.autoprobe.directivesFile property
        anErr = ["errno":"MISSINGDIRECTIVESFILEPROPERTY", "arguments":[agentProfile.toString()]]
        errs.add(anErr)
        return null
    }

    // Split line on comma and trim trailing spaces off each item
    return pbdpbl.split(",")*.trim()
}

/*
 * Expand a PBL, returning PBDs
 */
def expandPBL(File pbl) {

    pbls_ret = []

    try {
        pbl.eachLine {

            if (it == null || it.isEmpty() ||
            it[0] == "#" || it.contains(":") ||
            getExtension(it) != ".pbd") {
                logger.debug("Ignoring $it")
            }
            else {

                /*
                 * Check if the pbd contains a macro denoted by ${MACRO} e.g.
                 * ${JVMCOMPATIBILITY}.  If it does, replace this with a wildcard
                 * so that all matching pbds are returned.
                 */

                sublist = []
                it2 = it.replaceAll('\\$\\{.*\\}', '.*')
                if (it.equals(it2)) {
                    logger.trace("No macro expansion");
                    sublist.add(it)
                }
                else {
                    logger.debug("pbl macro $it converted to $it2")
                    pbl.getParentFile().eachFileMatch(~/$it2/) {
                        logger.debug("expanded $it")
                        sublist.add(it.toString())
                    }
                }

                for (sub in sublist) {
                    pbd = new File(sub)

                    if (pbd.isAbsolute()) {
                        full_path = pbd
                    }
                    else {
                        full_path = new File(pbl.getParentFile(), pbd.toString())
                    }

                    logger.debug("adding $full_path")

                    pbls_ret.add(full_path)
                }
            }
        }
    } catch (java.io.IOException e) {
        // Just catch exception here and do not any errors at this point
    }

    return pbls_ret
}

/*
 * Expand a directory of PBDs
 */
def expandDirectory(File dir_path) {
    files = []
    if (!addIOError(dir_path)) {
        dir_path.eachFile{ (getExtension(it.toString()) == ".pbd") ? files.add(it) : null}
    }
    return files
}

def expandPBD(File basedir, directives) {

    pbdpbls = []

    // Loop over each directive found in the profile
    for (directive_s in directives) {
        File full_path

        directive = new File(directive_s)

        ext = getExtension(directive_s)

        if (directive.isAbsolute()) {
            full_path = directive
        }
        else {
            full_path = new File(basedir, directive.toString())
        }

        if (ext == ".pbd") {
            logger.debug("Have direct pbd: $directive")
            pbdpbls.add(full_path)
        }
        else if (ext == ".pbl") {
            logger.debug("Expanding pbl: $directive")

            // Also include the pbl itself
            pbdpbls.add(full_path)
            pbdpbls += expandPBL(full_path)
        }
        else {
            logger.debug("Expanding directory: $directive")
            pbdpbls += expandDirectory(full_path)
        }
    }

    return pbdpbls
}

def getPBDs(File agentProfile) {

    pbdpbl = getDirectives(agentProfile)
    
    values_expanded = []
    if (pbdpbl) {
        values_expanded = expandPBD(agentProfile.getParentFile(), pbdpbl)
        // Remove duplicates from the expanded values
        values_expanded.unique()
        values_expanded.sort()
    }

    return values_expanded
}

def handler() {

    // Parse JSON Request
    inputs = new groovy.json.JsonSlurper().parseText(request)

    agentType = inputs["acc.agent.type"]

    pbd_details = []

    if (agentType != null && ((agentType == "ep-agent") || (agentType == "uia"))) {
        logger.info("Nothing to do for agent type $agentType")
    }
    else {
        agentProfile = new File(inputs.get("com.wily.introscope.agentProfile"))

        userDir = new File(inputs.get("user.dir"))

        if (!agentProfile.isAbsolute()) {
            logger.debug("Expand relative path for $agentProfile")
            agentProfile = new File(userDir, agentProfile.toString())
        }

        logger.debug("Agent profile: $agentProfile")

        pbds = getPBDs(agentProfile)

        logger.debug("Expanded values: $pbds")

        for (pbd in pbds) {
            if (!addIOError(pbd)) {
                aFile = ["filePath": pbd.toString(),
                    "fileType": "com.wily.introscope.pbd",
                    "fileContent": null,
                    "modified": pbd.lastModified(),
                    "size": pbd.size()]
                pbd_details.add(aFile)
            }
        }
    }

    // Create the json message
    json = new groovy.json.JsonBuilder()
    root = json {
        version "1.0"
        className "ZippedContent"
        files pbd_details
        errors errs
    }

    response = json.toString()

}

handler()
