/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

import com.ca.apm.acc.plugin.PluginException
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j

import java.nio.file.Files
import java.nio.file.Paths

@Slf4j(value = "LOGGER", category = "com.ca.apm.acc.plugin.JBossVersion")
class JBossVersion {
    static def run(final String request) {
        try {
            return worker(request);
        } catch (final Exception ex) {
            throw new PluginException("JBossVersion", ex);
        }
    }

    private static def worker(final String request) {
        def inputs = new JsonSlurper().parseText(request);

        def r = [:];
        r.version = "1.0";
        r.className = "KeyValuePairs";

        def jboss = inputs["java.class.path"]

        /* Look for jboss in the class path */
        if (jboss != null && jboss.contains("jboss-modules.jar")) {

            def kv=[:]

            try {
                /* It's there - now find directory with jboss-modules in it */
                def parts = jboss.split(inputs["path.separator"]);

                for (def i = 0; i < parts.length; i++) {
                    if (parts[i].contains("jboss-modules")) {
                        def jarFileDir = new File(parts[i]).getParent();
                        LOGGER.info("JBoss discovered at $jarFileDir");

                        kv["acc.agent.appserver.name"] = "JBoss";
                        kv["acc.agent.appserver.version"] = "unknown";

                        if (fileExists(jarFileDir)) {

                            /*
                             * Look in <dir>\modules\system\layers\base\org\jboss\as\product\eap\dir\META-INF\MANIFEST.MF
                             * -> JBoss-Product-Release-Version:
                             * That's where 'jboss-cli version' gets it from.
                             * If that file does not exist then try the same file under wildfly-full product.
                             * If that file also does not exist then it's an older JBoss or different wildfly.
                             * In which case wild-card find
                             *  <dir>\modules\org\jboss\as\process-controller\main\jboss-as-process-controller-<*>.jar
                             *  or
                             *  <dir>\modules\org\jboss\as\process-controller\main\wildfly-process-controller-<*>.jar
                             * ->JBossAS-Release-Version
                             *  (This is where org.jboss.as.version fetches the data)
                             */
                            def path = Paths.get(jarFileDir, "/modules/system/layers/base/org/jboss/as/product/eap/dir/META-INF/MANIFEST.MF");
                            def prefix = "JBoss ";
                            LOGGER.debug("path=$path");
                            if (!fileExists(path.toString())) {
                                path = Paths.get(jarFileDir, "/modules/system/layers/base/org/jboss/as/product/wildfly-full/dir/META-INF/MANIFEST.MF");
                                prefix = "";
                            }
                            if (fileExists(path.toString())) {
                                def f = path.toFile();
                                def p = new Properties();
                                def str = new FileInputStream(f)
                                p.load(str);
                                str.close();

                                kv["acc.agent.appserver.name"] = prefix + p.get("JBoss-Product-Release-Name");
                                kv["acc.agent.appserver.version"] = p.get("JBoss-Product-Release-Version");
                            } else {
                                LOGGER.debug("Did not exist, trying alternate method");
                                // This is an older release or WildFly.
                                def ctrlPath = Paths.get(jarFileDir, "/modules/org/jboss/as/process-controller/main");
                                def jarFile = "jboss-as-process-controller-*.jar";
                                def appServerName = "JBoss AS";
                                if (!fileExists(ctrlPath.toString()))
                                {
                                    // WildFly
                                    ctrlPath = Paths.get(jarFileDir, "/modules/system/layers/base/org/jboss/as/process-controller/main/");
                                    jarFile = "wildfly-process-controller-*.jar";
                                    appServerName = "WildFly";
                                }
                                def fileList = Files.newDirectoryStream(ctrlPath, jarFile);
                                LOGGER.debug("fileList=$fileList");
                                for (def it = fileList.iterator(); it.hasNext(); ) {
                                    def aJar = it.next();
                                    LOGGER.debug("aJar=" + aJar);
                                    def p = loadPropertiesFromJAR(aJar.toFile().getPath());
                                    kv["acc.agent.appserver.name"] = appServerName;
                                    kv["acc.agent.appserver.version"] = p.get("JBossAS-Release-Version");
                                }
                                fileList.close();
                            }
                        }
                    }
                }
                r.properties = kv;
            } catch(e) {
                LOGGER.error("Error while determining JBoss version {}", e)
                def err = new Error();
                err.errno = "EAPPSERVER";
                err.arguments = ["JBoss", e as String]
                r.errors = [err];
            }
        } else {
            // No JBoss JARs found.
            r.version = "1.0";
            r.className = "KeyValuePairs";
            r.properties = []
        }

        def json = new JsonBuilder(r);
        def response = json.toString();
        return response;
    }

    static loadPropertiesFromJAR(jarFile) {
        def props = new Properties()
        def jar = null
        try {
            jar = new java.util.zip.ZipFile(jarFile)
            props.load(jar.getInputStream(jar.getEntry("META-INF/MANIFEST.MF")))
        } finally {
            if (jar) {
                jar.close()
            }
        }
        return props
    }

/*
 * Java File.exists() and Files.exists() both
 * return 'false' for a 'permission denied' error.
 */
    static fileExists(file) {
        try {
            def tmp = Files.readAllBytes(new File(file).toPath());
            LOGGER.debug("$file read in");
            return true;
        } catch(e) {
            if (e instanceof java.nio.file.AccessDeniedException) {
                return true;
            }
            if (e instanceof java.nio.file.NoSuchFileException) {
                return false;
            }
            if (e instanceof java.io.IOException) {
                return true;
            }
        }
        return false;
    }

    static class Error {
        String errno;
        String[] arguments;
    }
}

def handler() {
    if (binding.variables.containsKey("request")) {
        logger.info("Running JBossVersion.groovy");

        handler = new JBossVersion();
        response = handler.run(request);
        return response;
    }
}

handler();
