/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j

import java.nio.file.AccessDeniedException
import java.nio.file.Files
import java.nio.file.NoSuchFileException
import java.util.zip.ZipFile

@Slf4j
class TomcatInfo {

    /**
     * strip embedded version number from Tomcat ServerInfo. Assumes that value is separated with a '/'
     */
    static def stripVersionNumber(s) {
        s.substring(0, (s + "/").indexOf('/'))
    }

    /**
     * Load Properties File from JAR
     */
    static def loadPropertiesFromJAR(jarFile, pathToProperties) {
        Properties props = new Properties()
        def jar = null
        try {
            jar = new ZipFile(jarFile)
            props.load(jar.getInputStream(jar.getEntry(pathToProperties)))
        } finally {
            if (jar) {
                jar.close()
            }
        }
        return props
    }

    /**
     * Java File.exists() and Files.exists() both
     * return 'false' for a 'permission denied' error.
     */
    static def fileExists(file) {
        try {
            Files.readAllBytes(new File(file).toPath())
            log.debug(file + " read in")
            return true
        } catch (AccessDeniedException e) {
            log.debug("Had " + e + " exception for " + file)
            return true
        } catch (NoSuchFileException e) {
            log.debug("File $file not found")
            return false
        }
        log.debug("file does not exist")
        return false
    }

    /**
     * make a JSON structure containing the Response message and return as a String
     */
    static def makeResponse(cb, errs = [], si = null, sn = null, sb = null) {
        def json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: [
                        "acc.agent.appserver.name"   : si,
                        "acc.agent.appserver.version": sn,
                        "acc.agent.appserver.built"  : sb
                ],
                errors: errs
        )
        return json.toString()
    }

    /**
     * make a JSON structure containing the Response message and return as a String
     */
    static def makeErrorResponse(errs = []) {
        def json = new JsonBuilder()
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: [],
                errors: errs
        )
        json.toString()
    }

    def handler(request) {
        // Parse JSON Request
        def inputs = new JsonSlurper().parseText(request)
        // Make path to Tomcat catalina.jar
        def path = inputs["catalina.home"]

        def json = new JsonBuilder()
        // Default response if we do not find any signs of Tomcat.
        json(
                version: "1.0",
                className: "KeyValuePairs",
                properties: []
        )
        def response = json.toString()

        if (path != null) {
            String arg = "$path/lib/catalina.jar"

            try {
                // Older versions have jar here
                log.debug("Trying " + arg)
                if (!fileExists(arg)) {
                    arg = "$path/server/lib/catalina.jar"

                    log.debug("Trying " + arg)
                }

                // Only extract info if catalin.jar exists.
                if (fileExists(arg)) {

                    // Load ServerInfo properties from catalina.jar and extract version information

                    log.info("Tomcat discovered at {}", arg)
                    Properties props = loadPropertiesFromJAR(new File(arg), "org/apache/catalina/util/ServerInfo.properties")
                    def name = stripVersionNumber(props.get("server.info"))
                    response = makeResponse(inputs["catalina.base"], [], name, props.get("server.number"), props.get("server.built"))
                } else {
                    log.debug("Does not exist ?")
                }
            } catch (Exception e) {
                log.error("Error while loading properties from JAR", e)
                def errors = []
                def anErr = ["errno": "EAPPSERVER", "arguments": ["Tomcat", e.toString()]]
                errors.add(anErr)
                response = makeErrorResponse(errors)
            }
        }
        log.debug("Tomcat detection response: ${response}")
        return response
    }
}

response = new TomcatInfo().handler(request)
