/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

import com.ca.apm.acc.plugin.PluginException
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j

@Slf4j(value = "LOGGER", category = "com.ca.apm.acc.plugin.GetCommandLine")
class GetCommandLine {
    static def run(final String request) {
        try {
            return worker(request);
        } catch (final Exception ex) {
            throw new PluginException("GetCommandLine", ex);
        }
    }

    private static def worker(final String request) {
        def response;

        def inputs = new JsonSlurper().parseText(request);
        String pid = inputs["acc.agent.pid"] as String;
        String os = inputs["os.name"] as String;

        LOGGER.info("Pid=$pid  os=$os");

        def r = [:];
        r["version"] = "1.0"
        r["className"] = "KeyValuePairs"

        def errors = [];
        def cmdline = getCommandLine(os, pid, errors);

        def kv = [:]
        if (cmdline != "unknown") {
            kv["acc.agent.commandline"] = new String(cmdline);
        }

        r.properties = kv;
        r.errors = errors;
        def json = new JsonBuilder(r);
        response = json.toString()
        return response;
    }

    static getCommandLine(os, pid, errors) {
        def commands = [:];
        boolean matchedOS = false;
        String lastLine = "unknown";

        /*
         * These are the command lines to run on a particular OS to get the command line
         * for a given PID.
         * The mapping is from Java's "os.name" property to a command. This is matched via a
         * regular expression. So "Windows.*" matches "Windows Server 2000" etc.
         */
        commands["Windows.*"] = "wmic.exe PROCESS where \"ProcessId = '" + pid + "'\" get Commandline";
        commands["Linux"] = "ps -p " + pid + " -o command=\"\"";

        /* Force version of 'ps' to use */
        commands["HP-UX"] = "UNIX95= ps -p " + pid + " -o args=\"\"";
        commands["SunOS"] = "ps -p " + pid + " -o args=\"\"";
        commands["Solaris"] = commands["SunOS"];

        /* Avoid truncation */
        commands["AIX"] = "/bin/sh COLUMNS=9128 ps -p " + pid + " -o args=\"\"";

        for (cmdEntry in commands) {
            def key = cmdEntry.key;
            LOGGER.debug("Key is $key");

            if (os ==~ key) {
                matchedOS = true;
                def command = cmdEntry.value;
                LOGGER.debug("MATCHED with $command");

                def p = java.lang.Runtime.getRuntime().exec(command);
                p.getOutputStream().close();
                p.getErrorStream().close();

                def s = new java.util.Scanner(p.getInputStream());
                /* Return last line of output in case there are headings */
                while (s.hasNextLine()) {
                    def curr = s.nextLine();
                    if (curr != "")
                        lastLine = curr;
                }

                s.close();
                p.waitFor();
                p.destroy();
                break;
            }
        }

        /* Remove leading/trailing space */
        lastLine = lastLine.trim();

        /* Either OS is not supported or PID not found */
        if (matchedOS && (lastLine == "unknown" || lastLine == "")) {
            LOGGER.info("Process ID {} not found", pid);
            def error = new Error();
            error.errno = "ESRCH";
            error.arguments = [pid];
            errors.add(error);
        } else if (!matchedOS) {
            LOGGER.info("OS not supported")
            def error = new Error();
            error.errno = "EUNSUPPORTED";
            error.arguments = [os];
            errors.add(error);
        }
        return lastLine;
    }

    static class Error {
        String errno;
        String[] arguments;
    }
}

def handler() {
    if (binding.variables.containsKey("request")) {
        logger.info("Running GetCommandLine.groovy");

        handler = new GetCommandLine();
        response = handler.run(request);
        return response;
    }
}

handler();
