/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

import com.ca.apm.acc.plugin.PluginException
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.util.logging.Slf4j

@Slf4j(value = "LOGGER", category = "com.ca.apm.acc.plugin.GetProfile")
class GetProfile {
    static def run(final String request) {
        try {
            return worker(request);
        } catch (final Exception ex) {
            throw new PluginException("GetProfile", ex);
        }
    }

    private static def worker(final String request) {
        def inputs = new JsonSlurper().parseText(request);

        def r = [:];
        r.version = "1.0";
        r.className = "FileContent";

        def filename = inputs["com.wily.introscope.agentProfile"];
        LOGGER.debug("Config filename from request: $filename");

        try {

            /*
             * If it's relative, use 'user.dir'
             */
            def apFile = new java.io.File(filename);
            if (!apFile.isAbsolute()) {

                LOGGER.debug("relative path $filename");

                def x = java.nio.file.Paths.get(inputs["user.dir"] ?: "", filename);
                apFile = x.toFile();
                filename = apFile.getAbsolutePath();

                LOGGER.debug("file now $filename");
            }

            def canonicalPath = apFile.getCanonicalPath();
            LOGGER.debug("Config canonical path: $canonicalPath");
            r.filePath = canonicalPath;
            r.fileType = "com.wily.introscope.agentProfile";

            def fn = java.nio.file.Paths.get(filename);

            def content = java.nio.file.Files.readAllBytes(fn);

            // Result of readAllBytes is a byte array and there's no JSON for that. Convert to String.
            r.fileContent = new java.lang.String(content, "UTF-8");

            // Get modified time
            def aFile = new java.io.File(filename);
            r.modified = aFile.lastModified();

        } catch(e) {
            LOGGER.debug("Caught error $e");
            def errors = [];
            def error = new Error();
            error.arguments = [filename];
            if (e instanceof java.nio.file.NoSuchFileException) {
                error.errno = "ENOENT";
            }
            else if (e instanceof java.nio.file.AccessDeniedException) {
                error.errno = "EACCESS";
            }
            else if (e instanceof java.io.IOException) {
                error.errno = "EIOERROR";
            }
            errors.add(error);
            r.errors = errors;
        }

        //logger.info(r.fileContent);
        def json = new JsonBuilder(r);
        def response = json.toString()
        return response;
    }

    static class Error {
        String errno;
        String[] arguments;
    }
}

def handler() {
    if (binding.variables.containsKey("request")) {
        logger.info("Running GetProfile.groovy");

        handler = new GetProfile();
        response = handler.run(request);
        return response;
    }
}

handler();
