
/*
 * Copyright (c) 2025 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import groovy.util.logging.Slf4j
import java.nio.file.Files
import java.nio.file.NoSuchFileException
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.StandardCopyOption

@Slf4j
class ACCException extends Exception {
    String errno
    String [] arguments
 
    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }

    void setParams(String ... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }

    def getAccError() {
        return ["errno":this.errno, "arguments":arguments]
    }
}

class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom)
    }
}

class ACCWriteAccessException extends ACCException {
    
    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

class ACCNoEntityException extends ACCException {
    
    ACCNoEntityException(String msg, enitity) {
        super(msg, "ENOENT")
        setParams(enitity.toString())
    }
}

class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

class ACCFrameworkException extends ACCException {
    ACCFrameworkException(String msg, String info) {
        super(msg, "EFRAMEWORKERROR")
        setParams(info)
    }
}


@Slf4j
class FilePuller  {

    def requestJson
    
    /**
     * Copy (or pull) file from the broker
     * and write locally
     */
    void copyBrokerFile(URL broker, String from, Path to) {
        URL url = new URL(broker.toString() + from)
        log.info("copyBrokerFile: $url -> $to")
        httpGet(url, to)
    }

    /**
     * Fetch the URL and write the file to a temporary file before moving
     * it atomically into place to avoid half-written files etc
     */
    void httpGet(URL url, Path to) {
    
        // Write the file to a temporary file first
        Path temp = Paths.get(to.toString() + ".tmp")

        InputStream is
        try {
            is = url.openConnection().getInputStream()
        }
        catch (IOException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }

        try {
            Files.deleteIfExists(temp)
            Files.copy(is, temp)
            Files.move(temp, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
        }
        catch (NoSuchFileException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
        finally {
            try {
                Files.deleteIfExists(temp)
            }
            catch (Exception ignore) {
                log.info("Failed to delete $temp") 
            }

            is.close()
        }
    }

    /**
     * Delete a file with exception to error mapping
     */
    void deleteFile(Path to) {
        log.info("deleteFile: $to")
    
        try {
            Files.delete(to);
        }
        catch (NoSuchFileException e) {
            throw new ACCNoEntityException(e.getMessage(), to)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
    }
    
    
    Path getAccPath(other) {

        String base = requestJson["acc.controller.dir"]

        if (!base) {
            base = ".";
        }

        return Paths.get(base).resolve(other);
    }

    /**
     * Read the given properties file to get the broker url
     * (configurationServer.url=thebrokername)
     */
    URL getBrokerUrl() {

 		String serverUrl = "com.ca.apm.acc.controller.configurationServer.url";
		String serverUrlShort = "configurationServer.url";
		
        String brokerString = System.getenv(serverUrl);
        if (brokerString != null) {
            log.info("Found " + serverUrl + " in system environment variables: $brokerString")
        } else {
            Path propertiesFilename = getAccPath("config/apmccctrl.properties")
            FileInputStream is

            log.info("Reading " + serverUrl + " from $propertiesFilename")
            try {
                // Get the broker URL from the properties file.
                Properties controllerProperties = new Properties()

                is = new FileInputStream(propertiesFilename.toFile())
                controllerProperties.load(is)
                brokerString = controllerProperties.getProperty(serverUrl)
				
				if(brokerString == null)
					brokerString = controllerProperties.getProperty(serverUrlShort);
            }
            catch (IOException e) {
                throw new ACCReadAccessException("Failed to read config", propertiesFilename)
            }
            finally {
                if (is != null) {
                    is.close()
                }
            }
        }

        if (!brokerString.endsWith("/")) {
            brokerString += "/"
        }

        return new URL(brokerString)
    }
    
    /**
     *  Derive the base dir of the agent
     */
    Path getAgentDir(other) {
        
        String base = extractParam("acc.agent.installpath")
    
        // Add / to Unix path in case it's missing.
        if (!base.startsWith("/") && !extractParam("os.name").contains("Windows")) {
            base = "/" + base
        }
    
        log.debug("acc.agent.installpath is $base")
        
        return Paths.get(base).resolve(other)
    }

    /**
     * Extract the given param from the json request, throwing an ACCBadParamException
     * if the value does not exist
     */
    String extractParam(String param) {
        
        String value = requestJson[param]
    
        if (value == null) {
            throw new ACCBadParamException("Missing value", param)
        }
    
        return value
    }

    /**
     * Handle the request
     */
    def handler(String request) {
        def errs = []
    
        try {
            log.debug("Raw request: $request")
            
            // Parse JSON Request, save in the object
            this.requestJson = new groovy.json.JsonSlurper().parseText(request)

            Path to = getAgentDir(extractParam("acc.to"))

            /*
             *  Give up if there's anything that looks like a:
             *  
             *  relative path using ".."
             *  directory specified as the "to" path (used for both copy and delete).
             *  absolute "to" path
             */
            if (to.toString().contains("..") ||
                extractParam("acc.to").startsWith("/") || extractParam("acc.to").startsWith("\\") ||
                new File(extractParam("acc.to")).isAbsolute()) {

                throw new ACCBadParamException("Bad path", "acc.to");
            }
            else if (Files.isDirectory(to)) {

                throw new ACCWriteAccessException("Path is a directory", to)
            }
            else if (Files.exists(to) && !Files.isWritable(to)) {

                // Do not copy over or delete read-only files
                throw new ACCWriteAccessException("File exists and is read-only", to)
            }
            else {
                String op = extractParam("acc.op")
                
                switch(op) {
                    case "COPY":
                        // Pull the file from the broker
                        copyBrokerFile(getBrokerUrl(), extractParam("acc.from"), to)

                        break
                    case "DELETE":
                        // Delete the file
                        deleteFile(to)
                        break
                    default:
                        throw new ACCBadParamException("Unrecognized operation $op", "acc.op")
                }
            }
        }
        catch (ACCException e) {
            errs.push(e.getAccError())
        }
        catch (Exception e) {

            /*
             * An exceptional exception occurred, so log an error
             */

            log.error("Caught Exception $e")
            e.printStackTrace()

            errs.push(new ACCFrameworkException("Internal Error", e.getMessage()).getAccError())
            
        }

        // Create the json message
        def json = new groovy.json.JsonBuilder()
        def root = json {
            version "1.0"
            className "KeyValuePairs"
            errors errs
        }

        String response = json.toString()
        
        log.debug("Response: $response")

        return response
    }

}

if (binding.variables.containsKey("request")) {
    logger.info("Running pullFile.groovy")
    
    fp = new FilePuller()
    response = fp.handler(request)
}
else {

    /*
     * To run in test mode, a logging implementation has to be supplied on the cp, 
     * so for example call with something like this:
     * 
     * groovy -cp /usr/share/java/slf4j-simple.jar:/usr/share/java/slf4j-api.jar 
     *     -Dorg.slf4j.simpleLogger.defaultLogLevel=debug ./pullFile.groovy 
     *         COPY docs/RELEASE-NOTES.txt out.txt
     */

    println("Running pullFile.groovy - TEST MODE")
    
    op = args[0]
    from = args[1]
    to = args[2]
    
    request = "{"
    request += String.format('"acc.from":"%s", ', from)
    request += String.format('"acc.to":"%s", ', to)
    request += String.format('"acc.op":"%s", ', op)
    request += String.format('"os.name":"%s", ', "Windows")
    request += String.format('"acc.agent.installpath":"%s"', "wily/blah")
    request += "}"

    fp = new FilePuller()

    response = fp.handler(request)

    println("Response is:")    
    println(response)

}





