/*
 * Copyright (c) 2024 Broadcom. All rights reserved. The term "Broadcom"
 * refers to Broadcom Inc. and/or its subsidiaries. All trademarks, trade
 * names, service marks, and logos referenced herein belong to their
 * respective companies.
 *
 * This software and all information contained therein is confidential and
 * proprietary and shall not be duplicated, used, disclosed or disseminated
 * in any way except as authorized by the applicable license agreement,
 * without the express written permission of Broadcom. All authorized
 * reproductions must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT
 * PERMITTED BY APPLICABLE LAW OR AS AGREED BY BROADCOM IN ITS APPLICABLE
 * LICENSE AGREEMENT, BROADCOM PROVIDES THIS DOCUMENTATION "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NONINFRINGEMENT.  IN NO EVENT WILL BROADCOM BE LIABLE TO THE END USER OR
 * ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE
 * OF THIS DOCUMENTATION, INCLUDING WITHOUT LIMITATION, LOST PROFITS, LOST
 * INVESTMENT, BUSINESS INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF
 * BROADCOM IS EXPRESSLY ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH LOSS
 * OR DAMAGE.
 */

package com.ca.apm.acc.plugin.impl.ScriptPlugin

import com.ca.apm.acc.controller.plugin.api.AccPluginApi
import com.ca.apm.acc.controller.plugin.api.uma.UmaService
import com.ca.apm.acc.controller.uma.AgentResponse
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j

import java.nio.file.Path
import java.util.concurrent.Future

@Slf4j
@CompileStatic
class ACCException extends Exception {
    String errno
    String[] arguments

    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }

    void setParams(String... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }

    def getAccError() {
        return ["errno": this.errno, "arguments": arguments]
    }
}

@CompileStatic
class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

@CompileStatic
class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom.toAbsolutePath().normalize().toString())
    }
}

@CompileStatic
class ACCWriteAccessException extends ACCException {

    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

@CompileStatic
class ACCNoEntityException extends ACCException {

    ACCNoEntityException(String msg, enitity) {
        super(msg, "ENOENT")
        setParams(enitity.toString())
    }
}

@CompileStatic
class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

@CompileStatic
class ACCFrameworkException extends ACCException {
    ACCFrameworkException(String msg, String info) {
        super(msg, "EFRAMEWORKERROR")
        setParams(info)
    }
}

@Slf4j
class UmaUpgrade {
    /**
     * make a JSON structure containing the Response message and return as a String
     */
    static def makeResponse(errs = [], respMsg = null, respException = null) {
        def json = new JsonBuilder()
        json(
            version: "1.0",
            className: "KeyValuePairs",
            properties: [
                "acc.uma.upgrade.response.message"  : respMsg,
                "acc.uma.upgrade.response.exception": respException
            ],
            errors: errs
        )
        return json.toString()
    }

    def handler(request, AccPluginApi accPluginApi) {
        def errs = []
        // Parse JSON Request
        def inputs = new JsonSlurper().parseText(request)

        def umaService = accPluginApi.getUmaService()
        UmaService.WorkloadType owner = inputs["acc.uma.upgrade.owner"]
        String ownerName = inputs["acc.uma.upgrade.ownerName"]
        String namespace = inputs["acc.uma.upgrade.namespace"]
        String podName = inputs["acc.uma.upgrade.podName"]
        String imageName = inputs["acc.uma.upgrade.dockerImage"]

        Future<AgentResponse> future = umaService.sendUmaApplicationUpgradeRequest(owner, ownerName, namespace, podName, imageName)

        com.ca.apm.acc.controller.plugin.api.uma.AgentResponse agentResponse = future.get()

        def response;

        if (agentResponse.getStatus() == com.ca.apm.acc.controller.plugin.api.uma.AgentResponse.Status.SUCCESS) {
            response = makeResponse(errs, agentResponse.getMessage())

        } else if (agentResponse.getStatus() == com.ca.apm.acc.controller.plugin.api.uma.AgentResponse.Status.FAILED) {
            errs.push(new ACCFrameworkException("UmaUpgrade failed",
                "Exception: " + agentResponse.getException() + ", message: " + agentResponse.getMessage()).getAccError())
            response = makeResponse(errs, agentResponse.getMessage(), agentResponse.getException())

        } else {
            errs.push(new ACCFrameworkException("Unknown agent response status received",
                (agentResponse.getStatus() != null) ? agentResponse.getStatus().toString() : null).getAccError())
            response = makeResponse(errs)
        }

        log.debug("UmaUpgrade response: " + response)
        return response
    }
}

response = new UmaUpgrade().handler(request, accPluginApi)
