/*
 * Copyright (c) 2016 CA. All rights reserved.
 *
 * This software and all information contained therein is confidential and proprietary and shall not
 * be duplicated, used, disclosed or disseminated in any way except as authorized by the applicable
 * license agreement, without the express written permission of CA. All authorized reproductions
 * must be marked with this language.
 *
 * EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT PERMITTED BY APPLICABLE
 * LAW, CA PROVIDES THIS SOFTWARE WITHOUT WARRANTY OF ANY KIND, INCLUDING WITHOUT LIMITATION, ANY
 * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT WILL CA BE
 * LIABLE TO THE END USER OR ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE
 * USE OF THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, LOST PROFITS, BUSINESS INTERRUPTION,
 * GOODWILL, OR LOST DATA, EVEN IF CA IS EXPRESSLY ADVISED OF SUCH LOSS OR DAMAGE.
 */


/*
 * https://cawiki.ca.com/display/APM/ARB+-+Hot+Bundle+Delivery
 *
 * This plugin performs the following tasks:
 * 
 * 1) Fetches the package referenced in the 'acc.package.url' key.
 *
 * 2) Fetch this directly to the %acc.agent.installpath%/extensions/deploy directory using a
 * temporary filename (temp file must not end .tar.gz). We can
 * thus catch any 'permission denied' / 'file system full' error and abort the operation without
 * having to 'undo' any bundles that have already been enabled.
 *
 * 3) Updates IntroscopeAgent.profile with the new acc.package.id/name/version fields - this must be
 * done before any bundles are loaded. This must be done before unpacking the bundles as the agent
 * profile is re-read when a hot bundle is loaded. So we want to have changed the profile before
 * this occurs.
 * 
 * 4) Unpacks the hot bundles from the package into the extensions/deploy directory.
 * 
 * 5) Deletes any bundles that are being removed in this operation
 * 
 * 6) Unpacks the package manifest file and install instructions into the APM wily directory.
 * 
 * 7) Potentially unpacks IntroscopeAgent.profile and acc-master-toggles.pbd and any other files in the request
 * (see extractExtraFiles).  This is hot support hot properties in cold bundles. 
 * This will overwrite the change made in step 3)
 *
 * 8) Sends a JSON response to the Config Server. At this point the Config Server can set the tasks
 * state to STARTED of FAILED appropriately.
 *
 * The bundle operations will be performed in the order that the plugin reads them from the request.
 * It is up to the agent to ensure dependencies are satisfied.
 * 
 */


import java.net.URL
import javax.net.ssl.SSLContext
import java.nio.file.Path
import java.nio.file.Paths
import java.nio.file.Files
import java.io.FileOutputStream;
import java.nio.file.DirectoryStream
import java.nio.file.StandardCopyOption
import java.nio.file.NoSuchFileException
import java.nio.file.attribute.PosixFilePermissions;
import java.io.IOException
import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.LinkedHashMap
import java.util.EnumSet;
import java.util.zip.ZipFile
import java.util.zip.ZipEntry
import java.util.Enumeration
import java.util.regex.Pattern
import java.util.regex.Matcher
import java.util.zip.GZIPInputStream

import org.apache.commons.io.FileUtils;

import javax.net.ssl.TrustManager
import javax.net.ssl.X509TrustManager
import javax.net.ssl.HostnameVerifier
import javax.net.ssl.HttpsURLConnection
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream

import groovy.util.logging.Slf4j
import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.transform.TypeChecked
import groovy.json.JsonBuilder
import java.nio.charset.StandardCharsets


import static groovy.io.FileType.FILES

@Slf4j
@CompileStatic
class ACCException extends Exception {
    String errno
    String[] arguments
 
    ACCException(String msg, String errno) {
        super((msg == null) ? "<null msg>" : msg)
        this.errno = errno
        log.info("ACCException: $msg")
    }
    
    void setParams(String ... arguments) {
        log.info("ACCException args: $arguments")
        this.arguments = arguments
    }
    
    def getAccError() {
        return ["errno":this.errno, "arguments":arguments]
    }
}

@CompileStatic
class ACCBadParamException extends ACCException {

    ACCBadParamException(String msg, String parameterName) {
        super(msg, "EBADPARAM")
        setParams(parameterName)
    }
}

@CompileStatic
class ACCReadAccessException extends ACCException {

    ACCReadAccessException(String msg, Path readingFrom) {
        super(msg, "EACCESS")
        setParams(readingFrom.toAbsolutePath().normalize().toString())
    }
}

@CompileStatic
class ACCWriteAccessException extends ACCException {
    
    ACCWriteAccessException(String msg, Path writingTo) {
        super(msg, "EWRITEACCESS")
        setParams(writingTo.toString())
    }
}

@CompileStatic
class ACCNoEntityException extends ACCException {
    
    ACCNoEntityException(String msg, enitity) {
        super(msg, "ENOENT")
        setParams(enitity.toString())
    }
}

@CompileStatic
class ACCUrlException extends ACCException {
    ACCUrlException(String msg, URL from, Path to, Exception details) {
        super(msg, "EURLERROR")
        setParams(from.toString(), to.toString(), details.toString())
    }
}

@CompileStatic
class ACCFrameworkException extends ACCException {
    ACCFrameworkException(String msg, String info) {
        super(msg, "EFRAMEWORKERROR")
        setParams(info)
    }
}


@Slf4j
@CompileStatic
class FilePuller  {

    /**
     * Fetch the URL and write the file to a temporary file before moving
     * it atomically into place to avoid half-written files etc
     */
    void httpGet(URL url, Path to, String token, boolean trustSelfSigned, boolean useHttp) {
    
        // Write the file to a temporary file first
        Path temp = Paths.get(to.toString() + ".tmp")

        InputStream is
        try {
            URLConnection urlConnection = url.openConnection()
            if (urlConnection instanceof HttpURLConnection) {
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "text/plain")
                ((HttpURLConnection) urlConnection).setRequestProperty("Accept", "application/octet-stream")
            }

            if (useHttp) {
                is = urlConnection.getInputStream()
            } else if (urlConnection instanceof HttpsURLConnection) {
                HttpsURLConnection connection = (HttpsURLConnection)urlConnection
                connection.setRequestProperty("Authorization", "Bearer " + token)
                if(trustSelfSigned)
                {
                    def nullTrustManager = [
                            checkClientTrusted: { chain, authType ->  },
                            checkServerTrusted: { chain, authType ->  },
                            getAcceptedIssuers: { null }
                    ]

                    SSLContext sc = SSLContext.getInstance("SSL")
                    sc.init(null, [nullTrustManager as  X509TrustManager] as  X509TrustManager[], null)
                    connection.setSSLSocketFactory(sc.getSocketFactory())

                    def nullHostnameVerifier = [
                            verify: { hostname, session -> true }
                    ]
                    connection.setHostnameVerifier(nullHostnameVerifier as HostnameVerifier)
                }
                is = connection.getInputStream()
            } else {
                ((HttpURLConnection) urlConnection).setRequestProperty("Authorization", "Bearer " + token)
                is = urlConnection.getInputStream()
            }
        }
        catch (IOException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }

        try {
            Files.deleteIfExists(temp)
            Files.copy(is, temp)
            Files.move(temp, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
        }
        catch (NoSuchFileException e) {
            throw new ACCUrlException(e.getMessage(), url, to, e)
        }
        catch (IOException e) {
            throw new ACCWriteAccessException(e.getMessage(), to)
        }
        finally {
            try {
                Files.deleteIfExists(temp)
            }
            catch (Exception ignore) {
                log.info("Failed to delete $temp")
            }

            is.close()
        }
    }

    /**
     * Copy (or pull) file from the broker
     * and write locally
     */
    void copyBrokerFile(URL broker, String from, Path to, String token, boolean trustSelfSigned, boolean useHttp) {
        
        URL url = new URL(broker.toString() + from)
        
        log.debug("copyBrokerFile: $url -> $to")
        httpGet(url, to, token, trustSelfSigned, useHttp)
    }
}

@Slf4j
@CompileStatic
class Archive {

    Path archive
	String releaseVersion
	String os = System.properties['os.name']
	Pattern p = Pattern.compile("(.*(?:[\\\\/])jre(?:[\\\\/])([^\\\\/]*))(?:[\\\\/]).*")

    Archive(Path archive, String releaseVersion) {
        this.archive = archive
		this.releaseVersion = releaseVersion
    }
	
	Path unzip(Path outDir, Path agentDeployDir, Boolean isNewRelease) {
		log.info("Unzipping $archive to $outDir")
		def bundles = [];
		ZipFile zipFile = null
		Map<String, String> jre = new HashMap<String, String>()
		try {
			zipFile = new ZipFile(archive.toFile())
			if (!isNewRelease) {
				agentDeployDir.toFile().traverse(type: FILES, maxDepth: 0) 
				{ 
					if (it.getName().endsWith("tar.gz"))
						bundles.add(it.getName())
				};
			} else {
				log.info("Cleaning up for the release " + releaseVersion)
				if (Files.exists(outDir.resolve("uia/releases/" + releaseVersion + "/extensions")))
					FileUtils.cleanDirectory(outDir.resolve("uia/releases/" + releaseVersion + "/extensions").toFile())
				if (Files.exists(outDir.resolve("uia/releases/" + releaseVersion + "/robot/pkg")))
					FileUtils.cleanDirectory(outDir.resolve("uia/releases/" + releaseVersion + "/robot/pkg").toFile())
			}
			zipFile.entries().each { it -> 
				ZipEntry entry = (ZipEntry) it
				if (isNewRelease) {
					Path dest = outDir.resolve(entry.getName())
					if ((entry.getName().endsWith("/") || entry.getName().endsWith("\\")) && !Files.exists(dest)) {
						Files.createDirectories(dest)
					} else if (!Files.exists(dest.getParent())) {
						Files.createDirectories(dest.getParent())
					}

					if (!Files.exists(dest)) {
						// Unzip the file
						log.debug("trying to copy/replace " + entry.getName() + " to dest " + dest)
						Files.copy(zipFile.getInputStream(entry), dest, StandardCopyOption.REPLACE_EXISTING)
						if (!os.toLowerCase().contains("windows")) {
							Files.setPosixFilePermissions(dest, PosixFilePermissions.fromString("rwxr-x---"))
						}
					} else if (Files.exists(dest) && Files.isRegularFile(dest)) {
						boolean isJrePath = false
						Matcher m = p.matcher(dest.toString())
						while (m.find()) {
							isJrePath = true
							Path java = Paths.get(m.group(1), "bin", (os.toLowerCase().contains("windows")) ? "java.exe" : "java")
							if (!jre.keySet().contains(m.group(2))) {
								FileOutputStream fos = null
								try {
									fos = new FileOutputStream(java.toFile(), true)
									jre.put(m.group(2), "false")
								} catch(IOException e) {
									// -> file still open
									log.info(java.toString() + " is NOT writable. Skipping update for " + m.group(1))
									jre.put(m.group(2), "true")
								} finally {
									if(fos != null) {
										try {
											fos.close();
										} catch (IOException e) {
											e.printStackTrace();
										}
									}
								}
							} else if (jre.get(m.group(2)).equals("false")) {
								// Unzip the file
								log.debug("trying to copy/replace " + entry.getName() + " to dest " + dest)
								Files.copy(zipFile.getInputStream(entry), dest, StandardCopyOption.REPLACE_EXISTING)
								if (!os.toLowerCase().contains("windows")) {
									Files.setPosixFilePermissions(dest, PosixFilePermissions.fromString("rwxr-x---"))
								}
							}
						}
						if (!isJrePath) {
							// Unzip the file
							log.debug("trying to copy/replace " + entry.getName() + " to dest " + dest)
							Files.copy(zipFile.getInputStream(entry), dest, StandardCopyOption.REPLACE_EXISTING)
							if (!os.toLowerCase().contains("windows")) {
								Files.setPosixFilePermissions(dest, PosixFilePermissions.fromString("rwxr-x---"))
							}
						}
					}
				} else {
					if (entry.getName().contains("extensions/deploy") || 
							entry.getName().contains("extensions\\deploy")) {
						def bundleName = new File(entry.getName()).getName()
						Path dest = outDir.resolve(entry.getName())
						if (!Files.exists(dest.getParent())) {
							Files.createDirectories(dest.getParent())
						}
						
						if (bundles.contains(bundleName))
							bundles.remove(bundleName)
						
						// Unzip the file
						Files.copy(zipFile.getInputStream(entry), dest, StandardCopyOption.REPLACE_EXISTING)
					} else if (entry.getName().contains("core/config/IntroscopeAgent.profile")) {
						Path dest = outDir.resolve(entry.getName() + ".updated")
						log.info("Unzipping {} to {}", entry.getName(), dest)
						Files.copy(zipFile.getInputStream(entry), dest)
					}
				}
			}
		}
		finally {
			if (zipFile != null) {
				zipFile.close()
			}
		}
		
		return null
	}
	
    Path unzip(Path outDir, Path... toExtracts) {
		log.info("Unzipping $toExtracts from $archive to $outDir")
		
		ZipFile zipFile = null
		Path extracted = null
		try {
			zipFile = new ZipFile(archive.toFile())

			ZipEntry entry = null
			toExtracts.each { toExtract ->
				if (entry == null) {
					log.debug("Trying to unzip $toExtract from $archive to $outDir")
					entry = zipFile.getEntry(toExtract.toString().replace("\\", "/"))
					if (entry == null) {
						log.debug("Failed to find file in zip with /.  Re-trying with \\")
						entry = zipFile.getEntry(toExtract.toString().replace("/", "\\"))
					}
					if (entry != null) {
						extracted = (Path)toExtract
					}
				}
			}

			if (entry == null) {
				throw new ACCNoEntityException("Failed to unzip $toExtracts", "$toExtracts was not found in zip file")
			}
			else {
				Path dest = outDir.resolve(extracted.getFileName())
				if (!Files.exists(dest.getParent())) {
					Files.createDirectories(dest.getParent())
				}
				Files.deleteIfExists(dest)
	
				// Unzip the file
				Files.copy(zipFile.getInputStream(entry), dest)
				if (!os.toLowerCase().contains("windows")) {
					Files.setPosixFilePermissions(dest, PosixFilePermissions.fromString("rwxr-x---"))
				}
	
				return dest
			}
		}
		finally {
			if (zipFile != null) {
				zipFile.close()
			}
		}

        return null
    }

    Path untar(Path outFile, Path toExtract){
        log.info("Untarring $toExtract from $archive to $outFile")
        
        TarArchiveInputStream tarInput = null
        try {

            tarInput = new TarArchiveInputStream(new GZIPInputStream(new FileInputStream(archive.toFile())));

            TarArchiveEntry currentEntry = tarInput.getNextTarEntry();
            TarArchiveEntry entry = null
            while(null != currentEntry) {
                
                if(currentEntry.getName().equals(toExtract.toString().replace("\\", "/"))){
                    entry = currentEntry;
                    break
                }else if(currentEntry.getName().equals(toExtract.toString().replace("/", "\\"))){
                    entry = currentEntry;
                    break
                }
                
                currentEntry = tarInput.getNextTarEntry()
            }

            if (null == entry) {
                log.info("File $toExtract was not found in tar file $archive")
                return null
            }else {
                log.debug("Found $entry.name with size $entry.size in $archive")
                
                Files.createDirectories(outFile.getParent())
                Files.deleteIfExists(outFile)
    
                // Untar the file
                Files.copy(tarInput, outFile)
    
                return outFile
            }
        }
        finally {
            if (tarInput != null) {
                tarInput.close()
            }
        }

        return null
    }

}


@Slf4j
class AgentProfile {
    Path profile
    String content
    String fileContent
    boolean doneEol = false

    AgentProfile(Path profile) {
        this.profile = profile
        this.content = profile.getText()
        this.fileContent = content  // copy content to fileContent for env vars processing
    }

    String getProperty(String property) {
        log.debug("Getting ${property}")

        // Use a so called "slashy" string to avoid escaping        
        String matchStr = String.format(/^(%s)[ \t]*=[ \t]*(.*)$/, Matcher.quoteReplacement(property))
        
        log.debug("Regex is: $matchStr")

        Pattern p = Pattern.compile(matchStr, Pattern.MULTILINE)
        Matcher m = p.matcher(content)        
        boolean exists = m.find()

        log.debug("Property $property exists: $exists")

        if (exists) {
            return m.group(2)
        }
    }
    
    String setProperty(String property, String value) {
        log.debug("Setting ${property}=${value}")

        // Use a so called "slashy" string to avoid escaping        
        String matchStr = String.format(/^(%s)[ \t]*=[ \t]*(.*)$/, Matcher.quoteReplacement(property))
        
        log.debug("Regex is: $matchStr")

        Pattern p = Pattern.compile(matchStr, Pattern.MULTILINE)
        Matcher m = p.matcher(content)        
        boolean exists = m.find()

        log.debug("Property $property already exists: $exists")

        if (exists) {
            // Change existing value
            String prevValue = m.group(2)

            log.debug("Changing existing property $property=$prevValue to $property=$value")
            StringBuffer sb = new StringBuffer()
            m.appendReplacement(sb, "$property=$value")
            m.appendTail(sb)

            content = sb.toString()
        }
        else {
            // Need to append the value
            log.debug("Appending new property: $property=$value")
            
            if (!doneEol) {
                content += "\n"
                doneEol = true
            }

            content += "${property}=${value}\n"
        }
    }

    String getFileContent() {
        return fileContent
    }

    void writeTo(Path out) {
        log.info("Writing updated profile content to: $out")
        log.debug("Profile content of $out:\n{}", content)
        out.write(content)
    }
}


@Slf4j
@CompileStatic
class ApplyPackage {
    def requestJson

    Path archive
    
    URL broker
    String partialUrl
    String authToken
    boolean trustSelfSigned
    boolean useHttp
	String releaseVersion
	boolean isNewRelease
    Path accDir, agentDir, agentDeployDir, userDir

    void initEnv() {

        log.debug("Running on: {}", System.getProperty("os.name"))

        accDir = Paths.get(extractOptionalParam("acc.controller.dir", "."))
        log.debug("accDir: {}", accDir.toAbsolutePath().normalize())

        agentDir = extractDirParam("acc.agent.installpath")
        log.info("agentDir: {}", agentDir.toAbsolutePath().normalize())

        // Make sure we can resolve this path before we start.
        log.debug("Profile path={}", makeUserPath(extractParam("com.wily.introscope.agentProfile")));

        agentDeployDir = agentDir.resolve("extensions/deploy")
        Files.createDirectories(agentDeployDir)
        
        partialUrl = extractParam("acc.package.url")

        String b = requestJson["acc.broker.url"]
        setBroker((b != null) ? b : readBrokerUrl())

        authToken = extractParam("com.ca.apm.acc.controller.configurationServer.token")

        trustSelfSigned = extractParam("com.ca.apm.acc.cs.trustSelfSignedCerts").toBoolean()

        String useHttpProp = requestJson["com.ca.apm.acc.cs.useHttp"]
        if(useHttpProp == null) {
            useHttp = false
        } else {
            useHttp = useHttpProp.toBoolean()
        }

        log.debug("broker: {}", this.broker)
        log.debug("partialUrl: {}", this.partialUrl)

        archive = agentDeployDir.resolve("package.tmp")
        log.debug("archive: {}", this.archive)
    }

	void verifyInstallType() {
		isNewRelease = false
		releaseVersion = "99.99.0.0"

        ZipFile zipFile = null
        try {
            zipFile = new ZipFile(archive.toFile())
			Enumeration<? extends ZipEntry> entries = zipFile.entries()
			while (entries.hasMoreElements()) { 
				ZipEntry entry = entries.nextElement(); 
				def match = entry.getName() =~ /(?:apmia|uia|wily|dotnetagent)\/releases\/([0-9\.]*)\/core\/config\/.*$/
				if (match.find()) {
					releaseVersion = match.group(1)
					break
				}	
			}
        }
        finally {
            if (zipFile != null) {
                zipFile.close()
            }
        }
		
		log.info("Current release version is {}", releaseVersion)
		if (!agentDir.endsWith(releaseVersion)) {
			log.info("setting flag isNewRelease to true")
			isNewRelease = true
		}
	}
	
	List getEnv(){
		def env = System.getenv();
		def envlist = [];
		env.each() { k,v -> envlist.push( "$k=$v" ) }
		
		return envlist
	
	}
    
    void restartService() {
		String command
		File workingDir = agentDir.getParent().getParent().toFile()
		String os = System.properties['os.name']
		if (os.toLowerCase().contains("windows"))
			command = "cmd /c " + workingDir.getPath() +  File.separator + "UIACtrl.bat restart"
		else
			command = "/usr/bin/bash " + workingDir.getPath() + File.separator + "UIACtrl.sh restart"
		
		def uiaService = command.execute(getEnv(), workingDir);
		def outputStream = new StringBuffer();
		def errorStream = new StringBuffer();
		uiaService.waitForProcessOutput(outputStream, errorStream);
		log.info("Service restart completed with output {} and error {}", outputStream.toString(), errorStream.toString());
    }

	void uninstallService() {
		String command
		File workingDir = agentDir.getParent().getParent().toFile()
		String os = System.properties['os.name']
		if (os.toLowerCase().contains("windows"))
			command = "cmd /c " + workingDir.getPath() +  File.separator + "UIACtrl.bat uninstall"
		else
			command = "/usr/bin/bash " + workingDir.getPath() + File.separator + "UIACtrl.sh uninstall"
		
		def uiaService = command.execute(getEnv(), workingDir);
		def outputStream = new StringBuffer();
		def errorStream = new StringBuffer();
		uiaService.waitForProcessOutput(outputStream, errorStream);
		log.info("Service uninstall completed with output {} and error {}", outputStream.toString(), errorStream.toString());
	}
	
	void installService() {
		String command
		File workingDir = agentDir.getParent().getParent().toFile()
		String os = System.properties['os.name']
		if (os.toLowerCase().contains("windows"))
			command = "cmd /c " + workingDir.getPath() +  File.separator + "UIACtrl.bat install"
		else
			command = "/usr/bin/bash " + workingDir.getPath() + File.separator + "UIACtrl.sh install"
		
		def uiaService = command.execute(getEnv(), workingDir);
		def outputStream = new StringBuffer();
		def errorStream = new StringBuffer();
		uiaService.waitForProcessOutput(outputStream, errorStream);
		log.info("Service install completed with output {} and error {}", outputStream.toString(), errorStream.toString());
	}
	
    void setBroker(String b) {
        this.broker = new URL((b.endsWith("/")) ? b : b + "/" )
    }

    Path makeAccPath(String other) {
        return (other != null) ? accDir.resolve(other) : accDir
    }

    Path makeUserPath(String other) {
        
        if (other != null) {
            Path pathOther = Paths.get(other)
            if (pathOther.isAbsolute()) {
                return pathOther;
            }
        }

        /*
         * Need to resolve the paths against userDir.
         * Note that we only try this when the other path is relative 
         * because we might not have permissions to actually read user.dir
         */
        if (userDir == null) {
            userDir = extractDirParam("acc.agent.installpath")
            log.debug("userDir: {}", userDir.toAbsolutePath().normalize())
        }

        return (other != null) ? userDir.resolve(other) : userDir
    }
    
    /**
     * Read the given properties file to get the broker url
     * (configurationServer.url=thebrokername)
     */
    String readBrokerUrl() {

		String serverUrl = "com.ca.apm.acc.controller.configurationServer.url";
		String serverUrlShort = "configurationServer.url";
		
        String brokerString = System.getenv(serverUrl);
        if (brokerString != null) {
            log.debug("Found " + serverUrl + " in system environment variables: $brokerString")
        } else {
			Properties controllerProperties = readControllerProperties()
			brokerString = controllerProperties.getProperty(serverUrl)
			
			if(brokerString == null)
				brokerString = controllerProperties.getProperty(serverUrlShort);
        }

        if (!brokerString.endsWith("/")) {
            brokerString += "/"
        }

        return brokerString
    }

	Properties readControllerProperties() {
		FileInputStream is
		Properties controllerProperties = new Properties()

		// Get the broker URL from the properties file.
		Path propertiesFilename = makeAccPath("config/apmccctrl.properties")

		log.debug("Reading properties from $propertiesFilename")
		try {
			is = new FileInputStream(propertiesFilename.toFile())
			controllerProperties.load(is)
		}
		catch (IOException e) {
			throw new ACCReadAccessException("Failed to read config", propertiesFilename)
		}
		finally {
			if (is != null) {
				is.close()
			}
		}
		return controllerProperties
	}

    /**
     * Updates IntroscopeAgent.profile with the new acc.package.id/name/version fields - this must be
     * done before any bundles are loaded. This must be done before unpacking the bundles as the agent
     * profile is re-read when a hot bundle is loaded. So we want to have changed the profile before
     * this occurs.
     */
    void updateAgentProfile() {
		int updateCount = 3;
		
        Path original = makeUserPath(extractParam("com.wily.introscope.agentProfile"))
        Path updated = makeUserPath(original.toString() + ".updated")
        log.info("Updating agent profile {}", original)

        try {
            AgentProfile profile = new AgentProfile(original) 
/*
            for (prop in ["acc.package.id", "acc.package.name", "acc.package.version"]) {
                profile.setProperty(prop, extractParam(prop))
            }
            profile.writeTo(updated)
*/

			do {
				try {
					Files.move(updated, original, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
					break
				} catch (IOException e) {
					sleep(1000)
				}
				updateCount--
			} while(updateCount > 0)
        }
        catch (IOException e) {
            throw new ACCReadAccessException("Failed to update agent profile", original)
        }
        finally {
        }
    }
    
    void updateConfigurations() {
        try {
            Path original = !isNewRelease ? makeUserPath(extractParam("com.wily.introscope.agentProfile")) : Paths.get(agentDir.getParent().toString(), releaseVersion, "core/config/IntroscopeAgent.profile")
            AgentProfile profile = new AgentProfile(original)
            
            /*
             * Update o2 configuration
            */
            def o2Configuration = !isNewRelease ? Paths.get(agentDir.toString(), "probes/gateway/o2_connector/o2_connector.cfg").toFile() : Paths.get(agentDir.getParent().toString(), releaseVersion, "probes/gateway/o2_connector/o2_connector.cfg").toFile()
            String o2LogSize = getConfigValue(o2Configuration, "logsize")
            String bundleLogSize = profile.getProperty("uia.o2.connector.logsize")
            if ((bundleLogSize != null) && !bundleLogSize.equals(o2LogSize)) {
                updateConfigValue(o2Configuration, "logsize", bundleLogSize)
            }
            
            /*
             * Update robot configuration
            */
            boolean restartAgent = false
            def robotConfiguration = !isNewRelease ? Paths.get(agentDir.toString(), "robot/robot.cfg").toFile() : Paths.get(agentDir.getParent().toString(), releaseVersion, "robot/robot.cfg").toFile()
            String controllerPort = getConfigValue(robotConfiguration, "controller_port")
            String updatedControllerPort = profile.getProperty("uia.o2.controller.controller_port")
            if ((updatedControllerPort != null) && !updatedControllerPort.equals(controllerPort)) {
                updateConfigValue(robotConfiguration, "controller_port", updatedControllerPort)
                restartAgent = true
            }
            String robotLogLevel = getConfigValue(robotConfiguration, "loglevel")
            String updatedLogLevel = profile.getProperty("uia.o2.controller.loglevel")
            if ((updatedLogLevel != null) && !updatedLogLevel.equals(robotLogLevel)) {
                updateConfigValue(robotConfiguration, "loglevel", updatedLogLevel)
                restartAgent = true
            }
            String robotLogSize = getConfigValue(robotConfiguration, "logsize")
            String updatedLogSize = profile.getProperty("uia.o2.controller.logsize")
            if ((updatedLogSize != null) && !updatedLogSize.equals(robotLogSize)) {
                updateConfigValue(robotConfiguration, "logsize", updatedLogSize)
                restartAgent = true
            }
 
         // Process environment variables from IntroscopeAgent.profile
            Map<String, String> envVars = new LinkedHashMap<>()
            String fileContent = profile.getFileContent()
            List<String> lines = Arrays.asList(fileContent.split("\\r?\\n") as String[])
            Pattern envPattern = Pattern.compile("^\\s*uia\\.o2\\.controller\\.env\\.([^=]+)\\s*=\\s*(.*)\\s*\$")

            for (String line : lines) {
                Matcher matcher = envPattern.matcher(line)
                if (matcher.find()) {
                    String key = matcher.group(1).trim()
                    String value = matcher.group(2).trim()
                    println "Found environment variable: ${key}=${value}"
                    envVars.put(key, value)
                }
            }

            // Read robot.cfg content
            String robotContent = FileUtils.readFileToString(robotConfiguration, StandardCharsets.UTF_8.name())


            // Check if <environment> section exists
            if (!robotContent.contains("<environment>") && envVars.size() > 0) {
                println "No environment section found, creating one"
                robotContent = robotContent.replace("</controller>", "   <environment>\n   </environment>\n</controller>")
            }

            if(envVars.size() > 0) {
                restartAgent = true 
            }

            // Update environment variables in robot.cfg
            List<String> newLines = new ArrayList<>()
            boolean inEnvironmentSection = false
            boolean addedVars = false

            for (String line : robotContent.split("\\r?\\n")) {
                if (line.trim().equals("<environment>")) {
                    inEnvironmentSection = true
                    newLines.add(line)
                } else if (line.trim().equals("</environment>")) {
                    inEnvironmentSection = false
                    // Add all new environment variables right before closing tag
                    for (Map.Entry<String, String> entry : envVars.entrySet()) {
                        String envLine = "      ${entry.getKey()} = ${entry.getValue()}"
                        newLines.add(envLine)
                    }
                    newLines.add(line)
                } else if (inEnvironmentSection) {
                    // Skip existing environment variables that we're going to update
                    boolean skip = false
                    for (String key : envVars.keySet()) {
                        if (line.trim().startsWith(key + " =")) {
                            skip = true
                            break
                        }
                    }
                    if (!skip) {
                        newLines.add(line)
                    }
                } else {
                    newLines.add(line)
                }
            }

            // Write the updated content back to robot.cfg
            println "Writing updated content to robot.cfg"
            FileUtils.writeStringToFile(robotConfiguration, newLines.join("\n"), StandardCharsets.UTF_8.name())


            /*
             * Update spooler configuration
            */
            def spoolerConfiguration = !isNewRelease ? Paths.get(agentDir.toString(), "robot/spooler.cfg").toFile() : Paths.get(agentDir.getParent().toString(), releaseVersion, "robot/spooler.cfg").toFile()
            String spoolerLogLevel = getConfigValue(spoolerConfiguration, "debug")
            updatedLogLevel = profile.getProperty("uia.o2.spooler.debug")
            if ((updatedLogLevel != null) && !updatedLogLevel.equals(spoolerLogLevel)) {
                updateConfigValue(spoolerConfiguration, "debug", updatedLogLevel)
                restartAgent = true
            }
            
            /*
             * Update hdb configurations
            */
            def hdbConfiguration = !isNewRelease ? Paths.get(agentDir.toString(), "probes/service/hdb/hdb.cfg").toFile() : Paths.get(agentDir.getParent().toString(), releaseVersion, "probes/service/hdb/hdb.cfg").toFile()
            String hdbLogLevel = getConfigValue(hdbConfiguration, "loglevel")
            updatedLogLevel = profile.getProperty("uia.o2.hdb.loglevel")
            if ((updatedLogLevel != null) && !updatedLogLevel.equals(hdbLogLevel)) {
                updateConfigValue(hdbConfiguration, "loglevel", updatedLogLevel)
                restartAgent = true
            }
            if (restartAgent) {
                restartService();
            }
        }
        catch (IOException e) {
            throw new ACCReadAccessException("Failed to update configuration files", Paths.get(agentDir.toString()))
        }
        finally {
        }
    }

    String getConfigValue(File file, String key) {
        log.debug("opening config file for reading: " + file.getPath())
        def pattern = ~/^\s*${key}\s*=\s*(.*)\s*$/

        String value = null
        file.eachLine { line ->
            def matcher = line =~ pattern
            if (matcher.find()) {
                log.debug("found value for $key" + matcher[0])
                value = matcher.group(1).trim()
            }
        }
        return value
    }

    void updateConfigValue(File file, String key, String newValue) {
        log.debug("opening config file for writing: " + file.getPath())
        def pattern = ~/^(\s*${key}\s*=\s*)(.*)$/
        def updatedLines = file.readLines().collect { line ->
            line.replaceAll(pattern, "\$1${newValue}")
        }
        log.info("updated file with \n" + updatedLines.join('\n'))
        file.text = updatedLines.join('\n')
    }

    /**
     * Maybe unpack the whole set to temporary filenames and then move them
     * to ensure have sufficient disk space / filepermissions etc?
     */
    void applyHotBundles(Archive zip) {
        int added = 0, deleted = 0

		/* 
		 * Since an upgrade operation is a delete old bundle and add new bundle
		 * First delete all the bundles, so that UIA agent can uninstall the probe first
		 */
        for (int i = 1 ; true ; i++) {
            String bundleName = null
            boolean bundleDelete = false

            bundleName = requestJson["acc.bundle.${i}.name"]
            
            if (bundleName == null)  {
                log.debug("Ran out of bundles at index $i")
                break
            }

            // Is this a deletion?
            bundleDelete = ("true".equalsIgnoreCase((String)requestJson["acc.bundle.${i}.delete"]))

            if (bundleDelete) {
                log.debug("Deleting bundle $bundleName")
                
                Path bundleToDelete = agentDeployDir.resolve(bundleName)

                if (!Files.exists(bundleToDelete)) {
                    log.warn("Bundle to be removed does not exist: $bundleToDelete")
                }
                else {
                    Files.deleteIfExists(bundleToDelete)
                    deleted++
                }
            }
        }

        for (int i = 1 ; true ; i++) {
            String bundleName = null
            boolean bundleDelete = false

            bundleName = requestJson["acc.bundle.${i}.name"]
            
            if (bundleName == null)  {
                log.debug("Ran out of bundles at index $i")
                break
            }

            // Is this a deletion?
            bundleDelete = ("true".equalsIgnoreCase((String)requestJson["acc.bundle.${i}.delete"]))

            if (!bundleDelete) {
                log.debug("Adding bundle $bundleName")
            
                zip.unzip(agentDeployDir, Paths.get("uia/releases/${releaseVersion}/extensions/deploy/${bundleName}"))

                // Extract the file from the zip
                added++
            }
        }
		
		/**
		 * This is not a new release but we got full package, this needs to be corrected at ACC side to reduce the package size
		 */
		zip.unzip(agentDir.getParent().getParent().getParent(), agentDeployDir, false)

        // Now extract the manifest and the install instructions, at present we are not shipping installInstructions.md
		try {
			zip.unzip(agentDir, Paths.get("uia/releases/${releaseVersion}/installInstructions.md"))
		} catch (ACCNoEntityException noe) {
			log.info("installInstructions.md part of the zip file - " + noe.getMessage())
		}
        zip.unzip(agentDir, Paths.get("uia/releases/${releaseVersion}/manifest.txt"))

        log.info("Added $added bundles to the agent, removed $deleted")
    }

    /**
     * Extra arbitrary files from the archive.  e.g. IntroscopeAgent.profile acc-master-toggles.pbd
     * for when writing configuration properties to a "traditional" introscope properties file
     */
    void extractExtraFiles(Archive zip) {
        int added = 0, deleted = 0

        for (int i = 1 ; true ; i++) {
            String fileName = null

            fileName = requestJson["acc.package.file.${i}"]

            if (fileName == null)  {
                log.debug("Ran out of acc.package.file at index $i")
                break
            }
			
            log.debug("Adding file $fileName")

            // Strip the leading and trailing path elements to get the sub path in the agent installation
            Path p = Paths.get(fileName)
            log.trace("p is {}, nameCount is {}", p, p.getNameCount())

            Path base = null
            if (p.getNameCount() == 2) {
                base = agentDir
            }
            else if (p.getNameCount() > 2) {
				base = agentDir.resolve(p.subpath(1, p.getNameCount() - 1))
				if (!p.subpath(1, 2).toString().equalsIgnoreCase("releases"))
					p = Paths.get(p.subpath(0, 1).toString(), "releases/${releaseVersion}", p.subpath(1, p.getNameCount()).toString())
            }
            else {
                throw new ACCBadParamException("Bad path " + fileName, "acc.package.file.${i}")
            }

            log.trace("base is {}", base)

            zip.unzip(base, p)

            // Extract the file from the zip
            added++
        }

        log.info("Added $added files to the agent")
    }
    
    /**
     * Extra arbitrary files from the archived bundles for light deploy
     * Used for bundle.properties and toggles.pbd
     */
    void extractExtraBundleFiles(Archive zip) {
        int added = 0, deleted = 0

        for (int i = 1 ; true ; i++) {

            // extract tarred bundle from package archive
            String tarFileName = requestJson["acc.bundle.file.${i}.tar"]
            if (tarFileName == null)  {
                log.debug("Ran out of acc.bundle.file.<i>.zip at index $i")
                break
            }

            log.debug("Extracting file $tarFileName")
            Path extractedBundleTar = zip.unzip(agentDir, Paths.get(tarFileName))

            // extract file from zipped bundle
            Archive bundleTar = new Archive(extractedBundleTar, releaseVersion)
            String pathInTar = requestJson["acc.bundle.file.${i}.pathInTar"]
            
            String outputPath = "extensions/" + Paths.get( tarFileName.replace(".tar.gz", "").replace(".zip", "")).getFileName() + "/" + pathInTar
            bundleTar.untar(agentDir.resolve(outputPath), Paths.get(pathInTar))

            try {
                log.debug("cleanup: removing $extractedBundleTar")
                Files.deleteIfExists(extractedBundleTar)
            }
            catch (IOException e) {
                log.debug("Failed to delete $e")
            }

            // Extract the file from the zip
            added++
        }

        log.info("Added $added bundle files to the agent")
    }

    Path applyPackageToAgent() {

        initEnv()

        // Pull the package file from the broker
        new FilePuller().copyBrokerFile(broker, partialUrl, archive, authToken, trustSelfSigned, useHttp)

		// Verify installation type - updates to current release or new release. Set release version as well to use in further steps
		verifyInstallType()
		if (isNewRelease) {
			// Unzip the new release, use current install path and get to the base of the same
			Archive zip = new Archive(archive, releaseVersion)
			zip.unzip(agentDir.getParent().getParent().getParent(), agentDeployDir, true)
			
			// un-install the service
			uninstallService()
			
			// install the service again
			installService()
		} else {
			Archive zip = new Archive(archive, releaseVersion)

			// Extract the hot bundles from the archive to the agent
			applyHotBundles(zip)
			updateAgentProfile();

			extractExtraFiles(zip)
			extractExtraBundleFiles(zip)
		}
        updateConfigurations();
    }
 
    /**
     * Extract the given param from the json request, throwing an ACCBadParamException
     * if the value does not exist
     */
    String extractParam(String param) {
        
        String value = requestJson[param]
    
        log.debug("{}={}", param, value)

        if (value == null) {
            throw new ACCBadParamException("Missing value", param)
        }
        return value
    }
    
    String extractOptionalParam(String param, String defaultValue) {
        String value = requestJson[param]
        return (value == null) ? defaultValue : value
    }
    
    Path extractDirParam(String param) {
        String path = extractParam(param)
        
        // Add / to Unix path in case it's missing.
        if (!path.startsWith("/") && !extractParam("os.name").contains("Windows")) {
            path = "/" + path
        }
        Path value = Paths.get(path)
        
        if (!Files.exists(value)) {
            throw new ACCBadParamException("Parameter defined, but file/directory $value does not exist", param)
        }
        return value
    }
    
    def handler(String request) {
        def errs = []

        try {
            log.debug("Raw request: $request")

            // Parse JSON Request, save in the object
            this.requestJson = new groovy.json.JsonSlurper().parseText(request)
            log.info("Request Json: $this.requestJson")

            applyPackageToAgent()

            cleanUp()
        }
        catch (ACCException e) {
            log.error("Error occured: $e")
            errs.push(e.getAccError())
            cleanUp()
        }
        catch (Exception e) {
            /*
             * An exceptional exception occured, so log an error
             */

            log.error("Caught Exception $e")
            e.printStackTrace()

            errs.push(new ACCFrameworkException("Internal Error", e.getMessage()).getAccError())
            cleanUp()
        }

        String response = genJson(errs)

        log.debug("Response: $response")

        return response
    }

    @CompileDynamic
    String genJson(errs) {
        // Create the json message
        def json = new JsonBuilder()
    
        def root = json {
            version "1.0"
            className "KeyValuePairs"
            errors errs
        }

        return json.toString()
    }
    
    def cleanUp(Path file) {
        try {
            if (file != null) {
                log.info("cleanup: removing $file")
				// For debugging purposes, if the below property is set lets keep the package.tmp file
				if (readControllerProperties().getProperty("acc.keep.tmp.package") != null)
					Files.move(file, Paths.get(file.toString() + "." + System.currentTimeMillis()))
				else
					Files.deleteIfExists(file)
            }
        }
        catch (IOException e) {
            log.debug("Failed to delete $e")
        }
    }

    def cleanUp() {
        cleanUp(archive)
    }
}


if (binding.variables.containsKey("request")) {
    logger.info("Running applyUiaPackageRelease.groovy")
    
    ap = new ApplyPackage()
    response = ap.handler(request)
}
else {

    /*
     * To run in test mode, a logging implementation has to be supplied on the cp, 
     * so for example call with something like this:
     * 
     * groovy -cp /usr/share/java/slf4j-simple.jar:/usr/share/java/slf4j-api.jar 
     *     -Dorg.slf4j.simpleLogger.defaultLogLevel=debug ./applyUiaPackageRelease.groovy
     *         /home/hilja07/temp/acc/APMCommandCenterController/
     *         /home/hilja07/temp/acc/wily/ 
     *         acc-controller-package-99.99.accEagle-SNAPSHOT.tar 
     *         http://accdemowin01.ca.com:8888
     *
     * I needed to install this:
     * sudo apt-get install libslf4j-java
     */

    println("Running applyUiaPackageRelease.groovy - TEST MODE")
    
    String accDir
    String agentDir
    String partialUrl
    URL broker
    try {
        accDir = Paths.get(args[0]).toAbsolutePath().toString()      // acc install dir
        agentDir = Paths.get(args[1]).toAbsolutePath().toString()    // agent install dir
        partialUrl = Paths.get(args[2])  // partial path to upgraded controller package - catted to broker url
        broker = new URL(args[3])
    }
    catch (ArrayIndexOutOfBoundsException e) {
    }

    request = "{"
    request += String.format('"acc.controller.dir":"%s", ', groovy.json.StringEscapeUtils.escapeJava( accDir) )  // this property only exists for testing
    request += String.format('"acc.broker.url":"%s", ', groovy.json.StringEscapeUtils.escapeJava( broker.toString().toString() ) )      // this property only exists for testing
    request += String.format('"acc.package.url":"%s", ', groovy.json.StringEscapeUtils.escapeJava( partialUrl.toString() ) )
    request += String.format('"acc.agent.installpath":"%s", ', groovy.json.StringEscapeUtils.escapeJava( agentDir) )
    request += '"acc.package.id":"1", '
    request += '"acc.package.name":"pname", '
    request += '"acc.package.version":"pversion", '
    request += '"com.wily.introscope.agentProfile":"c://sw//groovy//TomcatAcc//wily//core//config//IntroscopeAgent.profile", '
    
    request += '"acc.bundle.file.1.tar":"wily//extensions//deploy//spring-647d40d3-99.99.0.0.tar.gz", '
    request += '"acc.bundle.file.1.pathInTar":"bundle.properties", '

    request += '"acc.bundle.file.2.tar":"wily//extensions//deploy//spring-647d40d3-99.99.0.0.tar.gz", '
    request += '"acc.bundle.file.2.pathInTar":"directives/toggles.pbd", '

    request += '"os.name":"Windows" '
    request += "}"

    ap = new ApplyPackage()
    response = ap.handler(request)

    println("Response is:")
    println(response)

}

