/**
 * Copyright (c) 2016 CA, All rights Reserved.

This software and all information contained therein is confidential and 
proprietary and shall not be duplicated, used, disclosed or disseminated in any 
way except as authorized by the applicable license agreement, without the 
express written permission of CA. All authorized reproductions must be marked 
with this language.  

EXCEPT AS SET FORTH IN THE APPLICABLE LICENSE AGREEMENT, TO THE EXTENT PERMITTED 
BY APPLICABLE LAW, CA PROVIDES THIS SOFTWARE WITHOUT WARRANTY OF ANY KIND, 
INCLUDING WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY OR 
FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT WILL CA BE LIABLE TO THE END USER 
OR ANY THIRD PARTY FOR ANY LOSS OR DAMAGE, DIRECT OR INDIRECT, FROM THE USE OF 
THIS SOFTWARE, INCLUDING WITHOUT LIMITATION, LOST PROFITS, BUSINESS 
INTERRUPTION, GOODWILL, OR LOST DATA, EVEN IF CA IS EXPRESSLY ADVISED OF SUCH 
LOSS OR DAMAGE.

 ***********************************************************************/
package com.ca.fmp.ims.view.dialogs;

import java.util.ArrayList;

import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.PlatformUI;

import com.ca.fmp.ims.common.FileMasterServer;
import com.ca.fmp.ims.model.DatasetResponse;
import com.ca.fmp.ims.operation.RestoreDataSetJob;

/**
 * The RestoreArchiveDatasetsDialog opens up a dialog 
 * box.  It provides the end user with a list of data sets that 
 * needs to be archived.
 * 
 * If the "Restore and Wait" button is pressed,  application
 * sends a requests to mainframe to restore said data sets. It waits for the results.
 * If it is a successful restore operation, it returns OK else CANCEL.
 * 
 * If the "Restore and Do Not Wait" button is pressed, a job is created that
 * sends requests to the mainframe to restore said data sets. It does not
 * wait for the response.  The dialog box returns CANCEL.
 *  
 * If the "Exit" button is pressed, no job is created to send a request.
 * The dialog box returns CANCEL.
 *   
 * @author carel06
 */
public class RestoreArchiveDatasetsDialog extends TitleAreaDialog {
	
    private static String CONTEXT_HELP_ID = "com.ca.fmp.ims.help.archiveWizardPage"; //$NON-NLS-1$       
	private ArrayList<String> archivedDatasetsThatNeedsToBeRestored = new ArrayList<String>();
	
	// carel06
	// the id values of the buttons are random.  
	// I just couldn't use the default (1/2/3) since
	// it would be the default behavior for OK and CANCEL button
	// which we do not want.
	private int restoreAndWaitButtonId = 6000;
	private int restoreAndDoNotWaitButtonId = 6001;
	private int exitButtonId = 6002;

	// suggestion by Sarah in the future
	// maybe add a checkbox ("Run in Background") and 
	// just have one restore button
	
	private String restoreAndWaitButtonLabel = "&Restore Now";
	private String restoreDoNotWaitButtonLabel = "Restore in &Background";
	private String exitButtonLabel = "&Close";
	private String title = "Restore Data Sets";
	private String messageNeedToBeRestored = "Restore the following data sets";
	private String messageBeingRestored = "Restoring the following data sets...";

	private FileMasterServer fileMasterServer = null;
	private TableViewer tableViewer;
	private boolean moveOnFlag = false;
	
	/**
	 * The constructor.
	 * We just need to pass the list of archived datasets that needs to be restored.
	 * 
	 * @param parentShell
	 * @param archivedDatasetsThatNeedsToBeRestored list of archives that need to be restored
	 * @param fileMasterServer the server to connect to.
	 */
	public RestoreArchiveDatasetsDialog(Shell parentShell, ArrayList<String> archivedDatasetsThatNeedsToBeRestored, FileMasterServer fileMasterServer) {
		super(parentShell);		
		
		// we need to set the style for this shell so that 
		// we can have the help tray dialog open up.
        int style = SWT.CLOSE | SWT.TITLE | SWT.MIN | SWT.RESIZE;
        setShellStyle(style | SWT.APPLICATION_MODAL);
        
		this.fileMasterServer = fileMasterServer;
		this.archivedDatasetsThatNeedsToBeRestored = archivedDatasetsThatNeedsToBeRestored;
	}

	/*
	 * We need to pass this dialog to the RestoreDataSetJob 
	 * so that the tableViewer and message can be updated.
	 * Otherwise, we will get Invalid Thread Access issues.
	 */
	private RestoreArchiveDatasetsDialog getThisDialog() {
		return this;
	}
    
	/*
	 * We need to pass this dialog to the RestoreDataSetJob 
	 * so that the tableViewer and message can be updated.
	 * Otherwise, we will get Invalid Thread Access issues.
	 */
	public TableViewer getTableViewer() {
		return tableViewer;
	}
	
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		final Button buttonRestoreWait = createButton(parent, restoreAndWaitButtonId, restoreAndWaitButtonLabel, true);
		final Button buttonRestoreDoNotWait = createButton(parent, restoreAndDoNotWaitButtonId, restoreDoNotWaitButtonLabel, false);
		final Button buttonExit = createButton(parent, exitButtonId, exitButtonLabel, false);

		buttonRestoreWait.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(MouseEvent e) {
				// after pressing the "restore and wait button, the dialog should not 
				// disappear.  just disable the button
				buttonRestoreWait.setEnabled(false);
				buttonRestoreDoNotWait.setEnabled(false);

				setMessage(messageBeingRestored);
				
				final RestoreDataSetJob radj = new RestoreDataSetJob(archivedDatasetsThatNeedsToBeRestored, fileMasterServer, tableViewer, getThisDialog());
				radj.addJobChangeListener(new JobChangeAdapter() {
					@Override
					public void done(IJobChangeEvent event) {
						super.done(event);
						if(radj.isJobSuccessful() == true) {
							moveOnFlag = true;
						}
					}
				});
			}
		});

		buttonRestoreDoNotWait.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(MouseEvent e) {
				// need to invoke the feature to send a request to MF
				setMessage(messageBeingRestored);

                // invoking RestoreDataSetJob (it's a job , using a new Thread)
				// we don't really need the results since
				// we are going back to the DBD DSN wizard page
				new RestoreDataSetJob(archivedDatasetsThatNeedsToBeRestored, fileMasterServer, null, null);

				// we can't assume that the RestoreDataSetJob worked properly so 
				// we just doing a cancel press
				// close();
				cancelPressed();
			}
		});

		buttonExit.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(MouseEvent e) {
				// if the exit button is pressed,
				// we don't invoke any request to the mainframe
				// we just go back to the place where the dialog has
				// been invoked
				if(moveOnFlag == true) {
                    // the moveOnFlag scenario is used bc
					// if it is a successful restore data set operation
					// it should move to the next page.
					okPressed();
				} else {
					cancelPressed();
				}
			}
		});

		return;
	}


	@Override
	protected Control createDialogArea(Composite parent) {

		Composite area = (Composite) super.createDialogArea(parent);
		Composite container = new Composite(area, SWT.NONE);
		container.setLayoutData(new GridData(GridData.FILL_BOTH));
		GridLayout layout = new GridLayout(1, false);
		container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		container.setLayout(layout);
		
		setTitle(title);
		setMessage(messageNeedToBeRestored);
          
		createViewer(parent);

		setHelpAvailable(true);
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, CONTEXT_HELP_ID);
        
		return area;
	}


	private void createViewer(Composite parent) {
		tableViewer = new TableViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		createColumns(parent, tableViewer);
		final Table table = tableViewer.getTable();		                
		table.setLinesVisible(true);
		table.setHeaderVisible(true);

		tableViewer.setContentProvider(new ArrayContentProvider());
		// get the content for the tableViewer, setInput will call getElements in the
		// contentProvider
		
		ArrayList<DatasetResponse> d = new ArrayList<DatasetResponse>();
		for(String x: this.archivedDatasetsThatNeedsToBeRestored) {
			DatasetResponse dr = new DatasetResponse(x, "");
			d.add(dr);
		}	
		tableViewer.setInput(d);
		
		// make the selection available to other views
		// getSite().setSelectionProvider(tableViewer);
		// set the sorter for the table

		// define layout for the tableViewer
		GridData gridData = new GridData();
		gridData.verticalAlignment = GridData.FILL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.horizontalSpan = 2;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		tableViewer.getControl().setLayoutData(gridData);
	}



	// create the columns for the table
	private void createColumns(final Composite parent, final TableViewer viewer) {
		String[] titles = { "Data Set", "Message"};
		int[] bounds = { 300, 100, 100, 100 };

		// first column is for the dataset name
		TableViewerColumn col = createTableViewerColumn(titles[0], bounds[0], 0);
		col.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
		        DatasetResponse p = (DatasetResponse) element;
		        return p.getDataset();				
			}
		});
		
		// second column is error message retrieved from the mainframe
	    col = createTableViewerColumn(titles[1], bounds[1], 1);
	    col.setLabelProvider(new ColumnLabelProvider() {
	      @Override
	      public String getText(Object element) {
	        DatasetResponse p = (DatasetResponse) element;
	        return p.getMessage();
	      }
	    });
	}

	private TableViewerColumn createTableViewerColumn(String title, int bound, final int colNumber) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		final TableColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setWidth(bound);
		column.setResizable(true);
		return viewerColumn;
	}
}
