#!/bin/bash
# vim:et:ft=sh:sts=2:sw=2

# Copyright (C) 2016-2021 VMware, Inc. All Rights Reserved.
#
# Licensed under the GNU General Public License v2 (the "License");
# you may not use this file except in compliance with the License.
# The terms of the License are located in the COPYING file of this distribution.
#

[ -n "$_PHOTON_CUST_SH_" ] && return || readonly _PHOTON_CUST_SH_=1

. "${GOSC_DIR}/CustomizationUtils.sh"

RunCloudConfig()
{
  local configPath=$1

  local ciConfigPath="/tmp/.cloud-config.yaml"

  ConfigFile_LoadConfigFile $configPath

  # Check whether post custom script is present and the file is non empty then
  # Check "enable-custom-script" value from VM Tools configuration:
       # If "enable-custom-script" is absent / not set, then
           # If DEFAULT-RUN-POST-CUST-SCRIPT is yes, then run custom script.
           # If DEFAULT-RUN-POST-CUST-SCRIPT is no or not set, then report error and abort the guest customization.
       # If "enable-custom-script" is explicitly set, then enable/disable the post script per "enable-custom-scripts" value.
           # If “enable-custom-script” is set to true, then run custom script.
           # If “enable-custom-script” is set to false, then report error and abort the guest customization.

  local postScript=$(ConfigFile_GetCustomScriptName)
  if [[ -n "$postScript" ]]; then
     postScript=$(${DIRNAME} $configPath)"/$postScript"
     if [[ -f "$postScript" && -s "$postScript" ]]; then
        local value=$(GetToolsConfig "deployPkg" "enable-custom-scripts")
        if (${ECHO} $value | ${GREP} -iqwF "not set"); then
           local forceCustomScript=$(ConfigFile_GetForceRunPostCustScript)
           Debug "DEFAULT-RUN-POST-CUST-SCRIPT: [$forceCustomScript]"
           if [[ -z "$forceCustomScript" || $forceCustomScript == 0 ]]; then
              PostDeployPkgStatus "enable-custom-scripts is not set and DEFAULT-RUN-POST-CUST-SCRIPT is no or not set"
              Die "enable-custom-scripts is not set and DEFAULT-RUN-POST-CUST-SCRIPT is no or not set. Failing customization"
           fi
        elif (${ECHO} $value | ${GREP} -iqwF "false"); then
           PostDeployPkgStatus "enable-custom-scripts is set to false"
           Die "enable-custom-scripts is set to false. Failing customization"
        fi
     fi
  fi

  local tmpOvfEnvFile="/tmp/gctemp.xml"

  local scriptName=$(ConfigFile_GetCustomScriptName)

  if [[ -n "$scriptName" ]]; then
    scriptName=$(${DIRNAME} $configPath)"/$scriptName"
    Info "Handling customization script [$scriptName]"
    RunCustomScript $scriptName
  else
    Info "No customization script to run"
  fi

  local utc=$(ConfigFile_GetUtc)
  Debug "UTC: [$utc]"
  if [[ -z "$utc" ]]; then
    Debug "Not changing UTC"
  elif [[ $utc == 1 ]]; then
    Exec "${TIMEDATECTL} set-local-rtc 0" '' ''
    Exec "${HWCLOCK} --systohc" '' ''
  else
    Exec "${TIMEDATECTL} set-local-rtc 1" '' ''
    Exec "${HWCLOCK} --systohc" '' ''
  fi

  local tz=$(ConfigFile_GetTimeZone)
  Debug "Timezone: [$tz]"
  if [[ -z "$tz" ]]; then
    Debug "Not changing Timezone"
  else
    Exec "${TIMEDATECTL} set-timezone $tz" '' ''
  fi

  ${SLEEP} 2

  local photon_ver="$(${GREP} -w "VERSION=" /etc/os-release | cut -d= -f2)"
  # cloud-init on Photon 1.0 and 2.0 doesn't have clean option support
  if [ "${photon_ver}" == "\"1.0\"" -o  "${photon_ver}" == "\"2.0\"" ]; then
    ${RMD} /var/lib/cloud/instance* /var/lib/cloud/seed
  else
    ${CLOUD_INIT} clean -s
  fi

  GenerateCloudInitConfig content "GetInterfaceByMacAddressIPAddrShow" $tmpOvfEnvFile
  # Once the cloud-config file been generated, remove the existing network config files.
  ${RM} /etc/systemd/network/*

  # If you don't quote the variable when you echo it, newlines are lost.
  ${ECHO} "${content}" > $ciConfigPath

  cloud-init --file $ciConfigPath init 2>&1

  Exec "${SYSTEMCTL} restart systemd-networkd" '' ''

  CustomizeHostsFile

  local adminPwd=$(ConfigFile_GetAdminPassword)
  if [[ -n "$adminPwd" ]]; then
     adminPwd=$(${ECHO} $adminPwd | ${BASE64} -di)
     Info "Updating root password..."
     ${ECHO} -e "${adminPwd}\n${adminPwd}" | ${PASSWD} root &> /dev/null
     local exitCode=$?
     if [[ $exitCode -ne 0 ]]; then
        Die "Password update failed"
     fi
     Info "Root password updated successfully..."
  fi

  if [[ $(ConfigFile_GetResetPassword) -eq 1 ]]; then
    Info "Resetting the password..."
    #'chage -d 0 U'/'passwd --expire U' breaks login due to missing PAM,
    #see https://github.com/coreos/bugs/issues/164
    local shadow=$(</etc/shadow)
    shadow=$(ExpireShadowRoot "$shadow")
    ${ECHO} "${shadow}" > /etc/shadow
  fi

  # Execute all commands present in blob service file
  ExecuteAllBlobServiceFileCommands

  # Call lightwave domain join
  LightWaveDomainJoin

  # Mount NFS if specified in config
  MountNFSDevice

  local machineId=$(GetBiosUuid "")
}

PhotonCustomization_Customize()
{
  local configFilePath=$1

  Info "Photon customization started"

  #Set Guest Customization status as Started
  PostGCStatus "Started"

  RunCloudConfig $configFilePath

  Info "Photon customization finished"

  # Run Post customization script

  local scriptName=$(ConfigFile_GetCustomScriptName)

  if [[ -n "$scriptName" ]]; then
    scriptName=$(${DIRNAME} $configFilePath)"/$scriptName"
    Info "Handling Post Customization script [$scriptName]"
    RunPostCustomScript $scriptName
  else
    Info "No post customization script to run"
  fi

  #Set Guest customization status as successful
  PostGCStatus "Successful"

  Info "Photon Customization Successful. GC Status is updated"
}
