#!/bin/bash
# vim:et:ft=sh:sts=2:sw=2

# Copyright (C) 2020 VMware, Inc. All Rights Reserved.
#
# Licensed under the GNU General Public License v2 (the "License");
# you may not use this file except in compliance with the License.
# The terms of the License are located in the COPYING file of this distribution.
#

# Unique names for "result" variables are used to not collide with any outer ones

# Wrong path causes ExitCode 1 inside the package

# OS-agnostic utility paths
DATE=/bin/date
ECHO=/bin/echo
GREP=/bin/grep
CP=/bin/cp
MV="/bin/mv -f"
SH=/bin/bash
RM="/bin/rm -f"
RMD="/bin/rm -rf"
TEE=/usr/bin/tee
CAT=/bin/cat
MKDIR=/usr/bin/mkdir
TOUCH=/bin/touch
CHMOD=/bin/chmod
TR=/usr/bin/tr
AWK="/usr/bin/awk"

. "$GOSC_DIR/Debug.sh"

# Executes the command.
#
# Optionally captures the output into the result variable.
#
# NOTE: should not be used in functions returning result using 'echo'. Only in those using 'eval'.
#
# Args:
#   cmd: string: the command to be executed
#   resultVar: string: name of the result variable [optional]
# Results:
#   None
# Throws:
#   None
Exec()
{
  local cmd="$1"
  local resultVar=$2
  local exitCodeVar=$3

  # must be declared before used
  local execResult='' # unique name

  Debug "Command: [$cmd]"

  # backticks should not be used, because they ignore pipe redirection and can't be nested

  # See http://stackoverflow.com/questions/4668640/how-to-execute-command-stored-in-a-variable on why this works:
  #
  # Using eval "$cmd" does nothing until the quote removal phase, where $cmd is returned as is, and passed as a parameter to eval,
  # whose function is to run the whole chain again before executing.
  execResult=$(eval $cmd)
  local errCode=$?

  Debug "Result: [$execResult]"

  if [[ $errCode -eq 0 ]]; then
    Debug "Exit Code: $errCode"
  else
    Warn "Exit Code: $errCode"
  fi

  if [[ -n "$resultVar" ]]; then
    eval $resultVar="'$execResult'"
  fi

  if [[ -n "$exitCodeVar" ]]; then
    eval $exitCodeVar="'$errCode'"
  fi
}

# Trims the input string.
#
# Args:
#   1: string: the input string
# Results:
#   echo'ed trimmed string
# Throws:
#   None
Trim()
{
  # '+' notation is part of 'Parameter Expansion' and prevents 'unbound variable' warning in tests
  (
    [[ $1 =~ [[:space:]]*([^[:space:]]|[^[:space:]].*[^[:space:]])[[:space:]]* ]]

    if eval '[ ${'BASH_REMATCH'[1]+bugaga} ]'; then
      echo -n "${BASH_REMATCH[1]}"
    fi
  )
}

# Interrupts the program with the error message.
#
# Args:
#   1: string: the error message
# Results:
#   None
# Throws:
#   None
Die()
{
  Error "$@"
  echo "DIE: $@" 1>&2
  # Set Guest customization status as failure
  PostGCStatus "Failed"
  exit 1
}

# Strips <#> comments from the line.
#
# Args:
#   line: string: the input line
# Results:
#   echo'ed line without <#> comments
# Throws:
#   None
GetLineWithoutComments()
{
  local line=$1

  if [[ $line =~ ^([^#]*)#.* ]]; then
    line="${BASH_REMATCH[1]}"
  fi

  line=$(Trim "$line")

  echo "$line"
}

# Replaces pattern with a string in the multi-line input.
#
# Args:
#   regex: string: the pattern to look for
#   replaceWith: string: the string to replace the pattern with
#   lines: string: multi-line input to look for the pattern
#   outputVar: string: name of the result variable
# Results:
#   None
# Throws:
#   None
ReplaceOrAppendInLines()
{
  local regex=$1
  local replaceWith=$2
  local lines=$3
  local outputVar=$4

  local inpLine=''
  local output=''

  local found=0

  OIFS=$IFS
  # preserve the whitespace
  IFS=''
  while read inpLine
  do
    # remove end char \n (chomp)
    inpLine="${inpLine%\\n}"

    #echo "Line (inp): $inpLine"

    if [[ $(GetLineWithoutComments "$inpLine") =~ $regex ]]; then
      export output="$output$replaceWith"$'\n'
      found=1
    else
      export output="$output$inpLine"$'\n'
    fi
  done <<<"$lines"
  IFS=$OIFS

  if [[ $found -eq 0 ]]; then
    export output="$output$replaceWith"$'\n'
  fi

  eval $outputVar="'$output'"
}

# Adds or replaces line in a file.
#
# NOTE: needs to be executed in its own shell () or it will mask some vars.
#
# Args:
#   filePath: string: path to the file
#   oldLine: string: the line to look for
#   newLine: string: the line to replace with
# Results:
#   None
# Throws:
#   None
AddOrReplaceInFile()
{
  local filePath="$1"
  local oldLine="$2"
  local newLine="$3"

  local lines=''
  local content=''

  if [[ -f "$filePath" ]]; then
    Debug "Adding or replacing lines in existing $filePath"
    lines=$(<$filePath)
  fi

  ReplaceOrAppendInLines "$oldLine" "$newLine" "$lines" content
  echo "${content}" > $filePath

  Debug "Patched [$oldLine] with [$newLine] in $filePath"
}

# Converts string to lower-case.
#
# Requires Bash 4.0.
#
# Args:
#   1: string: value
# Results:
#   string: lower-cased value
# Throws:
#   None
ToLowerCase()
{
  echo "${1,,}"
}

# Checks whether provided MAC is valid.
#
# Args:
#   macAddress: string: value
# Results:
#   boolean: 1 if valid, otherwise - 0.
# Throws:
#   None
IsValidMacAddress()
{
  local macAddress="$1"

  if [[ "$macAddress" =~ ^([a-fA-F0-9]{2}:){5}[a-zA-Z0-9]{2}$ ]]; then
    echo 1
  else
    echo 0
  fi
}

# Determines path to the 'ip' utility.
#
# Args:
#   ipcmd: string: 'whereis ip' output, optional
# Results:
#   string: path to 'ip', e.g. /usr/bin/ip or /bin/ip.
# Throws:
#   Dies in case can't find 'ip'.
GetIpPath()
{
  local ipcmd=$1

  if [[ -z "$ipcmd" ]]; then
    ipcmd=$(whereis ip)
  fi

  OIFS=$IFS
  IFS=' '
  for x in $ipcmd; do
    if [[ $x == 'ip:' ]]; then
      continue
    else
      echo "$x"
      IFS=$OIFS
      return
    fi
  done
  IFS=$OIFS

  Die "Can't figure out path to 'ip'"
}
