# !/bin/bash
# vim:et:ft=sh:sts=2:sw=2

# Wrong path causes ExitCode 1 inside the package
. "$GOSC_DIR/Utils.sh"

# Global variable containing key/value pairs. Requires Bash 4.0.
# See http://www.artificialworlds.net/blog/2012/10/17/bash-associative-array-examples/
declare -A CONFIG_DATA

# The IPv4 configuration mode which directly represents the user's goal.
#
# This mode effectively acts as a contract of the in-guest customization engine. It must
# be set based on what the user has requested via VMODL/generators API and should not be
# changed by those layers. It's up to the in-guest engine to interpret and materialize
# the user's request.
#
# Also defined in linuxconfiggenerator.h.

# The legacy mode which only allows dhcp/static based on whether IPv4 addresses list is empty or not
IPV4_MODE_BACKWARDS_COMPATIBLE='BACKWARDS_COMPATIBLE'
# IPv4 must use static address. Reserved for future use
IPV4_MODE_STATIC='STATIC'
# IPv4 must use DHCPv4. Reserved for future use
IPV4_MODE_DHCP='DHCP'
# IPv4 must be disabled
IPV4_MODE_DISABLED='DISABLED'
# IPv4 settings should be left untouched. Reserved for future use
IPV4_MODE_AS_IS='AS_IS'

# Removes all the properties.
#
# Args:
#   None
# Results:
#   None
# Throws:
#   None
ConfigFile_Clear()
{
  local key=

  for key in "${!CONFIG_DATA[@]}"; do
    unset CONFIG_DATA["$key"]
  done
}

# Inserts k/v pair.
#
# Args:
#   key: string: key
#   val: string: value
# Results:
#   None
# Throws:
#   None
ConfigFile_InsertKey()
{
  local key=$1
  local val=$2

  # cleaning up on all "input" paths
  key="${key%\\n}"
  key=$(Trim "$key")

  val="${val%\\n}"
  val=$(Trim "$val")

  CONFIG_DATA["$key"]="$val"
}

# Determines properties count.
#
# Args:
#   None
# Results:
#   integer: properties count
# Throws:
#   None
ConfigFile_Size()
{
  echo ${#CONFIG_DATA[@]}
}

# Parses properties from a .cfg file content.
#
# Any previously available properties will be removed.
#
# Sensitive data will not be logged in case key starts from '-'.
#
# Args:
#   content: string: e.g. content of config/cust.cfg
# Results:
#   None
# Throws:
#   None
ConfigFile_LoadConfigContent()
{
  local content="$1"

  local category=""
  local line=
  local lineLength=
  local key=
  local val=

  ConfigFile_Clear

  while read line
  do
    # remove end char \n (chomp)
    line="${line%\\n}"

    #TODO validate against a set of allowed characters (not done in Perl)

    # "sensitive" settings shall not be logged
    if [[ $line == -* ]]; then
        canLog=0
      else
        canLog=1
      fi

    if [[ $canLog -eq 1 ]]; then
      Debug "Processing line: '$line'"
    else
      Debug "Processing line: '***********************'"
    fi

    # spaces at the end are not allowed, things like passwords must be at least base64-encoded
    line=$(Trim "$line")

    if [[ -z "$line" ]]; then
      Debug "Empty line. Ignored."
      continue
    fi

    if [[ $line == '#'* ]]; then
      Debug "Comment found. Line ignored."
      continue
    fi

    if [[ $line =~ \[(.+)\] ]]; then
      Debug "FOUND CATEGORY = ${BASH_REMATCH[1]}"
      category=${BASH_REMATCH[1]}
    # POSIX.2 regex doesn't support non-greedy like in (.+?)=(.*)
    # key value pair (non-eager '=' for base64)
    elif [[ $line =~ ([^=]+)=(.*) ]]; then
      key="${BASH_REMATCH[1]}"
      val="${BASH_REMATCH[2]}"

      # cleaning up on all "input" paths
      key=$(Trim "$key")
      val=$(Trim "$val")

      key="$category|$key"

      # "sensitive" settings shall not be logged
      if [[ $canLog -eq 1 ]]; then
        Debug "ADDED KEY-VAL :: '$key' = '$val'"
      else
        Debug "ADDED KEY-VAL :: '$key' = '*****************'"
      fi

      CONFIG_DATA["$key"]="$val"
    else
      Die "Config file line unrecognizable. Line : '$line'"
    fi

  done <<<"$content"
}

# Parses properties from a .cfg file.
#
# Any previously available properties will be removed.
#
# Sensitive data will not be logged in case key starts from '-'.
#
# Args:
#   filename: string: full path to a .cfg file
# Results:
#   None
# Throws:
#   None
ConfigFile_LoadConfigFile()
{
  local filename=$1

  Info "Opening file name $filename."

  local content=$(<$filename)

  ConfigFile_LoadConfigContent "$content"
}

# Determines whether a property with a given key exists.
#
# Args:
#   key: string: key
# Results:
#   integer: 1 if such property exists, otherwise - 0.
# Throws:
#   None
ConfigFile_HasKey()
{
  local key=$1

  # '+' notation works as part of 'Parameter Expansion'
  if eval '[ ${'CONFIG_DATA'[$key]+bugaga} ]'; then
    echo 1
  else
    echo 0
  fi
}

# Determines whether a value for a property must be kept.
#
# If the property is missing, it's treated as it should be not changed by the engine.
#
# Args:
#   key: string: key
# Results:
#   integer: 1 if property must be kept, otherwise - 0.
# Throws:
#   None
ConfigFile_KeepCurrentValue()
{
  local key=$1

  # helps to distinguish from "empty" value which is used to indicate "removal"
  if [[ $(ConfigFile_HasKey $key) -eq 0 ]]; then
    echo 1
  else
    echo 0
  fi
}

# Determines whether a value for a property must be removed.
#
# If the property is empty, it's treated as it should be removed by the engine.
#
# Args:
#   key: string: key
# Results:
#   integer: 1 if property must be removed, otherwise - 0.
# Throws:
#   None
ConfigFile_RemoveCurrentValue()
{
  local key=$1

  # helps to distinguish from "missing" value which is used to indicate "keeping unchanged"
  if [[ $(ConfigFile_HasKey $key) -eq 1 ]]; then
    local val=${CONFIG_DATA["$key"]}
    if [[ -z "$val" ]]; then
      echo 1
    else
      echo 0
    fi
  else
    echo 0
  fi
}

# Retrieves value of an optional string.
#
# Args:
#   key: string: key
# Results:
#   string: value or empty string
# Throws:
#   None
ConfigFile_GetOptionalString()
{
  local key=$1

  # this key is optional
  if [[ $(ConfigFile_HasKey $key) -eq 1 ]]; then
    local val=${CONFIG_DATA["$key"]}
  else
    local val=""
  fi

  echo "$val"
}

# Verifies that the key is present.
#
# Args:
#   key: string: key
# Results:
#   None
# Throws:
#   Dies in case the key is not present.
ConfigFile_VerifyMandatory()
{
  local key=$1

  if [[ $(ConfigFile_HasKey $key) -eq 0 ]]; then
    Die "Mandatory [$key] is missing."
  fi
}

# Verifies that the key's value is empty or in the specified set.
#
# Args:
#   key: string: key
#   val: string: a possibly prepocessed value
#   enum: string: '|' separated set of possible values (case-sensitive)
# Results:
#   None
# Throws:
#   Dies in case the value is not in the specified set.
ConfigFile_VerifyEnum()
{
  local key=$1
  local val=$2
  local enum=$3

  if [[ -n "$val" ]] && [[ -n "$enum" ]]; then
    local i=
    OIFS=$IFS
    IFS='|'
    for i in $enum; do
      if [[ $val == "$i" ]]; then
        return
      fi
    done
    IFS=$OIFS

    Die "The value [$val] for [$key] is invalid."
  fi
}

# Retrieves value of an optional non-empty string.
#
# Args:
#   key: string: key
# Results:
#   string: value or empty string in case it's missing
# Throws:
#   Dies in case the key was specified, but the value is empty.
ConfigFile_GetOptionalNonEmptyString()
{
  local key=$1
  local val='' # has to be declared before assigned

  if [[ $(ConfigFile_HasKey $key) -eq 1 ]]; then
    val=$(ConfigFile_GetOptionalString $key)
    if [[ -z "$val" ]]; then
      Die "The value [$val] for [$key] is invalid."
    fi
  fi

  echo "$val"
}

# Retrieves value of a mandatory non-empty string.
#
# Args:
#   key: string: key
# Results:
#   string: value
# Throws:
#   Dies in case the key was not specified or its value is empty.
ConfigFile_GetMandatoryNonEmptyString()
{
  local key=$1

  local val='' # has to be declared before assigned

  (ConfigFile_VerifyMandatory $key) || exit 1

  val=$(ConfigFile_GetOptionalNonEmptyString $key) || exit 1

  echo "$val"
}

# Retrieves value of a mandatory possibly empty string.
#
# Args:
#   key: string: key
# Results:
#   string: value
# Throws:
#   Dies in case the key was not specified.
ConfigFile_GetMandatoryString()
{
  local key=$1

  (ConfigFile_VerifyMandatory $key) || exit 1

  echo "$(ConfigFile_GetOptionalString $key)"
}

# Retrieves value of an optional boolean.
#
# Args:
#   key: string: key
# Results:
#   boolean: 1 for yes/YES, 0 for no/NO, an empty string if it wasn't specified
# Throws:
#   Dies in case the key was specified, but is not yes/YES or no/NO.
ConfigFile_GetOptionalBoolean()
{
  local key=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString $key) || exit 1

  val=$(ToLowerCase "$val")

  (ConfigFile_VerifyEnum $key "$val" "yes|no") || exit 1

  if [[ -n "$val" ]]; then
    if [[ $val == 'yes' ]]; then
      val=1
    elif [[ $val == 'no' ]]; then
      val=0
    fi
  fi

  echo "$val"
}

# Retrieves count of the key starting with specified prefix.
#
# Args:
#   key: string: key prefix
# Results:
#   integer: count or 0 in case no properties with such prefix exist
# Throws:
#   None
ConfigFile_GetCnt()
{
  local key=$1

  local cnt=0
  local tmpKey=

  for tmpKey in "${!CONFIG_DATA[@]}"; do
    #echo "'$tmpKey' : '${CONFIG_DATA["$tmpKey"]}'"
    if [[ $tmpKey == "$key"* ]]; then
      cnt=$((cnt + 1))
    fi
  done

  echo "$cnt"
}

# Getting data

# Retrieves hostname.
#
# Args:
#   None
# Results:
#   string: hostname
# Throws:
#   Dies in case setting is present but empty or greater than 63 chars.
ConfigFile_GetHostName()
{
  local key="NETWORK|HOSTNAME"
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString $key) || exit 1

  local length=${#val}

  if [[ $length -gt 63 ]]; then
    Die "The value [$val] for [$key] is invalid."
  fi

  #TODO implement RFC 952/1123 chars validations (not implemented in Perl)

  echo "$val"
}

# Retrieves timezone.
#
# Args:
#   None
# Results:
#   string: timezone
# Throws:
#   Dies in case timezone is present but empty.
ConfigFile_GetTimeZone()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString 'DATETIME|TIMEZONE') || exit 1

  echo "$val"
}

# Retrieves whether to set time to UTC or Local.
#
# Args:
#   None
# Results:
#   boolean: 1 for yes/YES, 0 for no/NO, an empty string if it wasn't specified
# Throws:
#   Dies in case setting is present, but is not yes/YES or no/NO.
ConfigFile_GetUtc()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalBoolean 'DATETIME|UTC') || exit 1

  echo "$val"
}

# Retrieves root password to be set.
#
# Args:
#   None
# Results:
#   string: base64-encoded root password
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetAdminPassword()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString 'PASSWORD|-PASS') || exit 1

  echo "$val"
}

# Retrieves whether to reset root password.
#
# Args:
#   None
# Results:
#   boolean: 1 for yes/YES, 0 for no/NO, an empty string if it wasn't specified
# Throws:
#   Dies in case reset setting is present, but is not yes/YES or no/NO.
ConfigFile_GetResetPassword()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalBoolean "PASSWORD|RESET") || exit 1

  echo "$val"
}

# Retrieves custom script's relative path.
#
# Args:
#   None
# Results:
#   string: relative path
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetCustomScriptName()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "CUSTOM-SCRIPT|SCRIPT-NAME") || exit 1

  echo "$val"
}

# Retrieves domianname.
#
# Args:
#   None
# Results:
#   string: domainname or empty string in case it's missing or empty
# Throws:
#   None
ConfigFile_GetDomainName()
{
  echo "$(ConfigFile_GetOptionalString 'NETWORK|DOMAINNAME')"
}

# Retrieves whether to DNS should be set from DHCP.
#
# Args:
#   None
# Results:
#   boolean: 1 for yes/YES, 0 for no/NO, an empty string if it wasn't specified
# Throws:
#   Dies in case setting is present, but is not yes/YES or no/NO.
ConfigFile_GetDnsFromDhcp()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalBoolean "DNS|DNSFROMDHCP") || exit 1

  echo "$val"
}

# Retrieves customization marker.
#
# Args:
#   None
# Results:
#   string: marker
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetMarkerId()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "MISC|MARKER-ID") || exit 1

  echo "$val"
}

# Retrieves whether to post GOSC status via guestInfo.gc.status.
#
# Args:
#   None
# Results:
#   boolean: 1 for yes/YES, 0 for no/NO, an empty string if it wasn't specified
# Throws:
#   Dies in case setting is present, but is not yes/YES or no/NO.
ConfigFile_GetPostGcStatus()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalBoolean "MISC|POST-GC-STATUS") || exit 1

  echo "$val"
}

# Retrieves NICs count.
#
# Args:
#   None
# Results:
#   integer: count
# Throws:
#   Dies in case setting is not present.
ConfigFile_GetNicsCnt()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "NIC-CONFIG|NICS") || exit 1

  local myresult=0

  if [[ -n "$val" ]]; then
    local spl=(${val//,/ })
    myresult="${#spl[@]}"
  fi

  echo "$myresult"
}

# Retrieves NIC ID by index.
#
# Args:
#   index: integer: index ranging from 1 to NICs count
# Results:
#   integer: count
# Throws:
#   Dies in case NICs setting is not present or is empty.
ConfigFile_GetNicByIndex()
{
  local nicIndex=$1

  nicIndex=$((nicIndex-1))

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetMandatoryNonEmptyString "NIC-CONFIG|NICS") || exit 1

  local spl=(${val//,/ })
  local myresult="${spl[$nicIndex]}"

  echo "$myresult"
}

# Retrieves NIC's MAC address.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   string: MAC or empty string if missing
# Throws:
#   Dies in case setting is present but empty or is not a valid MAC.
ConfigFile_GetMacAddress()
{
  local nic=$1

  local key="${nic}|MACADDR"
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString $key) || exit 1

  val=$(ToLowerCase "$val")

  if [[ -n "$val" ]] && [[ $(IsValidMacAddress "$val") -eq 0 ]]; then
    Die "The value [$val] for [$key] is invalid."
  fi

  echo "$val"
}

# Retrieves NIC's boot protocol.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   string: dhcp/static or empty string if missing
# Throws:
#   Dies in case setting is present but empty or is not dhcp/DHCP or static/STATIC.
ConfigFile_GetBootProto()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "${nic}|BOOTPROTO") || exit 1

  val=$(ToLowerCase "$val")

  ConfigFile_VerifyEnum "${nic}|BOOTPROTO" "$val" "dhcp|static"
  if [[ $? -gt 0 ]]; then exit 1; fi

  echo "$val"
}

# Retrieves NIC's IPv4 address.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetFirstIpV4Addr()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "${nic}|IPADDR") || exit 1

  echo "$val"
}

# Retrieves NIC's IPv4 mask.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetFirstIpV4Mask()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "${nic}|NETMASK") || exit 1

  echo "$val"
}

# Retrieves NIC's IPv4 gateway.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetFirstIpV4Gw()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "${nic}|GATEWAY") || exit 1

  echo "$val"
}

# Retrieves NIC's IPv6 addresses count.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   integer: count or 0
# Throws:
#   None
ConfigFile_GetIpV6Cnt()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetCnt "$nic|IPv6ADDR|")
  if [[ $? -gt 0 ]]; then exit 1; fi

  echo "$val"
}

# Retrieves NIC's IPv6 address by index.
#
# Args:
#   nic: string: NIC's ID
#   ind: integer: setting index
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetIpV6AddrByInd()
{
  local nic=$1
  local ind=$2

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "$nic|IPv6ADDR|$ind") || exit 1

  echo "$val"
}

# Retrieves NIC's IPv6 prefix by index.
#
# Args:
#   nic: string: NIC's ID
#   ind: integer: setting index
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetIpV6PrefixByInd()
{
  local nic=$1
  local ind=$2

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "$nic|IPv6NETMASK|$ind") || exit 1

  echo "$val"
}

# Retrieves NIC's IPv6 gateways count.
#
# IPv6 gateways can be set independently of IPv6 addresses.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   integer: count or 0
# Throws:
#   None
ConfigFile_GetIpV6GwCnt()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetCnt "$nic|IPv6GATEWAY|")
  if [[ $? -gt 0 ]]; then exit 1; fi

  echo "$val"
}

# Retrieves NIC's IPv6 gateway by index.
#
# Args:
#   nic: string: NIC's ID
#   ind: integer: setting index
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetIpV6GwByInd()
{
  local nic=$1
  local ind=$2

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "$nic|IPv6GATEWAY|$ind") || exit 1

  echo "$val"
}

# Retrieves DNS Servers count.
#
# Args:
#   None
# Results:
#   integer: count or 0
# Throws:
#   None
ConfigFile_GetNameServersCnt()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetCnt "DNS|NAMESERVER|")
  if [[ $? -gt 0 ]]; then exit 1; fi

  echo "$val"
}

# Retrieves DNS Server by index.
#
# Args:
#   ind: integer: setting index
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetNameServerByInd()
{
  local ind=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "DNS|NAMESERVER|$ind") || exit 1

  echo "$val"
}

# Retrieves DNS Suffixes count.
#
# Args:
#   None
# Results:
#   integer: count or 0
# Throws:
#   None
ConfigFile_GetDNSSuffixesCnt()
{
  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetCnt "DNS|SUFFIX|")
  if [[ $? -gt 0 ]]; then exit 1; fi

  echo "$val"
}

# Retrieves DNS Suffix by index.
#
# Args:
#   ind: integer: setting index
# Results:
#   string: value or empty string if missing
# Throws:
#   Dies in case setting is present but empty.
ConfigFile_GetDNSSuffixByInd()
{
  local ind=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "DNS|SUFFIX|$ind") || exit 1

  echo "$val"
}

# Retrieves NIC's IPv4 mode.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   string: IPV4_MODE_* value or IPV4_MODE_BACKWARDS_COMPATIBLE value if missing
# Throws:
#   Dies in case setting is present but empty or is not IPV4_MODE_*.
ConfigFile_GetIpV4Mode()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalNonEmptyString "${nic}|IPv4_MODE")
  if [[ $? -gt 0 ]]; then exit 1; fi

  if [[ -z "$val" ]]; then
    val=$IPV4_MODE_BACKWARDS_COMPATIBLE
  fi

  ConfigFile_VerifyEnum "${nic}|IPv4_MODE" "$val" \
    "$IPV4_MODE_BACKWARDS_COMPATIBLE|$IPV4_MODE_STATIC|$IPV4_MODE_DHCP|$IPV4_MODE_DISABLED|$IPV4_MODE_AS_IS"
  if [[ $? -gt 0 ]]; then exit 1; fi

  echo "$val"
}

# Retrieves NIC's 'primary' setting.
#
# Args:
#   nic: string: NIC's ID
# Results:
#   boolean: 1/0 or empty string if missing
# Throws:
#   Dies in case setting is present but empty or is not yes/YES or no/NO.
ConfigFile_GetPrimary()
{
  local nic=$1

  local val='' # has to be declared before assigned

  val=$(ConfigFile_GetOptionalBoolean "${nic}|PRIMARY") || exit 1

  echo "$val"
}

# Retrieves 'primary' NIC's ID.
#
# Args:
#   None
# Results:
#   string: ID or empty string if not applicable
# Throws:
#   Dies in case setting is present but empty or is two+ NICs are marked as yes/YES.
ConfigFile_GetPrimaryNic()
{
  local cnt='' # has to be declared before assigned
  local primary=''

  cnt=$(ConfigFile_GetNicsCnt)
  if [[ $? -gt 0 ]]; then exit 1; fi

  local i=''
  for i in $(seq 1 $cnt); do
    local nic=$(ConfigFile_GetNicByIndex $i)
    if [[ $(ConfigFile_GetPrimary $nic) -eq 1 ]]; then
      if [[ -n "$primary" ]]; then
        Die "There can be only one primary NIC defined."
      fi
      primary=$nic
    fi
  done

  echo "$primary"
}
