# !/bin/bash
# vim:et:ft=sh:sts=2:sw=2

# Unique names for "result" variables are used to not collide with any outer ones

# Wrong path causes ExitCode 1 inside the package
. "$GOSC_DIR/ConfigFile.sh"
AWK_XML_PARSER=$GOSC_DIR/getXML.awk

POST_CUSTOMIZATION_TMP_DIR="/usr/share/oem/.customization"
POST_CUSTOMIZATION_TMP_SCRIPT_NAME="$POST_CUSTOMIZATION_TMP_DIR/customize.sh"
POST_REBOOT_PENDING_MARKER="$POST_CUSTOMIZATION_TMP_DIR/.guest-customization-post-reboot-pending"


# Detects the current hostname.
#
# Args:
#   None
# Results:
#   string: hostname
# Throws:
#   None
OldHostNameCmd()
{
  local val=$(hostname 2>/dev/null)

  # remove end char \n (chomp)
  val="${val%\\n}"
  val=$(Trim "$val")

  echo "$val"
}

# Detects the current FQDN.
#
# Args:
#   None
# Results:
#   string: FQDN
# Throws:
#   None
GetResolverFqdn()
{
  local val=$(hostname -f 2>/dev/null)

  # remove end char \n (chomp)
  val="${val%\\n}"
  val=$(Trim "$val")

  echo "$val"
}

# Detects short hostname out of FQDN.
#
# Args:
#   fqdn: string: FQDN
# Results:
#   string: short hostname (e.g. test1 out of test1.vmware.com)
# Throws:
#   None
GetShortnameFromFQDN()
{
  local fqdn=$1

  fqdn=$(Trim "$fqdn")

  if [[ $fqdn =~ ^([^.]+) ]]; then
    fqdn=$(Trim "${BASH_REMATCH[1]}")
  fi

  echo "$fqdn"
}

# Detects domainname out of FQDN.
#
# Args:
#   fqdn: string: FQDN
# Results:
#   string: domainname (e.g. vmware.com out of test1.vmware.com)
# Throws:
#   None
GetDomainnameFromFQDN()
{
  local fqdn=$1

  fqdn=$(Trim "$fqdn")

  if [[ $fqdn =~ ^([^.]+)\.(.+)$ ]]; then
    fqdn=$(Trim "${BASH_REMATCH[2]}")
  fi

  echo "$fqdn"
}

#TODO IPv6 support for /etc/hosts (not implemented in Perl)
#TODO remove old IPs from /etc/hosts (not implemented in Perl)

# Figures out the new hostname.
#
# Based on old hostname and the value provided/not-provided inside the config.
#
# Args:
#   oldHostname: string: old hostname
# Results:
#   string: new hostname
# Throws:
#   None
NewHostName()
{
  local oldHostname=$1

  local cfgHostname=$(ConfigFile_GetHostName)
  local newHostname=$cfgHostname

  if [[ $(ConfigFile_KeepCurrentValue "NETWORK|HOSTNAME") -eq 1 ]]; then
    newHostname=$oldHostname
  fi

  if [[ -z "$newHostname" ]]; then
    Die 'Invalid old hostname'
  fi

  echo "$newHostname"
}

# Figures out the new FQDN hostname.
#
# Based on old hostname/FQDN and the value provided/not-provided inside the config.
#
# Args:
#   oldHostname: string: old hostname
#   oldFQDN: string: old FQDN
# Results:
#   string: new FQDN hostname
# Throws:
#   None
NewFqdnHostName()
{
  local oldHostname=$1
  local oldFQDN=$2

  local cfgHostname=$(ConfigFile_GetHostName)
  # FQDN may not include hostname, prepare to preserve FQDN
  local newHostnameFQDN=$cfgHostname

  if [[ $(ConfigFile_KeepCurrentValue "NETWORK|HOSTNAME") -eq 1 ]]; then
     newHostnameFQDN=$(GetShortnameFromFQDN "$oldFQDN")

     # Old hostname is not resolved and hence old FQDN is not available
     # Use hostname as new FQDN
     if [[ -z "$newHostnameFQDN" ]]; then
        newHostnameFQDN=$oldHostname
     fi
  fi

  echo "$newHostnameFQDN"
}

# Figures out the new domainname.
#
# Based on old FQDN and the value provided/not-provided inside the config.
#
# Args:
#   oldFQDN: string: old FQDN
# Results:
#   string: new domainname
# Throws:
#   None
NewDomainName()
{
  local oldFQDN=$1

  local cfgDomainname=$(ConfigFile_GetDomainName)
  local newDomainname=$cfgDomainname

  if [[ $(ConfigFile_KeepCurrentValue "NETWORK|DOMAINNAME") -eq 1 ]]; then
    newDomainname=$(GetDomainnameFromFQDN "$oldFQDN")
  elif [[ $(ConfigFile_RemoveCurrentValue "NETWORK|DOMAINNAME") -eq 1 ]]; then
    newDomainname=''
  fi

  echo "$newDomainname"
}

# Finds the first IPv4 out of all NICs.
#
# Args:
#   None
# Results:
#   string: the first IPv4 or empty string in case there are no NICs or all are DHCP.
# Throws:
#   Dies in case NICs setting is empty or IPv4 address setting is empty.
FindFirstStaticIpV4()
{
  local ipv4Addr=''
  local nicsCnt='' # needs to be declared before assigned

  nicsCnt=$(ConfigFile_GetNicsCnt) || exit 1

  local i=
  for i in $(seq 1 $nicsCnt);  do
    local nic=$(ConfigFile_GetNicByIndex $i)
    ipv4Addr=$(ConfigFile_GetFirstIpV4Addr $nic) || exit 1

    if [[ -n "$ipv4Addr" ]]; then
      break
    fi
  done

  echo "$ipv4Addr"
}

# Processes content of the existing /etc/hosts file.
#
# Args:
#   hostsFile: string: contents of the existing /etc/hosts file
#   oldHostname: string: old hostname
#   oldFQDN: string: old FQDN
#   newHostname: string: new hostname
#   outputVar: string: name of the content OUT var
#   resultVar: string: name of the result OUT var
# Results:
#   None
# Throws:
#   None
ProcessEtcHosts()
{
  local hostsFile=$1
  local oldHostname=$2
  local oldFQDN=$3
  local newHostname=$4
  local outputVar=$5
  local resultVar=$6

  local hostNameWasSet=0 # unique name

  local inpLine=''
  local lineLength=0

  # Algorithm overview
  # 1.Do not modify '127... ...' entries unless oldhostname is there: programs may fail
  # 2.Do not replace a localhost: localhost should always remain
  # 3.Remove non loopback entries with old hostname (assuming this is the old ip)
  # 4.Remove non loopback entries with new hostname if already there
  # 5.Setting hostname does only replacements of oldhostname
  # 6.Setting FQDN does an insert as first name because FQDN should be there
  # 7.Add new line that is <newip> <newhostname>, if <newip> is available
  # 8.Unless new hostname is set by (5) or (7) add a 127.0.1.1 <newhostname> entry

  while read inpLine
  do
    # remove end char \n (chomp)
    inpLine="${inpLine%\\n}"

    Debug "Line (inp): $inpLine"

    lineLength=${#inpLine}

    # empty line
    if [ $lineLength == 0 ]; then
      Debug "Empty line. Ignored."
      continue
    fi

    inpLine=$(Trim "$inpLine")

    lineLength=${#inpLine}

    # empty line
    if [ $lineLength == 0 ]; then
      Debug "Empty line. Ignored."
      continue
    fi

    # comment
    if [[ $inpLine == '#'* ]]; then
      Debug "Comment found. Line ignored."
      continue
    fi

    inpLine=$(GetLineWithoutComments "$inpLine")

    if [[ $inpLine =~ ^[:space:]*([^[:space:]]+)(.*) ]]; then
      echo "yes: ${BASH_REMATCH[1]} : ${BASH_REMATCH[2]}"
      hostNameWasSet=1
      #TODO implement /ets/hosts algo here if we ever decide to go that route
    else
      echo "no: $inpLine"
    fi
  done <<<"$hostsFile"

  eval $resultVar="'$hostNameWasSet'"
}

# Generates customized /etc/hosts file.
#
# Works fine if no NICs are present.
#
# Args:
#   hostsFile: string: contents of the existing /etc/hosts file
#   oldHostname: string: old hostname
#   oldFQDN: string: old FQDN
#   outputVar: string: name of the content OUT var
# Results:
#   None
# Throws:
#   Dies in case NICs setting is empty or IPv4 address setting is empty.
CustomizeHostsFileContent()
{
  local hostsFile=$1
  local oldHostname=$2
  local oldFQDN=$3
  local outputVar=$4

  local output=''

  # Partial customization - calculate new hostname and new FQDN
  # based on the existing values and new customization spec values
  # Retrieve old hostname and FQDN

  Debug "Old hostname=[$oldHostname]"
  Debug "Old FQDN=[$oldFQDN]"

  local newHostname=$(NewHostName "$oldHostname")
  local newHostnameFQDN=$(NewFqdnHostName "$oldHostname" "$oldFQDN")

  Debug "New hostname=[$newHostname]"
  Debug "New FQDN hostname=[$newHostnameFQDN]"

  local newDomainname=$(NewDomainName "$oldFQDN")

  Debug "New domainname=[$newDomainname]"

  if [[ -n "$newDomainname" ]]; then
    local newFQDN="$newHostnameFQDN.$newDomainname"
  else
    local newFQDN="$newHostnameFQDN"
  fi

  Debug "New FQDN=[$newFQDN]"

  local hostNameWasSet=0 #$(ProcessEtcHosts "$hostsFile" "$oldHostname" "$oldFQDN" "$newHostname" configFileVar output)

  # Add mapping to the customized static ip
  local ipaddr=$(FindFirstStaticIpV4) || exit 1
  if [[ -n "$ipaddr" ]]; then
    local newStaticIPEntry="$ipaddr"$'\t'"$newFQDN"
    if [[ '$newFQDN' != '$newHostname' ]]; then
      newStaticIPEntry="$newStaticIPEntry $newHostname";
    fi

    hostNameWasSet=1
    output="$output""$newStaticIPEntry"$'\n'
    Debug "Static ip entry '$newStaticIPEntry' added"
  fi

  # Add mapping to loopback 127.0.1.1 if new hostname is still not set
  if [[ $hostNameWasSet -eq 0 ]]; then
    # Hostname still not added - use a loopback entry to
    # create mapping

    newLine="127.0.1.1"$'\t'"$newFQDN"
    if [[ '$newFQDN' != '$newHostname' ]]; then
       newLine="$newLine $newHostname"$'\n'
    fi

    output="$output""$newLine"$'\n'
    Debug "Loopback entry '$newLine' added"
  fi

  eval $outputVar="'$output'"
}

# Customizes /etc/hosts file.
#
# Args:
#   None
# Results:
#   None
# Throws:
#   None
CustomizeHostsFile()
{
  local oldHostname=$(OldHostNameCmd)
  local oldFQDN=$(GetResolverFqdn)
  local res=''
  Info "CHF"
  echo $pldHostname
  echo $oldFQDN
 
  CustomizeHostsFileContent "" "$oldHostname" "$oldFQDN" "res"

  Debug "Patching /etc/hosts with [$res]"
  echo "${res}" > /etc/hosts
}

# Detects interface name by MAC address out of 'ip addr show' output.
#
# 'ip addr show' is used instead of ifconfig, because the latter one is becoming deprecated.
#
# Invalid MAC will result into empty string set to the result var.
#
# Args:
#   macAddress: string: MAC
#   ipAddrContentVar string: 'ip addr show' output, optional
#   resultVar: string: name of the result OUT var
# Results:
#   None
# Throws:
#   None
GetInterfaceByMacAddressIPAddrShow()
{
  local macAddress=$1
  # name of the outer variable holding the content
  local ipAddrContentVar=$2
  local resultVar=$3

  macAddress=$(ToLowerCase "$macAddress")

  if [[ $(IsValidMacAddress "$macAddress") -eq 1 ]]; then
    Debug "MAC address [$macAddress] is valid"
  else
    Warn "MAC address [$macAddress] is not valid"
    eval $resultVar="''"
    return
  fi

  local result=""
  local line=
  local prevLine=

  local ipAddrContent=
  local ipAddrResult= # unique name

  if [ -z "$ipAddrContentVar" ]; then
    local ipPath=$(GetIpPath "")
    Exec "$ipPath addr show 2>&1" ipAddrContent ''
  else
    ipAddrContent=${!ipAddrContentVar}
  fi

  ipAddrContent=$(ToLowerCase "$ipAddrContent")

  while read -r line
  do
    #echo "line: '$line'"

    if [[ $line =~ ^link/.+$macAddress.+$ ]]; then
      #echo "found: $prevLine"

      local arrIN=(${prevLine//:/ })

      ipAddrResult=${arrIN[1]}
      ipAddrResult=$(Trim "$ipAddrResult")

      break
    fi

    prevLine=$line
  done <<<"$ipAddrContent"

  eval $resultVar="'$ipAddrResult'"
}

#TODO test MaskStrToBits

# Converts network mask to CIDR notation.
#
# http://rekrowten.wordpress.com/2013/01/09/convert-subnet-mask-to-ipv4-prefix-and-vice-versa-by-unix-script/
#
# Args:
#   1: string: mask
# Results:
#   string: mask in CIDR notation (e.g /16 instead of 255.255.0.0).
# Throws:
#   None
MaskStrToBits()
{
  local tmp=$1

  local bits

  local arr_sm[0]=" 0.0.0.0"
  local arr_sm[1]=" 128.0.0.0"
  local arr_sm[2]=" 192.0.0.0"
  local arr_sm[3]=" 224.0.0.0"
  local arr_sm[4]=" 240.0.0.0"
  local arr_sm[5]=" 248.0.0.0"
  local arr_sm[6]=" 252.0.0.0"
  local arr_sm[7]=" 254.0.0.0"
  local arr_sm[8]=" 255.0.0.0"
  local arr_sm[9]=" 255.128.0.0"
  local arr_sm[10]=" 255.192.0.0"
  local arr_sm[11]=" 255.224.0.0"
  local arr_sm[12]=" 255.240.0.0"
  local arr_sm[13]=" 255.248.0.0"
  local arr_sm[14]=" 255.252.0.0"
  local arr_sm[15]=" 255.254.0.0"
  local arr_sm[16]=" 255.255.0.0"
  local arr_sm[17]=" 255.255.128.0"
  local arr_sm[18]=" 255.255.192.0"
  local arr_sm[19]=" 255.255.224.0"
  local arr_sm[20]=" 255.255.240.0"
  local arr_sm[21]=" 255.255.248.0"
  local arr_sm[22]=" 255.255.252.0"
  local arr_sm[23]=" 255.255.254.0"
  local arr_sm[24]=" 255.255.255.0"
  local arr_sm[25]=" 255.255.255.128"
  local arr_sm[26]=" 255.255.255.192"
  local arr_sm[27]=" 255.255.255.224"
  local arr_sm[28]=" 255.255.255.240"
  local arr_sm[29]=" 255.255.255.248"
  local arr_sm[30]=" 255.255.255.252"
  local arr_sm[31]=" 255.255.255.254"
  local arr_sm[32]=" 255.255.255.255"

  for ((bits=32;bits>=0;bits--)); do
    tmp=`echo $tmp | sed "s/${arr_sm[$bits]}/\/$bits/g"`
  done

  echo "$tmp"
}

# Joins multiple 'countable' config setting using given separator.
#
# Args:
#   cntMethod: string: method to call go fetch settings count
#   indMethod: string: method to call go fetch setting by index
#   joinStr: string: separator
# Results:
#   string: joined result.
# Throws:
#   None
JoinSettings()
{
  local cntMethod=$1
  local indMethod=$2
  local joinStr=$3

  local joinSettingsResult=''

  local cnt=$(ConfigFile_$cntMethod)
  if [[ cnt -gt 0 ]]; then
    for j in $(seq 1 $cnt); do
      if [[ -z "$joinSettingsResult" ]]; then
        joinSettingsResult=$(ConfigFile_$indMethod $j)
      else
        joinSettingsResult=$joinSettingsResult$joinStr$(ConfigFile_$indMethod $j)
      fi
    done
  fi

  echo "$joinSettingsResult"
}

# Customizes DNS inside dhclient's config file either using DHCP or config values.
#
# Args:
#   confPath: string: full path to dhclient's config file
# Results:
#   None
# Throws:
#   Dies in case DNS/Suffix setting is present but empty.
CustomizeDNSFromDHCP()
{
  local confPath=$1

  if [[ -e /sbin/dhclient-script ]] || [[ -e $confPath ]]; then
    local dnsFromDhcp=$(ConfigFile_GetDnsFromDhcp)

    local domainNames=$(JoinSettings GetDNSSuffixesCnt GetDNSSuffixByInd " ")

    if [[ $dnsFromDhcp -eq 0 ]]; then
      if [[ -n "$domainNames" ]]; then
        tmp="supersede domain-name \"$domainNames\";"
        (AddOrReplaceInFile $confPath "supersede domain-name " "$tmp")
      fi

      local domainServers=$(JoinSettings GetNameServersCnt GetNameServerByInd ",")

      if [[ -n "$domainServers" ]]; then
        tmp="supersede domain-name-servers \"$domainServers\";"
        (AddOrReplaceInFile $confPath "supersede domain-name-servers " "$tmp")
      fi
    elif [[ $dnsFromDhcp -eq 1 ]]; then
      (AddOrReplaceInFile $confPath "supersede domain-name " "")

      (AddOrReplaceInFile $confPath "supersede domain-name-servers " "")

      if [[ -n "$domainNames" ]]; then
        tmp="append domain-name \"$domainNames\";"
        (AddOrReplaceInFile $confPath "append domain-name " "$tmp")
      fi
    else
      Die "Uknown dnsFromDhcp: $dnsFromDhcp"
    fi
  else
    Debug "Can't find DHCP configuration"
  fi
}

# Runs pre-customization script.
#
# Args:
#   scriptPath: string: relative path to the script
# Results:
#   None
# Throws:
#   Dies in case execution returns non-zero exit code.
RunCustomScript()
{
  local scriptPath=$1

  Info "RunCustomScript invoked"

  if [[ -e $scriptPath ]]; then
    # Strip any CR characters from the decoded script
    Exec "${CAT} $scriptPath | ${TR} -d '\r' > $scriptPath.tmp" '' ''
    Exec "${CHMOD} u+x $scriptPath.tmp" '' ''

    Info "Executing pre-customization script..."

    ${SH} -c "$scriptPath.tmp precustomization"
    local exitCode=$?
    if [[ $exitCode -ne 0 ]]; then
      Die "Execution of precustomization failed!"
    fi
  else
   Warn "Customization script '$scriptPath' does not exist"
  fi

  Info "RunCustomScript has completed"
}

# Installs post-customization script.
#
# Args:
#   scriptPath: string: full path to the script
# Results:
#   None
# Throws:
#   None
InstallCustomScript()
{
  local scriptPath=$1

  Info "InstallCustomScript invoked"

  Info "Installing post-reboot customization agent into /etc/systemd/system/"

  Exec "${CP} -f $GOSC_DIR/post-customize-guest.sh /etc/systemd/system/" '' ''
  Exec "${CHMOD} +x /etc/systemd/system/post-customize-guest.sh" '' ''

  Info "Scheduling post-customization script"

  Info "Copying post customization script"

  if [[ ! -d $POST_CUSTOMIZATION_TMP_DIR ]]; then
    Exec "${MKDIR} $POST_CUSTOMIZATION_TMP_DIR" '' ''
  fi

  # .tmp was created by RunCustomScript with \r stripped out
  Exec "${CP} $scriptPath.tmp $POST_CUSTOMIZATION_TMP_SCRIPT_NAME" '' ''

  Info "Creating post-reboot pending marker"

  Exec "${RM} $POST_REBOOT_PENDING_MARKER" '' ''
  Exec "${TOUCH} $POST_REBOOT_PENDING_MARKER" '' ''

  Info "InstallCustomScript has completed"
}

# Generates cloud-init configuration files which represents customization.
#
# Args:
#   resultVar: string: name of the result OUT var
#   addrShowVar: string: name of the method to fetch 'ip addr show' output
# Results:
#   string: contents of the file
# Throws:
#   Dies in case one of the config file parameters is not correct.
GenerateCloudInitConfig()
{
  local resultVar=$1
  local addrShowVar=$2
  local tmpOvfEnvFile=$3

  local formatResult= # unique name

  local hostname=$(ConfigFile_GetHostName)

  # using heredocs...
  export formatResult=$(${CAT} <<EOF
#cloud-config

hostname: ${hostname}
EOF
)

  local scriptName=$(ConfigFile_GetCustomScriptName)

  if [[ -n "$scriptName" ]]; then
    export formatResult=$formatResult$(${CAT} <<EOF


photon:
    units:
      - name: post-customize-guest.service
        enable: yes
        content: |
          [Unit]
          Description=GOSC

          [Service]
          TimeoutStartSec=0
          ExecStart=/etc/systemd/system/post-customize-guest.sh

          [Install]
          WantedBy=multi-user.target
EOF
)
  fi

  securitySshRsa=$(GetOvfPropertyValue 'security.ssh-rsa' $tmpOvfEnvFile)

  if [[ -n "$securitySshRsa" ]]; then
    if [[ "$securitySshRsa" != *ssh-rsa* ]]; then
      Debug "Appending 'ssh-rsa ' to the public key"
      securitySshRsa="ssh-rsa $securitySshRsa"
    fi

    export formatResult=$formatResult$(${CAT} <<EOF


ssh_authorized_keys:
  - $securitySshRsa
EOF
)
  fi

  local adminPwd=$(ConfigFile_GetAdminPassword)

  if [[ -n "$adminPwd" ]]; then
    adminPwd=$(echo $adminPwd | /usr/bin/base64 -di)
    
    local adminPwdHash=$(echo $adminPwd | openssl passwd -1 -stdin -salt root)
    export formatResult=$formatResult$(${CAT} <<EOF


users:
    - name: root
      passwd: $adminPwdHash
EOF
)
  fi

  local nicsCnt=$(ConfigFile_GetNicsCnt)

  local i=
  for i in $(seq 1 $nicsCnt); do
    local nic=$(ConfigFile_GetNicByIndex $i)
    local bootProto=$(ConfigFile_GetBootProto $nic)
    local mac=$(ConfigFile_GetMacAddress $nic)

    local ifCfg=
    ${addrShowVar} ${mac} "" ifCfg
    Debug "ifCfg: $ifCfg"

    export formatResult=$formatResult$(${CAT} <<EOF


write_files:
    - path: /etc/systemd/network/ifcfg-$ifCfg.network
      permissions: 0644
      content: |
        [Match]
        Name=$ifCfg

        [Network]
EOF
)

    if [[ $bootProto == "dhcp" ]]; then
      export formatResult=$formatResult$(${CAT} <<EOF

        DHCP=yes
EOF
)
    else # static
      ipv4Addr=$(ConfigFile_GetFirstIpV4Addr $nic)
      ipv4Mask=$(ConfigFile_GetFirstIpV4Mask $nic)
      ipv4Gw=$(ConfigFile_GetFirstIpV4Gw $nic)
      ipv4AddrPrefix=$(MaskStrToBits "$ipv4Addr $ipv4Mask")

      export formatResult=$formatResult$(${CAT} <<EOF

        Address=$ipv4AddrPrefix
        Gateway=$ipv4Gw
EOF
)
      ipv6Cnt=$(ConfigFile_GetIpV6Cnt $nic)
      local j=
      for j in $(seq 1 $ipv6Cnt); do
        ipv6Addr=$(ConfigFile_GetIpV6AddrByInd $nic $j)
        ipv6Pref=$(ConfigFile_GetIpV6PrefixByInd $nic $j)

        export formatResult=$formatResult$(${CAT} <<EOF

        Address=$ipv6Addr/$ipv6Pref
EOF
)
      done

      ipv6GwCnt=$(ConfigFile_GetIpV6GwCnt $nic)
      local j=
      for j in $(seq 1 $ipv6GwCnt); do
        ipv6Gw=$(ConfigFile_GetIpV6GwByInd $nic $j)

        export formatResult=$formatResult$(${CAT} <<EOF

        Gateway=$ipv6Gw
EOF
)
      done
    fi # boot proto

    local dnsCnt=$(ConfigFile_GetNameServersCnt)
    for j in $(seq 1 $dnsCnt); do
      local dns=$(ConfigFile_GetNameServerByInd $j)
      export formatResult=$formatResult$(${CAT} <<EOF

        DNS=$dns
EOF
)
    done

    local suffixCnt=$(ConfigFile_GetDNSSuffixesCnt)
    for j in $(seq 1 $suffixCnt); do
      local searchPath=$(ConfigFile_GetDNSSuffixByInd $j)
      
      export formatResult=$formatResult$(${CAT} <<EOF

        Domains=$searchPath
EOF
)
    done

    local dnsFromDhcp=$(ConfigFile_GetDnsFromDhcp)
    if [[ $dnsFromDhcp -eq 1 ]]; then
      dnsFromDhcp='true'
    else
      dnsFromDhcp='false'
    fi

    export formatResult=$formatResult$(${CAT} <<EOF


        [DHCP]
        UseDNS=$dnsFromDhcp
EOF
)
  done # NIC

  eval $resultVar="'$formatResult'"
}

# http://www.tldp.org/LDP/lame/LAME/linux-admin-made-easy/shadow-file-formats.html
ExpireShadowUser()
{
  local line=$1

  local out=''

  local i=1

  OIFS=$IFS
  IFS=':'
  for x in $line; do
    if [[ $i -eq 1 ]]; then
      out=$x
    elif [[ $i -eq 3 ]]; then
      out="$out:0"
    elif [[ $i -eq 5 ]]; then
      out="$out:99999"
    else
      out="$out:$x"
    fi
    i=$((i+1))
  done
  IFS=$OIFS
  out="$out:$x"

  echo "$out"
}

ExpireShadowRoot()
{
  local shadowContent=$1

  local line=

  local out=''

  while read -r line
  do
    if [[ -z "$out" ]]; then
      if [[ $line == "root:"* ]]; then
        line=$(ExpireShadowUser $line)
      fi
      out=$line
    else
      if [[ $line == "root:"* ]]; then
        line=$(ExpireShadowUser $line)
      fi
      out=$out$'\n'$line
    fi
  done <<<"$shadowContent"

  echo "$out"
}

GetBiosUuid()
{
  local dmidecodeContent=$1

  if [[ -z "$dmidecodeContent" ]]; then
    dmidecodePath=/bin/systemd-machine-id-setup
    if [[ ! -f $dmidecodePath ]]; then
      Die "$dmidecodePath does not exist"
    fi
    Info "DMIDECODEPATH : $dmidecodePath"
    Exec '$dmidecodePath'
  fi
}


GetToolsDaemonPath()
{
   echo "/usr/bin/vmtoolsd"
}

# This function runs vmware-guestd to extract OVF parameters and create a XML file
FetchOvfEnv() {
  local tempPath=$1
  local ovfEnv=$2
  Info "OVSENV"
  echo $ovfEnv

  if [[ -z "$ovfEnv" ]]; then
    local vmtoolsdPath=$(GetToolsDaemonPath)
    local rndPath="/tmp/$RANDOM"
    Info "VMTOOLSDPATH"
    echo $vmtoolsdPath
    ${vmtoolsdPath} --cmd "info-get guestinfo.ovfEnv" > $rndPath 2>&1
    Info "rndPath"
    echo $rndPath
    if [ $? != 0 ]; then
      Exec "${RM} $rndPath" '' ''
#     kaushik
      Die "Error reading OVF environment"
    fi

    ovfEnv=$(<$rndPath)
    Exec "${RM} $rndPath" '' ''
  fi

  local line=
  local out=

  local skipEntity=0

  while read -r line
  do
    if [[ -z "$out" ]]; then
      out=$line
    else
      if [[ $line == *"<Entity"* ]]; then
        skipEntity=1
      fi
      if [[ $skipEntity -eq 0 ]]; then
        out=$out$'\n'$line
      fi
      if [[ $line == *"</Entity"* ]]; then
        skipEntity=0
      fi
    fi
  done <<<"$ovfEnv"

  $ECHO "$out" > $tempPath

  ${GREP} "No value found" $tempPath >/dev/null 2>&1
  if [ $? -eq 0 ]; then
    Exec "${RM} $tempPath" '' ''
    Info "OVF Environment not Set"     
#   Die "OVF environment not set"
  fi

  Debug $'\n'"$out"
}

# This method looks up the parameter value in CUSTOMIZATION_PARAMS
# CUSTOMIZATION_PARAMS is assumed to be fetched from the OVF environment
# and set already.
GetOvfPropertyValue() {
  local key=$1
  local tempPath=$2

  local value=`${AWK} -f $AWK_XML_PARSER $tempPath $key`
  if [ $? -eq 0 ]; then
    echo "$value"
  else
    Warn "Key $key not found"
    echo ""
  fi
}
