/*
 * DNS Resolver Module User-space Helper for AFSDB records
 *
 * Copyright (C) Wang Lei (wang840925@gmail.com) 2010
 * Authors: Wang Lei (wang840925@gmail.com)
 *
 * Copyright (C) David Howells (dhowells@redhat.com) 2018
 *
 * This is a userspace tool for querying AFSDB RR records in the DNS on behalf
 * of the kernel, and converting the VL server addresses to IPv4 format so that
 * they can be used by the kAFS filesystem.
 *
 * As some function like res_init() should use the static liberary, which is a
 * bug of libresolv, that is the reason for cifs.upcall to reimplement.
 *
 * To use this program, you must tell /sbin/request-key how to invoke it.  You
 * need to have the keyutils package installed and something like the following
 * lines added to your /etc/request-key.conf file:
 *
 * 	#OP    TYPE         DESCRIPTION CALLOUT INFO PROGRAM ARG1 ARG2 ARG3 ...
 * 	====== ============ =========== ============ ==========================
 * 	create dns_resolver afsdb:*     *            /sbin/key.dns_resolver %k
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include "key.dns.h"
#include <profile.h>

static const char *afs_cellservdb[] = {
	"/etc/kafs/cellservdb.conf",
	"/usr/share/kafs/cellservdb.conf",
	NULL
};

static profile_t afs_conf;
static bool afs_cell_in_conf;
static bool afs_prefer_dns;
static unsigned long afs_ttl = ULONG_MAX;

/*
 * Check that a configured address is valid and add it to the list of addresses
 * if okay.
 */
static void afs_conf_add_address(char *addr)
{
	char *p, *q, *port = NULL;
	size_t plen = 0;

	if (!addr[0])
		return;

	if (addr[0] == '[') {
		/* IPv6 */
		struct in6_addr in6;

		p = strchr(addr + 1, ']');
		if (!p)
			return;
		*p = 0;
		if (inet_pton(AF_INET6, addr + 1, &in6) == 0)
			return;
		*p++ = ']';
	} else {
		struct in_addr in;

		p = strchr(addr, ':');
		if (p)
			*p = 0;
		if (inet_pton(AF_INET, addr, &in) == 0)
			return;
		if (p)
			*p = ':';
	}

	/* See if there's a port specifier as well */
	if (p && *p) {
		if (*p != ':')
			return;
		p++;
		port = p;
		plen = strlen(port);
		if (plen > 5)
			return;
		strtoul(p, &q, 10);
		if (q != port + plen)
			return;
	}

	append_address_to_payload(addr);
}

/*
 * Parse the cell database file
 */
static void afs_conf_find_cell(const char *cell)
{
	const char *filter[6];
	char **list;
	long res;
	int tmp;

	/* Parse the cell database file */
	res = profile_init(afs_cellservdb, &afs_conf);
	if (res != 0) {
		afs_prefer_dns = true;
		goto error;
	}

	/* Check to see if the named cell is in the list */
	filter[0] = "cells";
	filter[1] = cell;
	filter[2] = NULL;

	res = profile_get_subsection_names(afs_conf, filter, &list);
	if (res != 0) {
		afs_prefer_dns = true;
		goto error;
	}

	if (!list[0]) {
		info("cell not configured\n");
		afs_cell_in_conf = false;
		afs_prefer_dns = true;
	} else {
		afs_cell_in_conf = true;

		/* Retrieve the use_dns value for the cell */
		res = profile_get_boolean(afs_conf, "cells", cell, "use_dns", 1, &tmp);
		if (res != 0) {
			afs_prefer_dns = true;
			goto error;
		}

		if (tmp)
			afs_prefer_dns = true;
		else
			info("cell sets use_dns=no");
	}

	return;

error:
	_error("cellservdb: %s", error_message(res));
}

/*
 * Get list of server names from the config file.
 */
static char **afs_conf_list_servers(const char *cell)
{
	const char *filter[] = {
		"cells",
		cell,
		"servers",
		NULL
	};
	char **servers;
	long res;

	res = profile_get_subsection_names(afs_conf, filter, &servers);
	if (res != 0)
		goto error;

	return servers;

error:
	_error("cellservdb: %s", error_message(res));
	return NULL;
}

/*
 * Get list of addresses for a server from the config file.
 */
static int afs_conf_list_addresses(const char *cell, const char *server)
{
	const char *filter[] = {
		"cells",
		cell,
		"servers",
		server,
		"address",
		NULL
	};
	char **list, **p;
	long res;

	res = profile_get_values(afs_conf, filter, &list);
	if (res != 0)
		goto error;

	for (p = list; *p; p++)
		afs_conf_add_address(*p);
	return 0;

error:
	_error("cellservdb: %s", error_message(res));
	return -1;
}

/*
 *
 */
static void afsdb_hosts_to_addrs(ns_msg handle, ns_sect section)
{
	char *vllist[MAX_VLS];	/* list of name servers	*/
	int vlsnum = 0;		/* number of name servers in list */
	int rrnum;
	ns_rr rr;
	int subtype, i, ret;
	unsigned int ttl = UINT_MAX, rr_ttl;

	debug("AFSDB RR count is %d", ns_msg_count(handle, section));

	/* Look at all the resource records in this section. */
	for (rrnum = 0; rrnum < ns_msg_count(handle, section); rrnum++) {
		/* Expand the resource record number rrnum into rr. */
		if (ns_parserr(&handle, section, rrnum, &rr)) {
			_error("ns_parserr failed : %m");
			continue;
		}

		/* We're only interested in AFSDB records */
		if (ns_rr_type(rr) == ns_t_afsdb) {
			vllist[vlsnum] = malloc(MAXDNAME);
			if (!vllist[vlsnum])
				error("Out of memory");

			subtype = ns_get16(ns_rr_rdata(rr));

			/* Expand the name server's domain name */
			if (ns_name_uncompress(ns_msg_base(handle),
					       ns_msg_end(handle),
					       ns_rr_rdata(rr) + 2,
					       vllist[vlsnum],
					       MAXDNAME) < 0)
				error("ns_name_uncompress failed");

			rr_ttl = ns_rr_ttl(rr);
			if (ttl > rr_ttl)
				ttl = rr_ttl;

			/* Check the domain name we've just unpacked and add it to
			 * the list of VL servers if it is not a duplicate.
			 * If it is a duplicate, just ignore it.
			 */
			for (i = 0; i < vlsnum; i++)
				if (strcasecmp(vllist[i], vllist[vlsnum]) == 0)
					goto next_one;

			/* Turn the hostname into IP addresses */
			ret = dns_resolver(vllist[vlsnum], NULL);
			if (ret) {
				debug("AFSDB RR can't resolve."
				      "subtype:%d, server name:%s, netmask:%u",
				      subtype, vllist[vlsnum], mask);
				goto next_one;
			}

			info("AFSDB RR subtype:%d, server name:%s, ip:%*.*s, ttl:%u",
			     subtype, vllist[vlsnum],
			     (int)payload[payload_index - 1].iov_len,
			     (int)payload[payload_index - 1].iov_len,
			     (char *)payload[payload_index - 1].iov_base,
			     ttl);

			/* prepare for the next record */
			vlsnum++;
			continue;

		next_one:
			free(vllist[vlsnum]);
		}
	}

	afs_ttl = ttl;
	info("ttl: %u", ttl);
}

/*
 *
 */
static void srv_hosts_to_addrs(ns_msg handle, ns_sect section)
{
	char *vllist[MAX_VLS];	/* list of name servers	*/
	int vlsnum = 0;		/* number of name servers in list */
	int rrnum;
	ns_rr rr;
	int subtype, i, ret;
	unsigned short pref, weight, port;
	unsigned int ttl = UINT_MAX, rr_ttl;
	char sport[8];

	debug("SRV RR count is %d", ns_msg_count(handle, section));

	/* Look at all the resource records in this section. */
	for (rrnum = 0; rrnum < ns_msg_count(handle, section); rrnum++) {
		/* Expand the resource record number rrnum into rr. */
		if (ns_parserr(&handle, section, rrnum, &rr)) {
			_error("ns_parserr failed : %m");
			continue;
		}

		if (ns_rr_type(rr) == ns_t_srv) {
			vllist[vlsnum] = malloc(MAXDNAME);
			if (!vllist[vlsnum])
				error("Out of memory");

			subtype = ns_get16(ns_rr_rdata(rr));

			/* Expand the name server's domain name */
			if (ns_name_uncompress(ns_msg_base(handle),
					       ns_msg_end(handle),
					       ns_rr_rdata(rr) + 6,
					       vllist[vlsnum],
					       MAXDNAME) < 0) {
				_error("ns_name_uncompress failed");
				continue;
			}

			rr_ttl = ns_rr_ttl(rr);
			if (ttl > rr_ttl)
				ttl = rr_ttl;

			pref   = ns_get16(ns_rr_rdata(rr));
			weight = ns_get16(ns_rr_rdata(rr) + 2);
			port   = ns_get16(ns_rr_rdata(rr) + 4);
			info("rdata %u %u %u", pref, weight, port);

			sprintf(sport, "+%hu", port);

			/* Check the domain name we've just unpacked and add it to
			 * the list of VL servers if it is not a duplicate.
			 * If it is a duplicate, just ignore it.
			 */
			for (i = 0; i < vlsnum; i++)
				if (strcasecmp(vllist[i], vllist[vlsnum]) == 0)
					goto next_one;

			/* Turn the hostname into IP addresses */
			ret = dns_resolver(vllist[vlsnum], sport);
			if (ret) {
				debug("SRV RR can't resolve."
				      "subtype:%d, server name:%s, netmask:%u",
				      subtype, vllist[vlsnum], mask);
				goto next_one;
			}

			info("SRV RR subtype:%d, server name:%s, ip:%*.*s, ttl:%u",
			     subtype, vllist[vlsnum],
			     (int)payload[payload_index - 1].iov_len,
			     (int)payload[payload_index - 1].iov_len,
			     (char *)payload[payload_index - 1].iov_base,
			     ttl);

			/* prepare for the next record */
			vlsnum++;
			continue;

		next_one:
			free(vllist[vlsnum]);
		}
	}

	afs_ttl = ttl;
	info("ttl: %u", ttl);
}

/*
 * Instantiate the key.
 */
static __attribute__((noreturn))
void afs_instantiate(const char *cell)
{
	int ret;

	/* set the key's expiry time from the minimum TTL encountered */
	if (!debug_mode) {
		ret = keyctl_set_timeout(key, afs_ttl);
		if (ret == -1)
			error("%s: keyctl_set_timeout: %m", __func__);
	}

	/* handle a lack of results */
	if (payload_index == 0)
		nsError(NO_DATA, cell);

	/* must include a NUL char at the end of the payload */
	payload[payload_index].iov_base = "";
	payload[payload_index++].iov_len = 1;
	dump_payload();

	/* load the key with data key */
	if (!debug_mode) {
		ret = keyctl_instantiate_iov(key, payload, payload_index, 0);
		if (ret == -1)
			error("%s: keyctl_instantiate: %m", __func__);
	}

	exit(0);
}

/*
 * Look up an AFSDB record to get the VL server addresses.
 */
static int dns_query_AFSDB(const char *cell)
{
	int	response_len;		/* buffer length */
	ns_msg	handle;			/* handle for response message */
	union {
		HEADER hdr;
		u_char buf[NS_PACKETSZ];
	} response;		/* response buffers */

	debug("Get AFSDB RR for cell name:'%s'", cell);

	/* query the dns for an AFSDB resource record */
	response_len = res_query(cell,
				 ns_c_in,
				 ns_t_afsdb,
				 response.buf,
				 sizeof(response));

	if (response_len < 0) {
		/* negative result */
		_nsError(h_errno, cell);
		return -1;
	}

	if (ns_initparse(response.buf, response_len, &handle) < 0)
		error("ns_initparse: %m");

	/* look up the hostnames we've obtained to get the actual addresses */
	afsdb_hosts_to_addrs(handle, ns_s_an);

	info("DNS query AFSDB RR results:%u ttl:%lu", payload_index, afs_ttl);
	return 0;
}

/*
 * Look up an SRV record to get the VL server addresses [RFC 5864].
 */
static int dns_query_VL_SRV(const char *cell)
{
	int	response_len;		/* buffer length */
	ns_msg	handle;			/* handle for response message */
	union {
		HEADER hdr;
		u_char buf[NS_PACKETSZ];
	} response;
	char name[1024];

	snprintf(name, sizeof(name), "_afs3-vlserver._udp.%s", cell);

	debug("Get VL SRV RR for name:'%s'", name);

	response_len = res_query(name,
				 ns_c_in,
				 ns_t_srv,
				 response.buf,
				 sizeof(response));

	if (response_len < 0) {
		/* negative result */
		_nsError(h_errno, cell);
		return -1;
	}

	if (ns_initparse(response.buf, response_len, &handle) < 0)
		error("ns_initparse: %m");

	/* look up the hostnames we've obtained to get the actual addresses */
	srv_hosts_to_addrs(handle, ns_s_an);

	info("DNS query VL SRV RR results:%u ttl:%lu", payload_index, afs_ttl);
	return 0;
}

/*
 * Look up VL servers for AFS.
 */
void afs_look_up_VL_servers(const char *cell, char *options)
{
	char **servers;

	/* Is the IP address family limited? */
	if (strcmp(options, "ipv4") == 0)
		mask = INET_IP4_ONLY;
	else if (strcmp(options, "ipv6") == 0)
		mask = INET_IP6_ONLY;

	afs_conf_find_cell(cell);

	if (afs_prefer_dns) {
		if (dns_query_VL_SRV(cell) == 0)
			goto instantiate;
		if (dns_query_AFSDB(cell) == 0)
			goto instantiate;
	}

	if (!afs_cell_in_conf)
		goto instantiate; /* Record a negative result */

	servers = afs_conf_list_servers(cell);
	if (!servers) {
		debug("conf: no servers");
		goto instantiate; /* Record a negative result */
	}

	for (; *servers; servers++) {
		char *server = *servers;

		debug("conf server %s", server);
		if (dns_resolver(server, NULL) < 0)
			afs_conf_list_addresses(cell, server);
	}

instantiate:
	afs_instantiate(cell);
}
