#!/usr/bin/python3

'''
functionalities:

update ca certificate for a airgap repo to existing clusters
input: 1. airgap repo fqdn
       2. ca certificate file
       3. vc credential

output: dump updated resources results
'''

import argparse
import logging
import logging.handlers
import os
import json
import subprocess
import sys
import re
import requests
import base64
import configparser
import yaml
import traceback
import pathlib

logger = logging.getLogger(__name__)
TMP_DIR = "/tmp/update-ca"
CWD = os.path.dirname(os.path.realpath(__file__))
ansible_log_file = None

def logger_setup(log_dst="console", log_level='info', log_file='/common/logs/update-cert.log'):
    log_formatter = logging.Formatter('%(module)s[%(levelname)s]: %(message)s')
    handler = None
    global ansible_log_file

    log_levels = {
        "debug": logging.DEBUG,
        "info": logging.INFO,
        "warning": logging.WARNING,
        "error": logging.ERROR,
        "critical": logging.CRITICAL,
    }

    if log_dst == "file":
        handler = logging.FileHandler(log_file)
        ansible_log_file = "/common/logs/update-cert-ansible.log"
    elif log_dst == "syslog":
        handler = logging.handlers.SysLogHandler("/dev/log")
    else:
        handler = logging.StreamHandler()

    handler.setFormatter(log_formatter)
    logger.addHandler(handler)

    if log_level:
        if log_level == "off":
            logging.disable(logging.CRITICAL)
        else:
            logger.setLevel(log_levels[log_level])

def str_presenter(dumper, data):
    if len(data.splitlines()) > 1:  # check for multiline string
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)


def init_yaml():
    yaml.add_representer(str, str_presenter)
    # to use with safe_dump:
    yaml.representer.SafeRepresenter.add_representer(str, str_presenter)

class Cert:
    GET_SUBJECT_CMD = "openssl x509 -noout -subject -in %s"

    def __init__(self, cert_content):
        self.content = cert_content.strip()
        self.subject = None

    def get_dn(self):
        if self.subject:
            return self.subject
        cert_file_name = "%s/tmp_cert" % TMP_DIR
        with open(cert_file_name, 'w') as f:
            f.write(self.content)
        cmd = self.GET_SUBJECT_CMD % cert_file_name
        p = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE, close_fds=True)
        res, stderr = p.communicate()
        rc = p.returncode
        if rc:
            error = stderr.decode('utf-8')
            logger.warning("failed to extract certificate subject, cert content"
                           ": %s" % self.content)
            logger.warning("assignning cert base64 as dn of this certificate")
            ca_base64 = base64.b64encode(self.content.encode('utf-8')).decode('utf-8')
            self.subject = ca_base64
        else:
            self.subject = res.decode('utf-8')
        return self.subject

    def get_content(self):
        return self.content

class Certs:
    START_LINE = '-----BEGIN CERTIFICATE-----'
    def __init__(self, certs_content):
        self.certs = {}
        self.certs_size = 0
        logger.debug("cert content: %s" % certs_content)
        cert_slots = certs_content.split(self.START_LINE)
        for pem_cert_content in cert_slots[1:]:
            cert = Cert(self.START_LINE + pem_cert_content)
            self.certs[cert.get_dn()] = cert
            self.certs_size += 1
        self.merged = 0

    def merge(self, new_cert_content):
        new_cert = Cert(new_cert_content)
        dn = new_cert.get_dn()
        if ((dn not in self.certs) or (dn in self.certs and
            self.certs[dn].get_content() != new_cert_content)):
            self.certs[dn] = new_cert
            self.merged = 1

    def is_merged(self):
        return self.merged

    def dump(self):
        certs_content = ""
        first = True
        for key, cert in self.certs.items():
            if not first:
                certs_content += '\n'
            else:
                first = False
            certs_content += cert.get_content()
        return certs_content

    def size(self):
        return self.certs_size

class AirgapRepo:
    verified = False

    def __new__(cls, fqdn, cafile):
        if not hasattr(cls, 'instance'):
            cls.instance = super(AirgapRepo, cls).__new__(cls)
        return cls.instance

    def __init__(self, fqdn, cafile):
        self.fqdn = fqdn
        self.cafile = cafile
        self.ca_plain_text = None
        self.ca_base64 = None
        return self.load_ca()

    def load_ca(self):
        with open(self.cafile, 'r', encoding='ascii') as file:
            self.ca_plain_text = file.read().strip()
            ca_bytes = self.ca_plain_text.encode('ascii')
            self.ca_base64 = base64.b64encode(ca_bytes).decode('ascii')
        return
    def get_base64_ca(self):
        return self.ca_base64
    def get_plain_ca(self):
        return self.ca_plain_text
    def get_fqdn(self):
        return self.fqdn
    def verify(self):
        # make sure only one certificate in the cafile, Root CA certificate or self-signed certificate.
        certs = Certs(self.ca_plain_text)
        if certs.size() == 0:
            return False, "the specified cafile doesn't contain a validate CA certificate"
        elif certs.size() > 1:
            return False, "the specified cafile contains more than one certificate, it is expected ONLY ONE Certficate, the root CA certficate in the certficate chain or the self-signed certificate"
        repo_url = "https://%s" % self.fqdn
        response = requests.get(repo_url, verify=self.cafile)
        if response.ok:
            self.verified = True
            return True, ""
        else:
            return False, response.reason
    def has_verified(self):
        return self.verified
    def is_valid(self):
        try:
            return self.verify()
        except Exception as e:
            return False, repr(e)
    def dump(self):
        logger.info("dumpping the repo: %s" % self.fqdn)
        logger.info("repo cafile: %s" % self.cafile)
        logger.info("repo ca plain text: %s" % self.ca_plain_text)
        logger.info("repo ca base64: %s" % self.ca_base64)
        logger.info("repo is_valid: %s, %s" % self.is_valid())

class Kbs:
    TKGCTXS_URL = "http://localhost:8888/api/v1/tkgcontexts"
    TKGCTX_URL = "http://localhost:8888/api/v1/tkgcontext"
    MGMTCLSTS_URL = "http://localhost:8888/api/v1/managementclusters"
    MGMTCLST_URL = "http://localhost:8888/api/v1/managementcluster"
    WRKLCLSTS_URL = "http://localhost:8888/api/v1/workloadclusters"
    WRKLCLST_URL = "http://localhost:8888/api/v1/workloadcluster"

    def __init__(self, repo):
        self.repo = repo
        self.tkgcontexts = None
        self.mgmtclusters = None
        self.workloadclusters = None
        self.wrklclusters = None
        return

    def __new__(cls, repo):
        if not hasattr(cls, 'instance'):
            cls.instance = super(Kbs, cls).__new__(cls)
        return cls.instance

    def load_contexts(self):
        response = requests.get(self.TKGCTXS_URL)
        self.tkgcontexts = []
        # get the vc password
        for tkgctx in response.json():
            resp = requests.get("%s/%s?plaintext=true" % (self.TKGCTX_URL, tkgctx['id']))
            self.tkgcontexts.append(resp.json())
        #self.dump_contexts()
        return

    def update_contexts(self):
        if self.tkgcontexts is None:
            self.load_contexts()
        for tkgctx in self.tkgcontexts:
            if 'airgap' in tkgctx and tkgctx['airgap']['fqdn'] == self.repo.get_fqdn():
                tkgctx['airgap']['caCert'] = self.repo.get_base64_ca()
                # put
                url = "%s/%s" % (self.TKGCTX_URL, tkgctx['id'])
                response = requests.put(url, json=tkgctx)
                response.encoding = 'utf-8'
                if response.status_code == 200:
                    logger.info("updated tkgcontext %s with response %s" % (tkgctx['id'], response))
                else:
                    logger.error("failed to update tkgcontext %s, err: %s" % (tkgctx['id'], response))
        return

    def verify_contexts(self):
        if self.tkgcontexts is None:
            self.load_contexts()
        for tkgctx in self.tkgcontexts:
            if 'airgap' in tkgctx and tkgctx['airgap']['fqdn'] == self.repo.get_fqdn():
                if 'caCert' in tkgctx['airgap'] and tkgctx['airgap']['caCert'] == self.repo.get_base64_ca():
                    logger.info("tkgcontext[%s]: up to date" % tkgctx['id'])
                else:
                    logger.error("tkgcontext[%s]: out of date" % tkgctx['id'])
            else:
                logger.debug("skip tkgcontext[%s]: non target airgap repo"
                            % tkgctx['id'])

    def dump_contexts(self):
        logger.info("dumping tkg contexts: %s" % self.tkgcontexts)
        return
    def get_tkgctx_by_id(self, id):
        if self.tkgcontexts is None:
            self.load_contexts()
        for ctx in self.tkgcontexts:
            if ctx['id'] == id:
               return ctx
        return None
    def get_mgmtclusters_of_repo(self):
        mgmtclusters = []
        clusters = self.get_mgmtclusters()
        for mgmt_cluster in clusters:
            tkgcontext_id = mgmt_cluster["tkgID"]
            tkgctx = self.get_tkgctx_by_id(tkgcontext_id)
            if tkgctx is None or 'airgap' not in tkgctx or tkgctx['airgap']['fqdn'] != self.repo.get_fqdn():
                logger.debug("cluster %s not associated target airgap repo" % mgmt_cluster['clusterName'])
                continue
            else:
                mgmtclusters.append(mgmt_cluster)
        return mgmtclusters
    def get_mgmtclusters(self):
        if self.mgmtclusters is None:
            try:
                response = requests.get(self.MGMTCLSTS_URL)
                self.mgmtclusters = response.json()
            except Exception as e:
                logger.critical("failed to get management clusters, err: %s" % repr(e))
                traceback.print_exc()
                return []
        return self.mgmtclusters

    def get_mgmtcluster_node_ips(self, mgmt_cluster):
        ips = []

        #get status of mgmt cluster
        url = "%s/%s/status" % (self.MGMTCLST_URL, mgmt_cluster['id'])
        logger.debug("query cluster %s node ips" % mgmt_cluster['clusterName'])
        try:
            response = requests.get(url)
            status = response.json()
        except Exception as e:
            logger.critical("failed to get management cluster[%s] node ips, err: %s"
                            % (mgmt_cluster['clusterName'], repr(e)))
            traceback.print_exc()
            return ips

        if 'nodes' not in status:
            logger.info("nodes is not in cluster %s status" % mgmt_cluster['clusterName'])
            return ips
        for node in status['nodes']:
            if 'ip' not in node:
                logger.warn("node of %s doesn't have ip assigned" % node['vmName'])
                continue
            else:
                logger.debug("appending ip %s" % node['ip'])
                ips.append(node['ip'])
        return ips
    def get_wrklclusters_of_repo(self):
        wrklclusters = []
        if self.wrklclusters is None:
            try:
                response = requests.get(self.WRKLCLSTS_URL)
                self.wrklclusters = response.json()
            except Exception as e:
                logger.critical("failed to get workload clusters, err: %s" % repr(e))
                traceback.print_exc()
        for wrkl_cluster in self.wrklclusters:
            tkgcontext_id = wrkl_cluster["tkgID"]
            tkgctx = self.get_tkgctx_by_id(tkgcontext_id)
            if tkgctx is None or 'airgap' not in tkgctx or tkgctx['airgap']['fqdn'] != self.repo.get_fqdn():
                logger.debug("cluster %s not associated target airgap repo" % wrkl_cluster['clusterName'])
                continue
            else:
                wrklclusters.append(wrkl_cluster)
        return wrklclusters

    def get_wrklcluster_node_ips(self, wrkl_cluster):
        ips = []

        #get status of workload cluster
        url = "%s/%s/status" % (self.WRKLCLST_URL, wrkl_cluster['id'])
        logger.info("query cluster %s node ips" % wrkl_cluster['clusterName'])
        try:
            response = requests.get(url)
            status = response.json()
        except Exception as e:
            logger.critical("failed to get workload cluster[%s] node ips, err: %s"
                            % (wrkl_cluster['clusterName'], repr(e)))
            traceback.print_exc()
            return ips

        if 'nodes' not in status:
            logger.info("nodes is not in cluster %s status" % wrkl_cluster['clusterName'])
            return ips
        for node in status['nodes']:
            if 'ip' not in node:
                logger.warn("node of %s doesn't have ip assigned" % node['vmName'])
                continue
            else:
                logger.debug("appending ip %s" % node['ip'])
                ips.append(node['ip'])
        return ips

class ClusterClient:
    RUN_SH = "%s/cmd.sh" % TMP_DIR
    def __init__(self, kubeconfig_file):
        self.kubeconfig_file = kubeconfig_file

    def run_cmd(self, cmd):
        kubectl_cmd = "kubectl --kubeconfig %s --request-timeout 30s %s" % (self.kubeconfig_file, cmd)
        with open(self.RUN_SH, 'w') as f:
            f.write(kubectl_cmd)
            f.close()

        logger.debug("running command: %s" % kubectl_cmd)
        cmd = 'bash ' + self.RUN_SH
        p = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE, close_fds=True)
        res, stderr = p.communicate()
        rc = p.returncode
        if rc:
            error = stderr.decode('utf-8')
            if "NotFound" in error:
                raise NameError("run \"%s\" failed, err: %s" % (kubectl_cmd, error))
            raise RuntimeError("run \"%s\" failed, err: %s" % (kubectl_cmd, error))
        logger.debug(res.decode('utf-8'))
        return res

    def get_json(self, cmd):
        get_cmd = "get %s -o json" % cmd
        return self.run_cmd(get_cmd)

    def get_yaml(self, cmd):
        get_cmd = "get %s -o yaml" % cmd
        return self.run_cmd(get_cmd)

    def apply_yaml(self, file):
        cmd = "apply -f %s" % (file)
        return self.run_cmd(cmd)

    def get_dict(self, cmd):
        return json.loads(self.get_json(cmd))

    def get_secret_datavalues(self, ns, name):
        get_cmd = "get secret %s -n %s -o jsonpath={.data.\"values\\.yaml\"} | base64 -d" % (name, ns)
        return yaml.safe_load(self.run_cmd(get_cmd))


class TkgClusterClient(ClusterClient):
    PATCH_KAPP_CM_CMD = "patch cm -n tkg-system kapp-controller-config -p"
    GET_KAPP_CM_CMD = "cm -n tkg-system kapp-controller-config"
    RESTART_KAPP_DEPLOYMENT_CMD = "rollout restart deployment -n tkg-system kapp-controller"
    def __init__(self, kubeconfig_file, repo):
        self.kubeconfig_file = kubeconfig_file
        self.repo = repo

    def merge_certs(self, old):
        #certstr = self.repo.get_plain_ca().replace('\n', '\\n')
        certstr = self.repo.get_plain_ca()
        if old is None:
             return certstr
        certs = Certs(old)
        certs.merge(certstr)
        if certs.is_merged():
            return certs.dump()
        return None

    def generate_cm_cacerts_patch_json(self, orig):
        patch_dict = {}
        patch_dict["data"] = {}

        newcerts = self.merge_certs(orig)
        if newcerts is None:
            return None

        patch_dict["data"]["caCerts"] = newcerts
        patch_cm_json = json.dumps(patch_dict, separators=(',',':')).replace('"', '\\"')
        return patch_cm_json

    def get_kapp_controller_cm(self):
        try:
            kapp_cm = self.get_dict(self.GET_KAPP_CM_CMD)
        except Exception as e:
            logger.critical("failed to get configmap kapp-controller-config/tkg-system, err: %s" % repr(e))
            traceback.print_exc()
            return None
        return kapp_cm

    def update_kapp_controller_cm(self):
        kapp_cm = self.get_kapp_controller_cm()
        if kapp_cm is None:
            return
        patch_cm_json = None
        if 'data' not in kapp_cm or 'caCerts' not in kapp_cm['data']:
            patch_cm_json = self.generate_cm_cacerts_patch_json(None)
        else:
            patch_cm_json = self.generate_cm_cacerts_patch_json(kapp_cm['data']['caCerts'])

        if patch_cm_json is None:
            logger.info("cluster kapp-controller-config is up to date, skip")
            return

        cmd = "%s \"%s\"" % (self.PATCH_KAPP_CM_CMD, patch_cm_json)
        self.run_cmd(cmd)
        self.run_cmd(self.RESTART_KAPP_DEPLOYMENT_CMD)
        logger.info("update cluster kapp-controller-config successfully")

    def verify_kapp_controller_cm(self):
        kapp_cm = self.get_kapp_controller_cm()
        if kapp_cm is None:
            return
        if 'data' in kapp_cm and 'caCerts' in kapp_cm['data']:
            cacerts = kapp_cm['data']['caCerts']
            new_cacerts = self.merge_certs(cacerts)
            if new_cacerts is None:
                logger.info("configmap kapp-controller-config/tkg-system: up to date")
                return
        logger.error("configmap kapp-controller-config/tkg-system: out of date")

class MgmtClusterClient(TkgClusterClient):
    PATCH_TKR_CM_CMD_TKR_SYSTEM = "patch cm -n tkr-system tkr-controller-config -p"
    GET_TKR_CM_CMD_TKR_SYSTEM = "cm -n tkr-system tkr-controller-config"
    RESTART_TKR_DEPLOYMENT_CMD_TKR_SYSTEM = "rollout restart deployment -n tkr-system tkr-controller-manager"
    PATCH_TKR_CM_CMD_TKG_SYSTEM = "patch cm -n tkg-system tkr-controller-config -p"
    GET_TKR_CM_CMD_TKG_SYSTEM = "cm -n tkg-system tkr-controller-config"
    RESTART_TKR_SOURCE_DEPLOYMENT_CMD_TKG_SYSTEM = "rollout restart deployment -n tkg-system tkr-source-controller-manager"
    RESTART_TKR_VSPHERE_DEPLOYMENT_CMD_TKG_SYSTEM = "rollout restart deployment -n tkg-system tkr-vsphere-resolver-webhook-manager"
    GET_ALL_KCP_CMD = "kcp -A"
    GET_ALL_KCT_CMD = "kubeadmconfigtemplate -A"
    TKG_ADDON_SECRET_HEADER = "#@data/values\n#@overlay/match-child-defaults missing_ok=True\n---\n"

    def __init__(self, kubeconfig_file, repo):
        super().__init__(kubeconfig_file, repo)
        self.kcps = None
        self.kcts = None

    def IsClusterClass(self, clusterName):
        GET_CLUSTER_CMD = "cluster -n tkg-system %s" % clusterName
        clusterCR = self.get_dict(GET_CLUSTER_CMD)
        if "topology" in clusterCR["spec"]:
            return True
        return False

    def update_tkg_pkg_secret(self):
        secretName = "tkg-pkg-tkg-system-values"
        ns = "tkg-system"
        try:
            _ = self.get_dict("secret -n %s %s" % (ns, secretName))
        except NameError:
            return
        get_cmd = "get secret %s -n %s -o jsonpath={.data.\"tkgpackagevalues\\.yaml\"} | base64 -d" % (secretName, ns)
        try:
            values_dict = yaml.safe_load(self.run_cmd(get_cmd))
        except Exception as e:
            logger.critical("failed to get tkgpackagevalues.yaml from secret [%s] in namespace [%s], err: %s"
                   % (secretName, ns, repr(e)))
            traceback.print_exc()
            raise e
        need_patch = False
        if 'configvalues' in values_dict:
            if 'TKG_CUSTOM_IMAGE_REPOSITORY_CA_CERTIFICATE' in values_dict['configvalues']:
                old = values_dict['configvalues']['TKG_CUSTOM_IMAGE_REPOSITORY_CA_CERTIFICATE']
                if old != self.repo.get_base64_ca():
                    values_dict['configvalues']['TKG_CUSTOM_IMAGE_REPOSITORY_CA_CERTIFICATE'] = self.repo.get_base64_ca()
                    need_patch = True
            if 'CUSTOM_TDNF_REPOSITORY_CERTIFICATE' in values_dict['configvalues']:
                old = values_dict['configvalues']['CUSTOM_TDNF_REPOSITORY_CERTIFICATE']
                if old != self.repo.get_base64_ca():
                    values_dict['configvalues']['CUSTOM_TDNF_REPOSITORY_CERTIFICATE'] = self.repo.get_base64_ca()
                    need_patch = True
        if ('tkrSourceControllerPackage' in values_dict
                and 'tkrSourceControllerPackageValues' in values_dict['tkrSourceControllerPackage']
                and 'caCerts' in values_dict['tkrSourceControllerPackage']['tkrSourceControllerPackageValues']):
            old = values_dict['tkrSourceControllerPackage']['tkrSourceControllerPackageValues']['caCerts']
            if old != self.repo.get_base64_ca():
                values_dict['tkrSourceControllerPackage']['tkrSourceControllerPackageValues']['caCerts'] = self.repo.get_base64_ca()
                need_patch = True
        if not need_patch:
            logger.info("secret %s/%s is up to date, skip" % (secretName, ns))
            return
        values = yaml.dump(values_dict)
        logger.debug("tkg-pkg secret new values: %s", values)
        base64_values = base64.b64encode(values.encode('utf-8')).decode('utf-8')
        patch_secret_dict = {}
        patch_secret_dict['data'] = {}
        patch_secret_dict['data']['tkgpackagevalues.yaml'] = base64_values
        patch_secret_json = json.dumps(patch_secret_dict,
                                    separators=(',',':')).replace('"', '\\"')
        patch_cmd = "patch secret %s -n %s --type=merge -p \"%s\"" % (
                    secretName, ns, patch_secret_json)
        self.run_cmd(patch_cmd)
        logger.info("update secret [%s] in namespace [%s] successfully"
                    % (secretName, ns))

    def verify_tkg_pkg_secret(self):
        secretName = "tkg-pkg-tkg-system-values"
        ns = "tkg-system"
        try:
            _ = self.get_dict("secret -n %s %s" % (ns, secretName))
        except NameError:
            return
        get_cmd = "get secret %s -n %s -o jsonpath={.data.\"tkgpackagevalues\\.yaml\"} | base64 -d" % (secretName, ns)
        try:
            values_dict = yaml.safe_load(self.run_cmd(get_cmd))
        except Exception as e:
            logger.critical("failed to get tkgpackagevalues.yaml from secret [%s] in namespace [%s], err: %s"
                   % (secretName, ns, repr(e)))
            traceback.print_exc()
            raise e
        if 'configvalues' in values_dict:
            if 'TKG_CUSTOM_IMAGE_REPOSITORY_CA_CERTIFICATE' in values_dict['configvalues']:
                cert = values_dict['configvalues']['TKG_CUSTOM_IMAGE_REPOSITORY_CA_CERTIFICATE']
                if cert != self.repo.get_base64_ca():
                    logger.error("secret %s/%s: out of date" % (secretName, ns))
                    return
            if 'CUSTOM_TDNF_REPOSITORY_CERTIFICATE' in values_dict['configvalues']:
                cert = values_dict['configvalues']['CUSTOM_TDNF_REPOSITORY_CERTIFICATE']
                if cert != self.repo.get_base64_ca():
                    logger.error("secret %s/%s: out of date" % (secretName, ns))
                    return
        if ('tkrSourceControllerPackage' in values_dict
                and 'tkrSourceControllerPackageValues' in values_dict['tkrSourceControllerPackage']
                and 'caCerts' in values_dict['tkrSourceControllerPackage']['tkrSourceControllerPackageValues']):
            cert = values_dict['tkrSourceControllerPackage']['tkrSourceControllerPackageValues']['caCerts']
            if cert != self.repo.get_base64_ca():
                logger.error("secret %s/%s: out of date" % (secretName, ns))
                return
        logger.info("secret %s/%s: up to date" % (secretName, ns))

    def update_tkr_controller_secrets(self):
        secretList = ["tkr-source-controller-values", "tkr-vsphere-resolver-values"]
        ns = "tkg-system"
        for secret_name in secretList:
            try:
                _ = self.get_dict("secret -n %s %s" % (ns, secret_name))
            except NameError:
                logger.info("secret %s/%s not found, skip" % (secret_name, ns))
                continue
            try:
                values_dict = self.get_secret_datavalues(ns, secret_name)
            except Exception as e:
                logger.critical("failed to get secret [%s] in namespace [%s], err: %s"
                                % (secret_name, ns, repr(e)))
                traceback.print_exc()
                raise e
            if 'caCerts' in values_dict:
                if values_dict['caCerts'] == self.repo.get_base64_ca():
                    logger.info("secret %s/%s is up to date, skip" % (secret_name, ns))
                    continue
                values_dict['caCerts'] = self.repo.get_base64_ca()
            values = yaml.dump(values_dict)
            logger.debug("tkr secret new values: %s", values)
            base64_values = base64.b64encode(values.encode('utf-8')).decode('utf-8')
            patch_secret_dict = {}
            patch_secret_dict['data'] = {}
            patch_secret_dict['data']['values.yaml'] = base64_values
            patch_secret_json = json.dumps(patch_secret_dict,
                                        separators=(',',':')).replace('"', '\\"')
            patch_cmd = "patch secret %s -n %s --type=merge -p \"%s\"" % (
                        secret_name, ns, patch_secret_json)
            self.run_cmd(patch_cmd)
            logger.info("update secret [%s] in namespace [%s] successfully"
                        % (secret_name, ns))

    def verify_tkr_controller_secrets(self):
        secretList = ["tkr-source-controller-values", "tkr-vsphere-resolver-values"]
        ns = "tkg-system"
        for secret_name in secretList:
            try:
                _ = self.get_dict("secret -n %s %s" % (ns, secret_name))
            except NameError:
                continue
            try:
                values_dict = self.get_secret_datavalues(ns, secret_name)
            except Exception as e:
                logger.critical("failed to get secret [%s] in namespace [%s], err: %s"
                                % (secret_name, ns, repr(e)))
                traceback.print_exc()
                return
            if 'caCerts' not in values_dict:
                logger.error("secret [%s] in namespace [%s]: out of date" % (
                            secret_name, ns))
                return
            cacerts = values_dict['caCerts']
            if cacerts == self.repo.get_base64_ca():
                logger.info("secret [%s] in namespace [%s]: up to date" % (
                            secret_name, ns))
            else:
                logger.error("secret [%s] in namespace [%s]: out of date" % (
                            secret_name, ns))

    def update_tkr_controller_cm(self):
        patch_dict = {}
        patch_dict["data"] = {}
        newcerts = self.repo.get_plain_ca()
        patch_dict["data"]["caCerts"] = newcerts
        patch_cm_json = json.dumps(patch_dict, separators=(',',':')).replace('"', '\\"')
        try:
            _ = self.get_dict(self.GET_TKR_CM_CMD_TKG_SYSTEM)
            cmd = "%s \"%s\"" % (self.PATCH_TKR_CM_CMD_TKG_SYSTEM, patch_cm_json)
            self.run_cmd(cmd)
            self.run_cmd(self.RESTART_TKR_SOURCE_DEPLOYMENT_CMD_TKG_SYSTEM)
            self.run_cmd(self.RESTART_TKR_VSPHERE_DEPLOYMENT_CMD_TKG_SYSTEM)
        except NameError:
            cmd = "%s \"%s\"" % (self.PATCH_TKR_CM_CMD_TKR_SYSTEM, patch_cm_json)
            self.run_cmd(cmd)
            self.run_cmd(self.RESTART_TKR_DEPLOYMENT_CMD_TKR_SYSTEM)
        logger.info("update management cluster tkr-controller-config successfully")

    def verify_tkr_controller_cm(self):
        try:
            tkr_cm = self.get_dict(self.GET_TKR_CM_CMD_TKG_SYSTEM)
        except NameError:
            tkr_cm = self.get_dict(self.GET_TKR_CM_CMD_TKR_SYSTEM)
        except Exception as e:
            logger.critical("failed to get configmap tkr-controller-config, err: %s" % repr(e))
            traceback.print_exc()
            return
        expected_ca = self.repo.get_plain_ca()
        if ('data' in tkr_cm and 'caCerts' in tkr_cm['data']
            and tkr_cm['data']['caCerts'].strip() == expected_ca):
            logger.info("configmap tkr-controller-config: up to date")
        else:
            logger.error("configmap tkr-controller-config: out of date")

    def get_kcp(self, ns):
        if self.kcps == None:
            self.kcps = self.get_dict(self.GET_ALL_KCP_CMD)['items']
        if ns is None:
            return self.kcps
        for kcp in self.kcps:
            if kcp['metadata']['namespace'] == ns:
                return kcp
        return None

    def update_kcp(self, ns):
        kcp = self.get_kcp(ns)
        if 'files' in kcp['spec']['kubeadmConfigSpec']:
            files = kcp['spec']['kubeadmConfigSpec']['files']
        else:
            files = []
        name = kcp['metadata']['name']
        ns = kcp['metadata']['namespace']
        fqdn = self.repo.get_fqdn()
        path = "/etc/containerd/%s.crt" % fqdn
        patch_json = None
        found = False
        for file in files:
            if file['path'] == path and file['encoding'] == 'base64':
                found = True
                if file['content'] != self.repo.get_base64_ca():
                    file['content'] = self.repo.get_base64_ca()
                else:
                    logger.info("kubecontrolplane cr[%s/%s]: up to date" % (name, ns))
                    return
                break
        if not found:
            f = {}
            f['path'] = path
            f['encoding'] = 'base64'
            f['content'] = self.repo.get_base64_ca()
            files.append(f)

        patch_dict = {}
        patch_dict['spec'] = {}
        patch_dict['spec']['kubeadmConfigSpec'] = {}
        patch_dict['spec']['kubeadmConfigSpec']['files'] = files
        patch_json = json.dumps(patch_dict, separators=(',',':'))
        patch_cmd = "patch kcp -n %s %s --type=merge -p '%s'" % (ns, name,
                    patch_json)
        self.run_cmd(patch_cmd)
        logger.info("update kcp %s airgap cacert file content successfully" % name)

    def verify_kcp(self, ns):
        try:
            kcp = self.get_kcp(ns)
        except Exception as e:
            logger.critical("failed to get kubecontrolplane in namespace[%s], err: %s" % (ns, repr(e)))
            traceback.print_exc()
            return
        name = kcp['metadata']['name']
        fqdn = self.repo.get_fqdn()
        if ('kubeadmConfigSpec' in kcp['spec'] and
            'files' in kcp['spec']['kubeadmConfigSpec']):
            files = kcp['spec']['kubeadmConfigSpec']['files']
            path = "/etc/containerd/%s.crt" % fqdn
            for file in files:
                if file['path'] == path:
                    if (file['encoding'] == 'base64' and
                        file['content'] == self.repo.get_base64_ca()):
                        logger.info("kubecontrolplane cr[%s/%s]: up to date" % (name, ns))
                        return
        logger.error("kubecontrolplane cr[%s/%s]: out of date" % (name, ns))

    def get_kcts(self, ns): #kubeadmconfigtemplate
        if self.kcts == None:
            self.kcts = self.get_dict(self.GET_ALL_KCT_CMD)['items']
        kcts = []
        for kct in self.kcts:
            if kct['metadata']['namespace'] == ns:
                kcts.append(kct)
        return kcts

    def update_kcts(self, ns):
        kcts = self.get_kcts(ns)
        fqdn = self.repo.get_fqdn()
        path = "/etc/containerd/%s.crt" % fqdn

        for kct in kcts:
            if 'files' in kct['spec']['template']['spec']:
                files = kct['spec']['template']['spec']['files']
            else:
                files = []
            name = kct['metadata']['name']
            if name == 'tkg-vsphere-default-v1.0.0-md-config':
                continue
            ns = kct['metadata']['namespace']
            patch_json = None
            found = False
            need_patch = False
            for file in files:
                if file['path'] == path and file['encoding'] == 'base64':
                    found = True
                    if file['content'] != self.repo.get_base64_ca():
                        file['content'] = self.repo.get_base64_ca()
                        need_patch = True
                    else:
                        logger.info("kubeadmconfigtemplate cr[%s/%s]: up to date" % (name, ns))
                    break
            if not found:
                need_patch = True
                f = {}
                f['path'] = path
                f['encoding'] = 'base64'
                f['content'] = self.repo.get_base64_ca()
                files.append(f)
            if need_patch:
                patch_dict = {}
                patch_dict['spec'] = {}
                patch_dict['spec']['template'] = {}
                patch_dict['spec']['template']['spec'] = {}
                patch_dict['spec']['template']['spec']['files'] = files
                patch_json = json.dumps(patch_dict, separators=(',',':'))
                patch_cmd = "patch kubeadmconfigtemplate -n %s %s --type=merge -p '%s'" % (
                            ns, name, patch_json)
                self.run_cmd(patch_cmd)
                logger.info("update kubeadmconfigtemplate %s airgap cacert file content successfully" % name)

    def verify_kcts(self, ns):
        try:
            kcts = self.get_kcts(ns)
        except Exception as e:
            logger.critical("failed to get kubeadmconfigtemplates in namespace[%s], err: %s" % (ns, repr(e)))
            traceback.print_exc()
            return

        fqdn = self.repo.get_fqdn()
        path = "/etc/containerd/%s.crt" % fqdn

        for kct in kcts:
            name = kct['metadata']['name']
            if name == 'tkg-vsphere-default-v1.0.0-md-config':
                continue
            ns = kct['metadata']['namespace']
            found = False
            if 'files' in kct['spec']['template']['spec']:
                files = kct['spec']['template']['spec']['files']
                for file in files:
                    if (file['path'] == path and file['encoding'] == 'base64' and
                        file['content'] == self.repo.get_base64_ca()):
                        logger.info("kubeadmconfigtemplate cr[%s/%s]: up to date"
                                    % (name, ns))
                        found = True
                        break

            if not found:
                logger.error("kubeadmconfigtemplate cr[%s/%s]: out of date"
                              % (name, ns))

    def update_kapp_secret(self, ns):
        secret_name = "%s-kapp-controller-addon" % ns
        try:
            values_dict = self.get_secret_datavalues(ns, secret_name)
        except Exception as e:
            logger.critical("failed to get secret [%s] in namespace [%s], err: %s"
                            % (secret_name, ns, repr(e)))
            traceback.print_exc()
            raise e
        if 'caCerts' not in values_dict['kappController']['config']:
            new_cacerts = self.merge_certs(None)
        else:
            cacerts = values_dict['kappController']['config']['caCerts']
            new_cacerts = self.merge_certs(cacerts)
        if new_cacerts is None:
            logger.info("caCerts in secret [%s] of namespace [%s] is up to date"
                        ", skip" % (secret_name, ns))
            return
        values_dict['kappController']['config']['caCerts'] = new_cacerts
        values = yaml.dump(values_dict)
        values = self.TKG_ADDON_SECRET_HEADER + values
        logger.debug("kapp secret new values: %s", values)
        base64_values = base64.b64encode(values.encode('utf-8')).decode('utf-8')
        patch_secret_dict = {}
        patch_secret_dict['data'] = {}
        patch_secret_dict['data']['values.yaml'] = base64_values
        patch_secret_json = json.dumps(patch_secret_dict,
                                       separators=(',',':')).replace('"', '\\"')
        patch_cmd = "patch secret %s -n %s --type=merge -p \"%s\"" % (
                    secret_name, ns, patch_secret_json)
        self.run_cmd(patch_cmd)
        logger.info("update secret [%s] in namespace [%s] successfully"
                    % (secret_name, ns))

    def verify_kapp_secret(self, ns):
        secret_name = "%s-kapp-controller-addon" % ns
        try:
            values_dict = self.get_secret_datavalues(ns, secret_name)
        except Exception as e:
            logger.critical("failed to get secret [%s] in namespace [%s], err: %s"
                            % (secret_name, ns, repr(e)))
            traceback.print_exc()
            return
        if 'caCerts' not in values_dict['kappController']['config']:
            logger.error("secret [%s] in namespace [%s]: out of date" % (
                         secret_name, ns))
            return
        cacerts = values_dict['kappController']['config']['caCerts']
        new_cacerts = self.merge_certs(cacerts)
        if new_cacerts is None:
            logger.info("secret [%s] in namespace [%s]: up to date" % (
                        secret_name, ns))
        else:
            logger.error("secret [%s] in namespace [%s]: out of date" % (
                         secret_name, ns))
            
    def cc_update(self, clusterName):
        self.update_kapp_controller_cm()
        self.update_tkg_pkg_secret()
        self.update_tkr_controller_secrets()
        self.update_tkr_controller_cm()
        GET_CLUSTER_CMD = "cluster -n tkg-system %s" % clusterName
        clusterCR = self.get_dict(GET_CLUSTER_CMD)
        need_patch = False
        variables = clusterCR['spec']['topology']['variables']
        for var in variables:
            if var['name'] == 'trust' and 'additionalTrustedCAs' in var['value']:
                for ca in var['value']['additionalTrustedCAs']:
                    if ca['name'] == 'imageRepository' and ca['data'] != self.repo.get_base64_ca():
                        ca['data'] = self.repo.get_base64_ca()
                        need_patch = True
            elif var['name'] == 'customTDNFRepository' and 'certificate' in var['value']:
                if var['value']['certificate'] != self.repo.get_base64_ca():
                    var['value']['certificate'] = self.repo.get_base64_ca()
                    need_patch = True
        if need_patch:
            patch_dict = {'spec': {'topology': {'variables': variables}}}
            patch_json = json.dumps(patch_dict, separators=(',',':')).replace('"', '\\"')
            patch_cmd = "patch cluster -n tkg-system %s --type=merge -p \"%s\"" % (clusterName, patch_json)
            self.run_cmd(patch_cmd)
            logger.info("update clusterclass mgmt cluster [%s] in namespace tkg-system successfully" % clusterName)
        else:
            logger.info("clusterclass mgmt cluster [%s] in namespace tkg-system up to date" % clusterName)

    def cc_verify(self, clusterName):
        self.verify_kapp_controller_cm()
        self.verify_tkg_pkg_secret()
        self.verify_tkr_controller_secrets()
        self.verify_tkr_controller_cm()
        GET_CLUSTER_CMD = "cluster -n tkg-system %s" % clusterName
        clusterCR = self.get_dict(GET_CLUSTER_CMD)
        upToDate = True
        for var in clusterCR['spec']['topology']['variables']:
            if var['name'] == 'trust' and 'additionalTrustedCAs' in var['value']:
                for ca in var['value']['additionalTrustedCAs']:
                    if ca['name'] == 'imageRepository' and ca['data'] != self.repo.get_base64_ca():
                        logger.error("cluster [%s] in namespace tkg-system: out of date" % clusterName)
                        upToDate = False
            elif var['name'] == 'customTDNFRepository' and 'certificate' in var['value']:
                if var['value']['certificate'] != self.repo.get_base64_ca():
                    logger.error("cluster [%s] in namespace tkg-system: out of date" % clusterName)
                    upToDate = False
        if upToDate:
            logger.info("cluster [%s] in namespace tkg-system: up to date" % clusterName)

    def update(self):
        self.update_kapp_controller_cm()
        self.update_tkg_pkg_secret()
        self.update_tkr_controller_secrets()
        self.update_tkr_controller_cm()
        self.update_kcp("tkg-system")
        self.update_kcts("tkg-system")

    def verify(self):
        self.verify_kapp_controller_cm()
        self.verify_tkg_pkg_secret()
        self.verify_tkr_controller_secrets()
        self.verify_tkr_controller_cm()
        self.verify_kcp("tkg-system")
        self.verify_kcts("tkg-system")

class WrklClusterClient(TkgClusterClient):
    def __init__(self, kubeconfig_file, repo):
        super().__init__(kubeconfig_file, repo)

    def update(self):
        self.update_kapp_controller_cm()

    def verify(self):
        self.verify_kapp_controller_cm()

class TkoClusterClient(ClusterClient):
    GET_ALL_TKC_CLUSTERS = "tkc -A"
    FQDN_ANNO = "telco.vmware.com/airgap-fqdn"
    CA_ANNO = "telco.vmware.com/airgap-ca-cert"
    GET_TKO_WEBHOOK_CMD = "ValidatingWebhookConfiguration tca-kubecluster-operator-validating-webhook-configuration"
    DISABLE_TKO_WEBHOOK_CMD = "delete ValidatingWebhookConfiguration tca-kubecluster-operator-validating-webhook-configuration"

    WEBHOOK_BACKUP_FILE = "%s/tko.webhook" % TMP_DIR

    def __init__(self, kubeconfig_file, airgap_repo):
        super().__init__(kubeconfig_file)
        self.airgap_repo = airgap_repo
        self.tkcs = None
        self.mc_client = MgmtClusterClient(self.kubeconfig_file, self.airgap_repo)
        self.webhook = None

    def update_cluster_anno_ca(self, cluster_name, ns):
        base64_ca = self.airgap_repo.get_base64_ca()
        patch_cmd = "annotate --overwrite tkc %s -n %s %s=%s" % (cluster_name, ns, self.CA_ANNO, base64_ca)
        self.run_cmd(patch_cmd)
        logger.info("update cluster %s tkc airgap ca annotation successfully" % cluster_name)

    def update_cluster_spec_ca(self, cluster_name, ns):
        base64_ca = self.airgap_repo.get_base64_ca()
        patch_json = "{\"spec\":{\"airgap\":{\"caCert\":\"%s\"}}}" % base64_ca
        patch_cmd = "patch tkc %s -n %s --type=merge -p \'%s\'" % (cluster_name, ns, patch_json)
        self.run_cmd(patch_cmd)
        logger.info("update cluster %s tkc spec airgap cacert successfully" % cluster_name)

    def get_all_tkcs(self):
        if self.tkcs:
            return self.tkcs
        try:
            res = self.get_dict(self.GET_ALL_TKC_CLUSTERS)
        except Exception as e:
            logger.critical("failed to get tcakubenetescluster crs, err: %s" % repr(e))
            traceback.print_exc()
            return []
        self.tkcs = res['items']
        return self.tkcs

    def backup_and_disable_webhook(self):
        try:
            webhook = self.get_yaml(self.GET_TKO_WEBHOOK_CMD)
        except NameError as e:
            logger.debug("webhook doesn't exist, skip disabling...")
            return
        with open(self.WEBHOOK_BACKUP_FILE, 'wb') as f:
            f.write(webhook)
            f.close()
        self.webhook = self.WEBHOOK_BACKUP_FILE
        logger.debug("back up tko webhook to %s successfully" % self.webhook)

        self.run_cmd(self.DISABLE_TKO_WEBHOOK_CMD)
        logger.debug("disable tko webhook successfully")

    def restore_webhook(self):
        if self.webhook is None:
            return
        self.apply_yaml(self.webhook)
        logger.debug("restore tko webhook successfully")

    def update_clusters_tkc(self, cluster_name):
        fqdn = self.airgap_repo.get_fqdn()

        self.backup_and_disable_webhook()
        try:
            for cluster in self.get_all_tkcs():
                name = cluster['metadata']['name']
                namespace = cluster['metadata']['namespace']
                if cluster_name is not None and cluster_name != name:
                    continue
                if 'airgap' in cluster['spec'] and cluster['spec']['airgap']['fqdn'] == fqdn:
                    logger.info("updating cluster [%s/%s] tkc spec airgap repo ca cert" % (name, namespace))
                    self.update_cluster_spec_ca(name, namespace)
                elif ('annotations' in cluster['metadata'] and
                      self.FQDN_ANNO in cluster['metadata']['annotations'] and
                      cluster['metadata']['annotations'][self.FQDN_ANNO] == fqdn):
                    logger.info("updating cluster [%s/%s] tkc airgap annotation repo ca cert" % (name, namespace))
                    self.update_cluster_anno_ca(name, namespace)
                else:
                    logger.debug("skip updating non-airgapped to [%s] cluster [%s/%s] tkc" % (fqdn, name, namespace))
        finally:
            self.restore_webhook()

    def update_clusters_kcpt(self, cluster_name):
        fqdn = self.airgap_repo.get_fqdn()
        for cluster in self.get_tkcs_of_repo():
            namespace = cluster['metadata']['namespace']
            name = cluster['metadata']['name']
            if cluster_name is not None and cluster_name != name:
                continue
            self.mc_client.update_kcp(namespace)
            self.mc_client.update_kcts(namespace)
            logger.info("update cluster [%s/%s] kcp and kct successfully" % (name, namespace))

    def get_tkcs_of_repo(self):
        tkcs = []
        fqdn = self.airgap_repo.get_fqdn()
        for tkc in self.get_all_tkcs():
            if (('airgap' in tkc['spec'] and tkc['spec']['airgap']['fqdn'] == fqdn) or
                ('annotations' in tkc['metadata'] and
                  self.FQDN_ANNO in tkc['metadata']['annotations'] and
                  tkc['metadata']['annotations'][self.FQDN_ANNO] == fqdn)):
                tkcs.append(tkc)
        return tkcs

    def get_tkc_kubeconfig(self, tkc):
        cluster_name = tkc['metadata']['name']
        cluster_ns = tkc['metadata']['namespace']
        kubeconfig_cmd = "get secret %s-kubeconfig -n %s -o jsonpath={.data.value} | base64 -d" % (cluster_name, cluster_ns)
        return self.run_cmd(kubeconfig_cmd)

    def get_tkc_node_ips(self, tkc):
        ips = []
        logger.debug("query cluster %s node ips" % tkc['metadata']['name'])
        get_vspherevms_cmd = "vspherevm -n %s" % tkc['metadata']['namespace']
        vms = self.get_dict(get_vspherevms_cmd)['items']
        for vm in vms:
            if 'status' in vm and 'addresses' in vm['status']:
                ips.append(vm['status']['addresses'][0])
        return ips

    def verify_tkc(self, tkc):
        fqdn = self.airgap_repo.get_fqdn()
        base64_ca = self.airgap_repo.get_base64_ca()
        name = tkc['metadata']['name']
        namespace = tkc['metadata']['namespace']
        if 'airgap' in tkc['spec'] and tkc['spec']['airgap']['fqdn'] == fqdn:
            if ('caCert' not in tkc['spec']['airgap'] or
                tkc['spec']['airgap']['caCert'] != base64_ca):
                logger.error("tcakubenetescluster cr[%s/%s]: out of date" % (name, namespace))
            else:
                logger.info("tcakubenetescluster cr[%s/%s]: up to date" % (name, namespace))
        elif ('annotations' in tkc['metadata'] and
              self.FQDN_ANNO in tkc['metadata']['annotations'] and
              tkc['metadata']['annotations'][self.FQDN_ANNO] == fqdn):
            if (self.CA_ANNO not in tkc['metadata']['annotations'] or
                tkc['metadata']['annotations'][self.CA_ANNO] != base64_ca):
                logger.error("tcakubenetescluster cr[%s/%s]: out of date" % (name, namespace))
            else:
                logger.info("tcakubenetescluster cr[%s/%s]: up to date" % (name, namespace))
        else:
            logger.debug("skip tcakubernetescluster cr[%s/%s]: non target airgap repo"
                        % (name, namespace))

    def verify_kcpt(self, tkc):
        fqdn = self.airgap_repo.get_fqdn()
        namespace = tkc['metadata']['namespace']
        name = tkc['metadata']['name']
        if 'airgap' in tkc['spec'] and tkc['spec']['airgap']['fqdn'] == fqdn:
            self.mc_client.verify_kcp(namespace)
            self.mc_client.verify_kcts(namespace)
        else:
            logger.debug("skip cluster [%s/%s] kcp and kct: non target airgap repo" % (name, namespace))

    def update_kapp_secret(self, tkc):
        name = tkc['metadata']['name']
        self.mc_client.update_kapp_secret(name)

    def verify_kapp_secret(self, tkc):
        name = tkc['metadata']['name']
        self.mc_client.verify_kapp_secret(name)

    def update(self, cluster_name):
        self.update_clusters_tkc(cluster_name)
        self.update_clusters_kcpt(cluster_name)

class MiniKubeClient(TkoClusterClient):
    MINIKUBE_CONFIG_FILE = "/home/admin/.kube/config"

    def __init__(self, airgap_repo):
        super().__init__(self.MINIKUBE_CONFIG_FILE, airgap_repo)

    def verify(self):
        fqdn = self.airgap_repo.get_fqdn()
        for tkc in self.get_all_tkcs():
            self.verify_tkc(tkc)
        return

    def update(self):
        self.update_clusters_tkc(None)


class NodeUpdater:
    ANSIBLE_PATH = "%s/ansible" % CWD
    ANSIBLE_PLAYBOOK = ANSIBLE_PATH + "/update_node_ca.yml"
    ANSIBLE_INVENTORY_FILE = ANSIBLE_PATH + "/hosts"
    ANSIBLE_VARS_FILE = ANSIBLE_PATH + "/vars.yml"

    def __init__(self, server_fqdn, ca_file):
        self.vars = {}
        self.vars['server_fqdn'] = server_fqdn
        self.vars['ca_file'] = ca_file
        self.vars['ansible_user'] = 'capv'
        self.hosts_config = configparser.ConfigParser(allow_no_value=True)

    def add_cluster(self, cluster_name):
        self.hosts_config.add_section(cluster_name.replace('-',''))

    def add_nodeips_of_cluster(self, cluster_name, node_ips):
        if not cluster_name in self.hosts_config:
            self.add_cluster(cluster_name)
        for ip in node_ips:
            self.hosts_config.set(cluster_name.replace('-',''), ip)

    def dump_configs(self):
        with open(self.ANSIBLE_INVENTORY_FILE, 'w') as host_file:
            self.hosts_config.write(host_file)
        with open(self.ANSIBLE_VARS_FILE, 'w') as vars_file:
            yaml.dump(self.vars, vars_file)

    def run(self):
        global ansible_log_file
        if len(self.hosts_config.sections()) == 0:
            logger.info("no legacy cluster node to update, skip")
            return
        self.dump_configs()
        # run ansible playbook
        cmd = "ansible-playbook -i %s %s" % (self.ANSIBLE_INVENTORY_FILE, self.ANSIBLE_PLAYBOOK)
        if ansible_log_file is not None:
            with open(ansible_log_file, 'w') as f:
                logger.debug("output ansible log to file")
                p = subprocess.Popen(cmd.split(), stdout=f,
                                     stderr=f, close_fds=True)
        else:
            logger.debug("output ansible log to console")
            p = subprocess.Popen(cmd.split(), stdout=sys.stdout,
                                 stderr=sys.stderr, close_fds=True)
        p.communicate()
        rc = p.returncode
        if rc:
            raise RuntimeError("concurr error when updating ca certificate on clusters nodes")
        logger.info("cluster nodes are updated successfully")


def create_airgap_repo(fqdn, cafile):
    repo = AirgapRepo(fqdn, cafile)
    if repo.has_verified():
        return repo
    valid, errmsg = repo.is_valid()
    if not valid:
        sys.exit("cafile %s for airgap repo %s is invalid: %s" % (cafile, fqdn, errmsg))
    else:
        logger.info("cafile %s for airgap repo %s is valid" % (cafile, fqdn))
    return repo

def update_tkg_contexts(args):
    repo = create_airgap_repo(args.fqdn, args.cafile)
    kbs = Kbs(repo)
    kbs.load_contexts()
    kbs.update_contexts()

def verify_tkg_contexts(args):
    repo = create_airgap_repo(args.fqdn, args.cafile)
    logger.info("########## verifying tkgcontexts ##########")
    try:
        kbs = Kbs(repo)
        kbs.load_contexts()
        kbs.verify_contexts()
    except Exception as e:
        logger.critical("failed to verify tkgcontexts, err: %s" % repr(e))
        traceback.print_exc()

def verify_nodes(ips, repo):
    fqdn = repo.get_fqdn()
    for ip in ips:
        try:
            cmd = ["ssh", "capv@%s" % ip, "-o", "ConnectTimeout=15",
                   "-o", "BatchMode=yes", "-o", "StrictHostKeyChecking=no",
                   "curl https://%s --head -s" % fqdn]
            p = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                                 stderr=subprocess.PIPE, close_fds=True)
            stdout, stderr = p.communicate()
            rc = p.returncode
            res = stdout.decode('utf-8')
            err = stderr.decode('utf-8')
            if rc == 0 and '200 OK' in res:
                logger.info("node[%s]: up to date" % ip)
            elif rc == 60:
                logger.error("node[%s]: out of date, SSL certificate problem" % ip)
            else:
                logger.error("node[%s]: cannot communicate with airgap repo, return code:%s, stdout: %s, stderr: %s" % (ip, rc, res, err))
        except Exception as e:
            logger.critical("node[%s]: failed to verify, err: %s" % (ip, repr(e)))

def update_mgmt_clusters(args):
    nodes_updated = True
    crs_updated = True
    not_found = True
    try:
        cluster_name = args.name
    except AttributeError as e:
        logger.debug("cluster name argument is not set")

    repo = create_airgap_repo(args.fqdn, args.cafile)

    kbs = Kbs(repo)
    mgmt_clusters = kbs.get_mgmtclusters_of_repo()
    mgmt_clusters_legacy = []
    #0. update clusterclass mc
    for cluster in mgmt_clusters:
        if cluster_name is not None and args.name != cluster["clusterName"]:
            continue
        if not_found:
            not_found = False
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = MgmtClusterClient(kubeconfig, repo)
        if cluster_client.IsClusterClass(cluster["clusterName"]):
            cluster_client.cc_update(cluster["clusterName"])
        else:
            mgmt_clusters_legacy.append(cluster)

    if not_found:
        if cluster_name is not None:
            logger.info("management cluster %s is not found or not associated with airgap repo %s" % (args.name, args.fqdn))
        else:
            logger.info("no management cluster is associating with airgap repo %s" % (args.fqdn))
        return

    #1. update mgmt cluster nodes
    logger.info("start updating legacy management clusters' nodes")
    node_updater = NodeUpdater(args.fqdn, args.cafile)
    for cluster in mgmt_clusters_legacy:
        ips = kbs.get_mgmtcluster_node_ips(cluster)
        logger.info("cluster %s, node ips: %s" % (cluster['clusterName'], ips))
        node_updater.add_nodeips_of_cluster(cluster['clusterName'], ips)
    try:
        node_updater.run()
        logger.info("end updating management clusters' nodes successfully")
    except Exception as e:
        nodes_updated = False
        logger.critical("failed to management clusters' nodes with errors: %s" % repr(e))
        traceback.print_exc()

    #2. update mgmt cluster's config maps, kcp, kct etc.
    logger.info("start updating management clusters' resources: config maps/kcp/kct")
    for cluster in mgmt_clusters_legacy:
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = MgmtClusterClient(kubeconfig, repo)
        try:
            cluster_client.update()
            logger.info("end updating management clusters' resources: config maps/kcp/kct successfully")
        except Exception as e:
            crs_updated = False
            logger.critical("failed to update mgmt cluster configmap/crs with errors: %s" % repr(e))
            traceback.print_exc()

    if nodes_updated and crs_updated: 
        logger.info("management clusters configuration has been updated successfully!")
        logger.warning("relevant management cluster nodes may be rolling updated, please check the results by running the verify-mgmtclusters command")

def verify_mgmt_clusters(args):
    logger.info("########## verifying management clusters ##########")
    repo = create_airgap_repo(args.fqdn, args.cafile)
    not_found = True
    try:
        cluster_name = args.name
    except AttributeError as e:
        logger.debug("cluster name argument is not set")

    kbs = Kbs(repo)
    mgmt_clusters = kbs.get_mgmtclusters_of_repo()

    for cluster in mgmt_clusters:
        if cluster_name is not None and args.name != cluster["clusterName"]:
            continue
        if not_found:
            not_found = False
        logger.info("# verifying management cluster[%s], id[%s]"
                    % (cluster['clusterName'], cluster['id']))
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = MgmtClusterClient(kubeconfig, repo)
        if cluster_client.IsClusterClass(cluster["clusterName"]):
            cluster_client.cc_verify(cluster["clusterName"])
        else:
            cluster_client.verify()
        ips = kbs.get_mgmtcluster_node_ips(cluster)
        verify_nodes(ips, repo)

    if not_found:
        if cluster_name is not None:
            logger.info("management cluster %s is not found or not associated with airgap repo %s" % (args.name, args.fqdn))
        else:
            logger.info("no management cluster is associating with airgap repo %s" % (args.fqdn))
        return

def update_v1_workload_clusters(args):
    nodes_updated = True
    crs_updated = True
    not_found = True
    try:
        cluster_name = args.name
    except AttributeError as e:
        logger.debug("cluster name argument is not set")

    repo = create_airgap_repo(args.fqdn, args.cafile)
    kbs = Kbs(repo)
    wrkl_clusters = kbs.get_wrklclusters_of_repo()

    node_updater = NodeUpdater(args.fqdn, args.cafile)
    for cluster in wrkl_clusters:
        if cluster_name is not None and args.name != cluster["clusterName"]:
            continue
        if not_found:
            not_found = False
        ips = kbs.get_wrklcluster_node_ips(cluster)
        logger.info("cluster %s, node ips: %s" % (cluster['clusterName'], ips))
        node_updater.add_nodeips_of_cluster(cluster['clusterName'], ips)

    if not_found:
        if cluster_name is not None:
            logger.info("v1 workload cluster %s is not found or not associated with airgap repo %s" % (args.name, args.fqdn))
        else:
            logger.info("no v1 workload cluster is associating with airgap repo %s" % (args.fqdn))
        return

    #1. update all workload clusters' nodes
    try:
        logger.info("start updating v1 workload clusters' nodes")
        node_updater.run()
        logger.info("end updating v1 workload clusters' nodes successfully")
    except Exception as e:
        nodes_updated = False
        logger.error("failed to update node certificate with errors: %s" % repr(e))
        traceback.print_exc()

    logger.info("start updating workload clusters' resources: kapp-controller-config/kcp/kcts")
    for cluster in wrkl_clusters:
        if cluster_name is not None and args.name != cluster["clusterName"]:
            continue
        #2. update workload cluster's kapp_secret/kcp/kcts on its associating mgmt cluster
        mc_kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["tkgMgmtClusterId"]
        mc_client = MgmtClusterClient(mc_kubeconfig, repo)
        try:
            mc_client.update_kapp_secret(cluster['clusterName'])
            mc_client.update_kcp(cluster['clusterName'])
            mc_client.update_kcts(cluster['clusterName'])
            logger.info("update cluster [%s] kapp, kcp and kcts successfully" % (cluster['clusterName']))
        except Exception as e:
            crs_updated = False
            logger.critical("failed to update cluster [%s] kapp/kcp/kct on mgmt with errors: %s" % (cluster['clusterName'], repr(e)))
            traceback.print_exc()

        #3. update workload cluster's kapp-controller-config
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = WrklClusterClient(kubeconfig, repo)
        try:
            cluster_client.update()
        except Exception as e:
            crs_updated = False
            logger.critical("failed to update cluster [%s] configmap with errors: %s" % (cluster['clusterName'], repr(e)))
            traceback.print_exc()

    if crs_updated:
        logger.info("end updating v1 workload clusters' resources: kapp-controller-config/kcp/kcts successfully")
    if crs_updated and nodes_updated:
        logger.info("v1 workload clusters configuration has been updated successfully!")
        logger.warning("relevant cluster control plane nodes may be rolling updated, please check the results by running the verify-v1cluster command")

def verify_v1_workload_clusters(args):
    not_found = True
    try:
        cluster_name = args.name
    except AttributeError as e:
        logger.debug("cluster name argument is not set")

    repo = create_airgap_repo(args.fqdn, args.cafile)
    logger.info("########## verifying v1 workload clusters ##########")
    kbs = Kbs(repo)
    wrkl_clusters = kbs.get_wrklclusters_of_repo()
    for cluster in wrkl_clusters:
        if cluster_name is not None and args.name != cluster["clusterName"]:
            continue
        if not_found:
            not_found = False

        logger.info("# verifying workload cluster[%s], id[%s]"
                    % (cluster["clusterName"], cluster["id"]))

        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = WrklClusterClient(kubeconfig, repo)
        cluster_client.verify()

        mc_kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["tkgMgmtClusterId"]
        mc_client = MgmtClusterClient(mc_kubeconfig, repo)
        mc_client.verify_kapp_secret(cluster['clusterName'])
        mc_client.verify_kcp(cluster['clusterName'])
        mc_client.verify_kcts(cluster['clusterName'])

        ips = kbs.get_wrklcluster_node_ips(cluster)
        verify_nodes(ips, repo)

    if not_found:
        if cluster_name is not None:
            logger.info("v1 workload cluster %s is not found or not associated with airgap repo %s" % (args.name, args.fqdn))
        else:
            logger.info("no v1 workload cluster is associating with airgap repo %s" % (args.fqdn))
        return

def update_v2_workload_clusters(args):
    nodes_updated = True
    crs_updated = True
    minikube_crs_updated = True
    not_found = True
    try:
        cluster_name = args.name
    except AttributeError as e:
        logger.debug("cluster name argument is not set")

    repo = create_airgap_repo(args.fqdn, args.cafile)
    node_updater = NodeUpdater(args.fqdn, args.cafile)
    kbs = Kbs(repo)
    mgmt_clusters = kbs.get_mgmtclusters()
    for cluster in mgmt_clusters:
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = TkoClusterClient(kubeconfig, repo)
        tkcs = cluster_client.get_tkcs_of_repo()
        for tkc in tkcs:
            name = tkc['metadata']['name']
            if cluster_name is not None and args.name != name:
                continue
            if not_found:
                not_found = False
                logger.info("start updating v2 workload clusters kapp-controller-config and nodes")
                #1. update minikube mgmt cr
                logger.info("start updating mgmt clusters' cr in minikube")
                try:
                    mk_client = MiniKubeClient(repo)
                    mk_client.update()
                    logger.info("end updating mgmt clusters' cr in minikube successfully")
                except Exception as e:
                    minikube_crs_updated = False
                    logger.critical("failed to udpate minikube mgmt clusters' cr with errors: %s" % repr(e))
                    traceback.print_exc()

            try:
                # 3.1. read workload cluster node ips
                ips = cluster_client.get_tkc_node_ips(tkc)
                logger.info("cluster %s, node ips: %s" % (name, ips))
                node_updater.add_nodeips_of_cluster(name, ips)
            except Exception as e:
                nodes_udpated = False
                logger.critical("failed to get v2 workload cluster [%s] node ips with errors: %s" % (name, repr(e)))
                traceback.print_exc()

            # 2.1 update workload cluster's kapp-controller-addon secret
            try:
                cluster_client.update_kapp_secret(tkc)
            except Exception as e:
                crs_updated = False
                logger.critical("failed to update kapp secret for cluster [%s], err: %s"
                                % (name, repr(e)))
                traceback.print_exc()

            try:
                # 2.2 update workload cluster's kapp-controller-config
                kc = cluster_client.get_tkc_kubeconfig(tkc)
                path = "/var/run/%s.kc" % name
                with open(path, 'wb') as f:
                    f.write(kc)
                    f.close()
                wc_client = WrklClusterClient(path, repo)
                wc_client.update()
            except Exception as e:
                crs_updated = False
                logger.critical("failed to update workload cluster [%s] kapp-controller-config with errors: %s" % (name, repr(e)))
                traceback.print_exc()

    if not_found:
        if cluster_name is not None:
            logger.info("v2 workload cluster %s is not found or not associated with airgap repo %s" % (args.name, args.fqdn))
        else:
            logger.info("no v2 workload cluster is associating with airgap repo %s" % (args.fqdn))
        return

    # 3.2. update workload clusters' node
    try:
        node_updater.run()
    except Exception as e:
        nodes_updated = False
        logger.critical("failed to update nodes certificate with errors: %s" % repr(e))
        traceback.print_exc()

    if nodes_updated and crs_updated:
        logger.info("end updating v2 workload clusters kapp-controller-config and nodes successfully")

    # 4. update workload clusters' tkc/kcp/kcts on its corresponding mgmt cluster
    logger.info("start updating workload clusters' crs on mgmt clusters")
    for cluster in mgmt_clusters:
        logger.info("checking mgmt cluster [%s] for wcs to update tkc/kcp/kct" % cluster['clusterName'])
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = TkoClusterClient(kubeconfig, repo)
        try:
            cluster_client.update(cluster_name)
            logger.info("end updating workload clusters' crs on mgmt cluster [%s] successfully" % cluster['clusterName'])
        except Exception as e:
            crs_updated = False
            logger.critical("failed to update airgap ca cert to tkc/kcp/kct on mgmt cluster [%s] with errors: %s" % (cluster['clusterName'], repr(e)))
            traceback.print_exc()

    if crs_updated and minikube_crs_updated and nodes_updated:
        logger.info("v2 workload clusters configuration has been updated successfully!")
        logger.warning("relevant cluster control plane nodes may be rolling updated, please check the results by running the verify-v2cluster command")

def verify_v2_workload_clusters(args):
    not_found = True
    try:
        cluster_name = args.name
    except AttributeError as e:
        logger.debug("cluster name argument is not set")

    repo = create_airgap_repo(args.fqdn, args.cafile)
    logger.info("########## verifying minikube ##########")
    try:
        mk_client = MiniKubeClient(repo)
        mk_client.verify()
    except Exception as e:
        logger.critical("failed to verify management cluster tkc in minikube")
        traceback.print_exc()

    logger.info("########## verifying v2 workload clusters ##########")
    kbs = Kbs(repo)
    mgmt_clusters = kbs.get_mgmtclusters()
    for cluster in mgmt_clusters:
        kubeconfig = "/opt/vmware/k8s-bootstrapper/%s/kubeconfig" % cluster["id"]
        cluster_client = TkoClusterClient(kubeconfig, repo)
        tkcs = cluster_client.get_tkcs_of_repo()
        for tkc in tkcs:
            name = tkc['metadata']['name']
            if cluster_name is not None and args.name != name:
                continue
            if not_found:
                not_found = False
            logger.info("# verifying v2 workload cluster[%s]" % name)
            cluster_client.verify_tkc(tkc)
            cluster_client.verify_kcpt(tkc)
            cluster_client.verify_kapp_secret(tkc)
            try:
                kc = cluster_client.get_tkc_kubeconfig(tkc)
                path = "/var/run/%s.kc" % name
                with open(path, 'wb') as f:
                    f.write(kc)
                    f.close()
            except Exception as e:
                logger.critical("failed to get workload cluster[%s] kubeconfig, err: %s" % (name, repr(e)))
                traceback.print_exc()
                continue
            wc_client = WrklClusterClient(path, repo)
            wc_client.verify()
            ips = cluster_client.get_tkc_node_ips(tkc)
            verify_nodes(ips, repo)
        if not_found:
            if cluster_name is not None:
                logger.info("v2 workload cluster %s is not found or not associated with airgap repo %s" % (args.name, args.fqdn))
            else:
                logger.info("no v2 workload cluster is associating with airgap repo %s" % (args.fqdn))
        return

def update_all(args):
    args.name = None
    update_tkg_contexts(args)
    update_mgmt_clusters(args)
    update_v1_workload_clusters(args)
    update_v2_workload_clusters(args)

def verify_all(args):
    args.name = None
    verify_tkg_contexts(args)
    verify_mgmt_clusters(args)
    verify_v1_workload_clusters(args)
    verify_v2_workload_clusters(args)

def parse_args():
    parser = argparse.ArgumentParser(
        description='TCA CaaS update airgap repository CA certificate tool')

    parser.add_argument('--loglevel', choices=['debug','info','error','warning','critical'],
                        default='info', help='log level of script')
    parser.add_argument('--logdst', choices=['console', 'file'],
                        default='console', help='log destination')
    parser.add_argument('--ansible_log_file', default='/common/logs/update-cert.log',
                        help='log file path if the logging to a file')

    subparser = parser.add_subparsers(help='-h for additional help')

    sp = subparser.add_parser('update-tkgcontexts', help='update KBS tkg contexts associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    sp.set_defaults(func=update_tkg_contexts)

    sp = subparser.add_parser('update-mgmtclusters',
                              help='update mgmt clusters associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    ag = sp.add_argument_group('optional arguments')
    ag.add_argument('--name', required=False, help='cluster name to update')
    sp.set_defaults(func=update_mgmt_clusters)

    sp = subparser.add_parser('update-v1clusters',
                              help='update v1 workload clusters associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    ag = sp.add_argument_group('optional arguments')
    ag.add_argument('--name', required=False, help='cluster name to update')
    sp.set_defaults(func=update_v1_workload_clusters)

    sp = subparser.add_parser('update-v2clusters',
                              help='update v2 workload clusters associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    ag = sp.add_argument_group('optional arguments')
    ag.add_argument('--name', required=False, help='cluster name to update')
    sp.set_defaults(func=update_v2_workload_clusters)

    sp = subparser.add_parser('update-all', help='update airgap CA certificate on all relevant components of TCA-CP')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    sp.set_defaults(func=update_all)

    sp = subparser.add_parser('verify-tkgcontexts', help='verify KBS tkg contexts associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    sp.set_defaults(func=verify_tkg_contexts)

    sp = subparser.add_parser('verify-mgmtclusters', help='verify mgmt clusters associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    ag = sp.add_argument_group('optional arguments')
    ag.add_argument('--name', required=False, help='cluster name to verify')
    sp.set_defaults(func=verify_mgmt_clusters)

    sp = subparser.add_parser('verify-v1clusters', help='verify v1 workload clusters associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    ag = sp.add_argument_group('optional arguments')
    ag.add_argument('--name', required=False, help='cluster name to verify')
    sp.set_defaults(func=verify_v1_workload_clusters)

    sp = subparser.add_parser('verify-v2clusters', help='verify v2 workload clusters associating with the airgap repository')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    ag = sp.add_argument_group('optional arguments')
    ag.add_argument('--name', required=False, help='cluster name to verify')
    sp.set_defaults(func=verify_v2_workload_clusters)

    sp = subparser.add_parser('verify-all', help='verify all relevant components of TCA-CP have updated CA content')
    ag = sp.add_argument_group('required arguments')
    ag.add_argument('--fqdn', required=True, help='FQDN of the airgap repository server')
    ag.add_argument('--cafile', required=True, help='new CA certificate file of the airgap repository server')
    sp.set_defaults(func=verify_all)

    return parser.parse_args()

def main(argv=None):
    pathlib.Path(TMP_DIR).mkdir(parents=True, exist_ok=True)
    args = parse_args()
    logger_setup(log_dst=args.logdst, log_level=args.loglevel, log_file=args.ansible_log_file)
    init_yaml()
    args.func(args)

if __name__== "__main__":
    sys.exit(main())
