import json
import time
import subprocess
import os
import sys

token_file_path = "/var/run/secrets/kubernetes.io/serviceaccount/token"

def check_pod_ready(container_statuses):
    ready = True
    for container_status in container_statuses:
        print("-> container {} readiness {}".format(container_status['name'], container_status['ready']))
        if container_status['ready'] == False:
            ready = False
    return ready

def get_pods(pod_namespace, pod_label):
    token_file = open(token_file_path, "r")
    token = token_file.read()
    cmd = 'curl -sk -H "Authorization: Bearer {}" https://kubernetes.default.svc/api/v1/namespaces/{}/pods?labelSelector={}'.format(token, pod_namespace, pod_label)
    ps = subprocess.Popen(cmd, shell=True,
                          stdout = subprocess.PIPE,
                          stderr = subprocess.STDOUT)
    output, err = ps.communicate()
    token_file.close()

    return output, err

def get_replica_count(ownerReference, pod_namespace):
    token_file = open(token_file_path, "r")
    token = token_file.read()
    ownerKind = ownerReference['kind']
    ownerName = ownerReference['name']
    ownerApiVersion = ownerReference['apiVersion']

    cmd = 'curl -sk -H "Authorization: Bearer {}" https://kubernetes.default.svc/apis/{}/namespaces/{}/{}s/{}'.format(token, ownerApiVersion, pod_namespace, ownerKind.lower(), ownerName)
    ps = subprocess.Popen(cmd, shell=True,
                          stdout = subprocess.PIPE,
                          stderr = subprocess.STDOUT)
    output, err = ps.communicate()
    token_file.close()
    replicaCount = 0
    if err:
        print('Error in getting {} {} '.format(ownerName, ownerKind))
        print(err)
    else:
        try:
            ownerResource = json.loads(output)
            replicaCount = ownerResource['status']['replicas']
        except KeyError:
            print('Error in getting replicaCount of {} {} '.format(ownerName, ownerKind))

    return replicaCount

def main():
    start = time.time()
    DEFAULT_TIMEOUT_IN_MINS = '5'
    TIMEOUT_IN_MINS = os.getenv('TIMEOUT_IN_MINS', DEFAULT_TIMEOUT_IN_MINS)
    timeout_in_mins = int(DEFAULT_TIMEOUT_IN_MINS)
    try:
        timeout_in_mins = int(TIMEOUT_IN_MINS)
    except ValueError:
        print('Given TIMEOUT_IN_MINS {} is not a number, Proceeding with default timeout of {} mins'.format(TIMEOUT_IN_MINS, DEFAULT_TIMEOUT_IN_MINS))
    timeout = 60*timeout_in_mins
    sleep_interval = 5
    pods_ready = False
    POD_LABEL = os.getenv('POD_LABEL', 'app.kubernetes.io/name=redis-upgrade')
    POD_NAMESPACE = os.getenv('MY_POD_NAMESPACE', 'default')
    while not pods_ready and time.time() - start < timeout:
        pods_ready = True
        pod_list, err = get_pods(POD_NAMESPACE, POD_LABEL)
        if err:
            print(err)
            pods_ready = False
            time.sleep(sleep_interval)
            continue
        data = json.loads(pod_list)
        pod_items = None
        try:
            pod_items = data['items']
        except KeyError:
            pods_ready = False
            print('Waiting for pods with label {} to be created...'.format(POD_LABEL))
            time.sleep(sleep_interval)
            continue

        if not pod_items:
            pods_ready = False
            print('Waiting for pods with label {} to be created...'.format(POD_LABEL))
            time.sleep(sleep_interval)
            continue

        for pod_item in pod_items:
            pod_name = pod_item['metadata']['name']
            container_statuses = None
            try:
                container_statuses = pod_item['status']['containerStatuses']
            except KeyError:
                pods_ready = False
                print('Waiting for containers in pod {} to be created...'.format(pod_name))
                time.sleep(sleep_interval)
                continue


            if not container_statuses:
                pods_ready = False
                print('Waiting for containers in pod {} to be created...'.format(pod_name))
                time.sleep(sleep_interval)
                continue
            print('')
            print('checking readiness of pod {}'.format(pod_name))
            is_pod_ready = check_pod_ready(container_statuses)
            if is_pod_ready == False:
                pods_ready = False

        if pods_ready == True:
            print('')
            print('Confirming whether replicaCount and readyReplicaCount are same for pods with label {}...'.format(POD_LABEL))
            ownerReference = None
            try:
                ownerReference = pod_items[0]['metadata']['ownerReferences'][0]
            except KeyError:
                print('No owner for pods with label {} '.format(POD_LABEL))
            if not ownerReference:
                print('No owner References for pod {} to check replicaCount'.format(POD_LABEL))
            else:
                replicaCount = get_replica_count(ownerReference, POD_NAMESPACE)
                if replicaCount:
                    if len(pod_items) == replicaCount:
                        print('replicaCount = readyReplicaCount ({}={})'.format(replicaCount, len(pod_items)))
                    else:
                        print('replicaCount != readyReplicaCount ({}!={}). Waiting for remaining pods to come up'.format(replicaCount, len(pod_items)))
                        pods_ready = False
                else:
                    print('Failed to get the replicaCount of {} {}. Assuming Success'.format(ownerReference['kind'], ownerReference['name']))

        if pods_ready == False:
            print('')
            print('Waiting for pods with label {} to be ready...'.format(POD_LABEL))
            time.sleep(sleep_interval)

    if pods_ready == True:
        print('Pods with label {} are ready'.format(POD_LABEL))
    else:
        sys.exit('timed out waiting for pods {} ready'.format(POD_LABEL))

if __name__ == "__main__":
    main()