import json
import time
import subprocess
import os
import sys

def check_pod_ready(container_statuses):
    ready = True
    for container_status in container_statuses:
        print("-> container {} readiness {}".format(container_status['name'], container_status['ready']))
        if container_status['ready'] == False:
            ready = False
    return ready

def get_pods(pod_namespace, pod_label):
    token_file_path = "/var/run/secrets/kubernetes.io/serviceaccount/token"
    token_file = open(token_file_path, "r")
    token = token_file.read()
    cmd = 'curl -sk -H "Authorization: Bearer {}" https://kubernetes.default.svc/api/v1/namespaces/{}/pods?labelSelector=app%3D{}'.format(token, pod_namespace, pod_label)
    ps = subprocess.Popen(cmd, shell=True,
                          stdout = subprocess.PIPE,
                          stderr = subprocess.STDOUT)
    output, err = ps.communicate()
    token_file.close()
    return output, err

def main():
    start = time.time()
    DEFAULT_TIMEOUT_IN_MINS = '5'
    TIMEOUT_IN_MINS = os.getenv('TIMEOUT_IN_MINS', DEFAULT_TIMEOUT_IN_MINS)
    timeout_in_mins = int(DEFAULT_TIMEOUT_IN_MINS)
    try:
        timeout_in_mins = int(TIMEOUT_IN_MINS)
    except ValueError:
        print('Given TIMEOUT_IN_MINS {} is not a number, Proceeding with default timeout of {} mins'.format(TIMEOUT_IN_MINS, DEFAULT_TIMEOUT_IN_MINS))
    timeout = 60*timeout_in_mins
    sleep_interval = 5
    pods_ready = False
    SERVICE_NAME_PREFIX = os.getenv('SERVICE_NAME_PREFIX', 'tca')
    POD_NAMESPACE = os.getenv('MY_POD_NAMESPACE', 'default')
    POD_NAME = os.getenv('POD_NAME', 'platform-manager')
    pod_label = '{}-{}'.format(SERVICE_NAME_PREFIX, POD_NAME)
    while not pods_ready and time.time() - start < timeout:
        pods_ready = True
        pod_list, err = get_pods(POD_NAMESPACE, pod_label)
        if err:
            print(err)
            pods_ready = False
            time.sleep(sleep_interval)
            continue
        data = json.loads(pod_list)
        pod_items = None
        try:
            pod_items = data['items']
        except KeyError:
            pods_ready = False
            print('Waiting for pods with label {}={} to be created...'.format("app", pod_label))
            time.sleep(sleep_interval)
            continue

        if not pod_items:
            pods_ready = False
            print('Waiting for pods with label {}={} to be created...'.format("app", pod_label))
            time.sleep(sleep_interval)
            continue

        for pod_item in pod_items:
            pod_name = pod_item['metadata']['name']
            container_statuses = None
            try:
                container_statuses = pod_item['status']['containerStatuses']
            except KeyError:
                pods_ready = False
                print('Waiting for containers in pod {} to be created...'.format(pod_name))
                time.sleep(sleep_interval)
                continue

            if not container_statuses:
                pods_ready = False
                print('Waiting for containers in pod {} to be created...'.format(pod_name))
                time.sleep(sleep_interval)
                continue
            print('')
            print('checking readiness of pod {}'.format(pod_name))
            is_pod_ready = check_pod_ready(container_statuses)

            # checks for minimum number of pods to be ready
            # in general you can get minimum = (replicaCount - maxUnavailable) in case the pod is from deployment
            # or if there is a PodDisruptionBudget set for this pod you can get either of minAvailable or maxUnavailable
            # currently assuming minimum number of pods is 1 as the serviceAccount doesn't have much privileges other than pods listing
            # also assuming given pod labels belongs to a single deployment or replicaset or a pod but not from job, sts, daemonset etc
            if is_pod_ready == True:
                pods_ready = True
                print('pod {} is ready'.format(pod_name))
                break
            else:
                pods_ready = False
                print('pod {} is not ready'.format(pod_name))
        if pods_ready == False:
            print('')
            print('Waiting for minimum number of pods with label {}={} to be ready...'.format("app", pod_label))
            time.sleep(sleep_interval)

    if pods_ready == True:
        print('Minimum number of pods with label {}={} are ready'.format("app", pod_label))
    else:
        sys.exit('timed out waiting for pods {} ready'.format(pod_label))

if __name__ == "__main__":
    main()